<!--
  ~ Copyright (c) 1999-2019 Allette Systems Pty Ltd
-->
<xsl:stylesheet version="3.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:f="http://www.pageseeder.com/function"
                xmlns:json="http://www.w3.org/2005/xpath-functions"
                exclude-result-prefixes="#all">

<xsl:template match="root[@service='api-group-feed' or @service='api-project-feed']" mode="json">
  <json:map>
    <json:boolean key="available"><xsl:value-of select="//search/@indexes != ''"/></json:boolean>
    <json:boolean key="reindexing"><xsl:value-of select="//search/@reindexing = 'true'"/></json:boolean>
    <json:array key="feed">
      <xsl:apply-templates select="//results" mode="feed"/>
    </json:array>
  </json:map>
</xsl:template>

<xsl:template match="results" mode="feed">
  <xsl:variable name="groups" select="//group" as="element(group)*"/>

  <!-- avoid duplicates (shared documents) -->
  <xsl:variable name="filtered-results">
    <xsl:for-each-group select="result" group-by="concat(field[@name='psid'], '/', field[@name='pstype'])">
      <xsl:variable name="first-result" select="current-group()[1]" />
      <filtered-result>
        <xsl:sequence select="$first-result" />
        <groups>
          <xsl:choose>
            <xsl:when test="$first-result/field[@name='pstype'] = 'url'">
              <xsl:sequence select="f:group('group', $first-result, 1, $groups)" />
              <json:array key="sharedGroups">
                <xsl:for-each select="($first-result/field[@name='psgroupid'])[position() gt 1]">
                  <xsl:sequence select="f:group('', $first-result, position() + 1, $groups)" />
                </xsl:for-each>
              </json:array>
            </xsl:when>
            <xsl:otherwise>
              <xsl:variable name="result-default-group" select="current-group()[starts-with(field[@name='psfolder'], concat($site-prefix, '/', translate(field[@name='psgroupname'], '-', '/'), '/'))]" />
              <xsl:variable name="result-main-group" select="if ($result-default-group) then $result-default-group else $first-result" />
              <xsl:sequence select="f:group('group', $result-main-group, $groups)" />
              <json:array key="sharedGroups">
                <xsl:for-each select="current-group()[field[@name='psgroupid'] != $result-main-group/field[@name='psgroupid']]">
                  <xsl:sequence select="f:group('', ., $groups)" />
                </xsl:for-each>
              </json:array>
            </xsl:otherwise>
          </xsl:choose>
        </groups>
    </filtered-result>
    </xsl:for-each-group>
  </xsl:variable>
  <!-- We group results by group, type, thread and time -->
  <xsl:for-each-group select="$filtered-results//result" group-adjacent="f:feed-type-grouping(.)">
    <xsl:sort select="field[@name='psmodified']" order="descending" />
    <xsl:variable name="result-groups" select="../groups/*" />

    <xsl:choose>
      <xsl:when test="field[@name='pstype'] = 'comment'">
        <xsl:variable name="status" select="field[@name='pscomment-status']"/>
        <xsl:variable name="is-task" select="$status and $status != 'None'"/>
        <xsl:variable name="original" select="field[@name='psid'] = field[@name='psdiscussionid']"/>
        <json:map>
          <!-- Core attributes -->
          <json:string key="feed"><xsl:value-of select="current-grouping-key()"/>/<xsl:value-of select="field[@name='psid']"/></json:string>
          <json:string key="title"><xsl:value-of select="field[@name='pstitle']"/></json:string>
          <json:string key="type"><xsl:value-of select="if ($is-task) then 'task' else 'discussion'"/></json:string>
          <json:string key="modified"><xsl:value-of select="field[@name='psmodifieddate']/@datetime"/></json:string>
          <json:boolean key="initial"><xsl:value-of select="if ($original) then 'true' else 'false'"/></json:boolean>
          <xsl:sequence select="$result-groups" />

          <!-- Comment specific -->
          <json:map key="{if ($is-task) then 'task' else 'discussion'}">
            <json:string key="id"><xsl:value-of select="field[@name='psid']"/></json:string>
            <json:array key="authors">
              <xsl:for-each select="distinct-values(current-group()/field[@name='psauthor'])">
                <json:string><xsl:value-of select="."/></json:string>
              </xsl:for-each>
            </json:array>
            <json:array key="authorids">
              <xsl:for-each select="distinct-values(current-group()/field[@name='psauthorid'])">
                <json:number><xsl:value-of select="."/></json:number>
              </xsl:for-each>
            </json:array>
            <!-- Tasks only -->
            <xsl:if test="$status and $status != 'None'">
              <xsl:variable name="priority" select="field[@name='pscomment-priority']"/>
              <xsl:variable name="assignedto" select="field[@name='pscomment-assignedto']"/>
              <json:string key="status"><xsl:value-of select="$status"/></json:string>
              <xsl:if test="$priority">
                <json:string key="priority"><xsl:value-of select="$priority"/></json:string>
              </xsl:if>
              <xsl:if test="$assignedto">
                <json:string key="assignedto"><xsl:value-of select="$assignedto"/></json:string>
              </xsl:if>
            </xsl:if>
          </json:map>

        </json:map>
      </xsl:when>

      <!-- Documents and URLs -->
      <xsl:when test="field[@name='pstype'] = 'document'">
        <xsl:choose>

          <!-- A batch of documents -->
          <xsl:when test="count(current-group()) gt 3">
            <json:map>
              <!-- Core attributes -->
              <json:string key="feed"><xsl:value-of select="current-grouping-key()"/>/batch</json:string>
              <json:string key="title"><xsl:value-of select="field[@name='pstitle']"/></json:string>
              <json:string key="type">document-list</json:string>
              <json:string key="modified"><xsl:value-of select="field[@name='psmodifieddate']/@datetime"/></json:string>
              <json:boolean key="initial"><xsl:value-of select="f:just-created(.)"/></json:boolean>

              <!-- Document-specific -->
              <json:array key="documents">
                <xsl:for-each select="current-group()">
                  <json:map>
                    <json:string key="id"><xsl:value-of select="field[@name='psid']"/></json:string>
                    <json:string key="description"><xsl:value-of select="field[@name='psdescription']"/></json:string>
                    <json:string key="filename"><xsl:value-of select="field[@name='psfilename']"/></json:string>
                    <json:string key="mediatype"><xsl:value-of select="field[@name='psmediatype']"/></json:string>
                    <json:string key="subtype"><xsl:value-of select="field[@name='pssubtype']"/></json:string>
                    <json:boolean key="attachment"><xsl:value-of select="count(field[@name='psfolder'][contains(., '/attachments/')]) gt 0"/></json:boolean>
                    <xsl:sequence select="$result-groups" />
                  </json:map>
                </xsl:for-each>
              </json:array>

              <!-- groups for the document list, use first document -->
              <xsl:sequence select="$result-groups" />
            </json:map>
          </xsl:when>

          <!-- Individual documents -->
          <xsl:otherwise>
            <xsl:for-each select="current-group()">
              <json:map>
                <!-- Core attributes -->
                <json:string key="feed"><xsl:value-of select="current-grouping-key()"/>/<xsl:value-of select="field[@name='psid']"/></json:string>
                <json:string key="title"><xsl:value-of select="field[@name='pstitle']"/></json:string>
                <json:string key="type">document</json:string>
                <json:string key="modified"><xsl:value-of select="field[@name='psmodifieddate']/@datetime"/></json:string>
                <json:boolean key="initial"><xsl:value-of select="f:just-created(.)"/></json:boolean>
                <xsl:sequence select="$result-groups" />

                <!-- Document-specific -->
                <json:map key="document">
                  <json:string key="id"><xsl:value-of select="field[@name='psid']"/></json:string>
                  <json:string key="description"><xsl:value-of select="field[@name='psdescription']"/></json:string>
                  <json:string key="filename"><xsl:value-of select="field[@name='psfilename']"/></json:string>
                  <json:string key="mediatype"><xsl:value-of select="field[@name='psmediatype']"/></json:string>
                  <json:string key="documenttype"><xsl:value-of select="field[@name='psdocumenttype']"/></json:string>
                  <json:string key="subtype"><xsl:value-of select="field[@name='pssubtype']"/></json:string>
                  <json:boolean key="attachment"><xsl:value-of select="count(field[@name='psfolder'][contains(., '/attachments/')]) gt 0"/></json:boolean>
                </json:map>
              </json:map>
            </xsl:for-each>
          </xsl:otherwise>

        </xsl:choose>
      </xsl:when>

      <!-- URLs -->
      <xsl:when test="field[@name='pstype'] = 'url'">
        <json:map>
          <!-- Core attributes -->
          <json:string key="feed"><xsl:value-of select="current-grouping-key()"/>/<xsl:value-of select="field[@name='psid']"/></json:string>
          <json:string key="title"><xsl:value-of select="field[@name='pstitle']"/></json:string>
          <json:string key="type">url</json:string>
          <json:string key="modified"><xsl:value-of select="field[@name='psmodifieddate']/@datetime"/></json:string>
          <json:boolean key="initial"><xsl:value-of select="f:just-created(.)"/></json:boolean>
          <xsl:sequence select="$result-groups" />

          <!-- URL-specific -->
          <json:map key="url">
            <json:string key="id"><xsl:value-of select="field[@name='psid']"/></json:string>
            <json:string key="description"><xsl:value-of select="field[@name='psdescription']"/></json:string>
          </json:map>

        </json:map>
      </xsl:when>
    </xsl:choose>

  </xsl:for-each-group>
</xsl:template>

<xsl:function name="f:group" as="element(json:map)">
  <xsl:param name="key" />
  <xsl:param name="result"/>
  <xsl:param name="groups" />
  <xsl:sequence select="f:group($key, $result, 1, $groups)" />
</xsl:function>

<xsl:function name="f:group" as="element(json:map)">
  <xsl:param name="key" />
  <xsl:param name="result"/>
  <xsl:param name="position" />
  <xsl:param name="groups" />
  <xsl:variable name="id" select="($result/field[@name='psgroupid'])[$position]" />
  <xsl:variable name="title" select="$groups[@id = $id]/@title" />
  <json:map>
    <xsl:if test="$key != ''"><xsl:attribute name="key"><xsl:value-of select="$key" /></xsl:attribute></xsl:if>
    <json:number key="id"><xsl:value-of select="if ($id and $id castable as xs:long) then $id else 0"/></json:number>
    <json:string key="name"><xsl:value-of select="($result/field[@name='psgroupname'])[$position]"/></json:string>
    <xsl:if test="$title"><json:string key="title"><xsl:value-of select="$title"/></json:string></xsl:if>
  </json:map>
</xsl:function>

<xsl:function name="f:feed-type-grouping" as="xs:string">
  <xsl:param name="result"/>
  <xsl:variable name="group" select="$result/field[@name='psgroupid']"/>
  <xsl:variable name="type"  select="$result/field[@name='pstype']"/>

  <xsl:choose>
    <!-- Documents are organised by time -->
    <xsl:when test="$type = 'document'">
      <xsl:variable name="minute" select="substring($result/field[@name='psmodifieddate'], 1, 12)"/>
      <xsl:variable name="created" select="f:just-created($result)"/>
      <xsl:value-of select="concat($group, '/document/', if ($created) then '0' else '1', '/', $minute)"/>
    </xsl:when>

    <!-- URLs are organised by time -->
    <xsl:when test="$type = 'url'">
      <xsl:variable name="minute" select="substring($result/field[@name='psmodifieddate'], 1, 12)"/>
      <xsl:variable name="created" select="f:just-created($result)"/>
      <xsl:value-of select="concat($group[1], '/url/', if ($created) then '0' else '1', '/', $minute)"/>
    </xsl:when>

    <!-- Comments are organised by thread -->
    <xsl:when test="$type = 'comment'">
      <xsl:variable name="discussion-id" select="$result/field[@name='psdiscussionid']"/>
      <xsl:variable name="original" select="$discussion-id = $result/field[@name='psid']"/>
      <xsl:value-of select="concat($group, '/comment/',  if ($original) then '0' else '1', '/', $discussion-id)"/>
    </xsl:when>

    <!-- There should be no others... -->
    <xsl:otherwise>
      <xsl:value-of select="concat($group, '/other')"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:function>

<xsl:function name="f:just-created" as="xs:boolean">
  <xsl:param name="result"/>
  <!-- In some rare cases or legacy data, the created date is missing so we fallback on `false` -->
  <xsl:choose>
    <xsl:when test="$result/field[@name='psmodifieddate']/@datetime and $result/field[@name='pscreateddate']/@datetime">
      <xsl:variable name="delta" select="xs:dateTime($result/field[@name='psmodifieddate']/@datetime) - xs:dateTime($result/field[@name='pscreateddate']/@datetime)"/>
      <xsl:sequence select="$delta lt xs:dayTimeDuration('PT120M')"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:sequence select="false()"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:function>

</xsl:stylesheet>
