/**
 * $Id: ShortenStringURIHandlerTest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.tests;

import  junit.framework.TestSuite;

import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.valueuri.ShortenStringURIHandler;

/**
 * Testsuite for {@linkplain ShortenStringValueURIHandler}.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,test
 **/

public final class ShortenStringURIHandlerTest extends ValueURIHandlerTestSkeleton
{
    private final static int MAXLEN = ShortenStringURIHandler.MAXLEN;
    private final static String ABC_ETC = "abcdefghijklmnopqrstuvwxyz?";
    private final String EXACTSTRING;
    private final String OVERSTRING;
    private final String UNDERSTRING;
    private final String EXACTLEFT, EXACTRIGHT;


    /**
     * Initializes a new test case for named method.
     * @param methodname test case method's name (non-null)
     **/
    public ShortenStringURIHandlerTest(String methodname)
    {
        super("ShortenStringURIHandler::",methodname);
        EXACTSTRING = xstring(MAXLEN);
        OVERSTRING  = xstring(MAXLEN+5);
        UNDERSTRING = xstring(Math.max(0,MAXLEN-5));
        EXACTLEFT   = "..."+xstring(MAXLEN-3);
        EXACTRIGHT  = OVERSTRING.substring(0,MAXLEN-3)+"...";
    }


    /**
     * Returns full test suite for XDefCheckValueURIHandler.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(ShortenStringURIHandlerTest.class);
    }

//  ---------------------------------------------------------------------------------------------------------
//  --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    protected ValueURIHandler newOUT()
    {
        return new ShortenStringURIHandler();
    }
    
    private String xstring(int length)
    {
        final String L = "0123456789";
        StringBuffer sb = new StringBuffer(Math.max(length,100));
        while (--length>=0) {
            sb.append(L.charAt(length % 10));
        }
        return sb.substring(0);
    }

//  ---------------------------------------------------------------------------------------------------------
//  ------------------------------------------- [ The Test Cases ] ------------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    /**
     * Verify a blank uri fragment is evaluated ok.
     * @since JWare/AntX 0.5
     **/
    public void testBaseline_AntX05()
    {
        checkBaseline();
        System.out.println("EXACTSTRING:"+EXACTSTRING);
        System.out.println("OVERSTRING:"+OVERSTRING);
        ValueURIHandler out = newOUT();
        String result = out.valueFrom("","$shorten:",m_rqlink);
        assertEqual(result,"","$shorten:");
    }


    /**
     * Verify an all blank options frag is handled properly.
     * @since JWare/AntX 0.5
     **/
    public void testAllBlankOptionsParsedOK_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result = out.valueFrom("?,,,,","$shorten:?,,,,",m_rqlink);
        assertEqual(result,"","'$shorten:?,,,,'");

        final String BLANK= "@(blank)?@(blank),,@(blank),,";
        final String BLANKURI="$shorten:"+BLANK;

        getProject().setProperty("blank","");
        result = out.valueFrom(BLANK,BLANKURI,m_rqlink);
        assertEqual(result,"",BLANKURI);
        result = out.valueFrom("  "+BLANK,"$shorten:  "+BLANK,m_rqlink);
        assertEqual(result,"  ","'$shorten:  "+BLANK+"'");
    }


    /**
     * Verify that exact and under length strings are returned 
     * as-is regardless of the incoming options.
     * @since JWare/AntX 0.5
     **/
    public void testStringsUnderAndExactMaxLenNotChanged_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result = out.valueFrom(EXACTSTRING,"$shorten:"+EXACTSTRING,m_rqlink);
        assertIdent(result,EXACTSTRING,"$shorten:EXACTSTRING");
        result = out.valueFrom(UNDERSTRING,"$shorten:"+UNDERSTRING,m_rqlink);
        assertIdent(result,UNDERSTRING,"$shorten:UNDERSTRING");
        
        String urifragment = EXACTSTRING+"?"+MAXLEN+",,right";
        result = out.valueFrom(urifragment,"$shorten:"+urifragment,m_rqlink);
        assertEqual(result,EXACTSTRING,"$shorten:EXACTSTRING?MAXLEN,,right");
        
        urifragment = UNDERSTRING+"?"+MAXLEN+",,left,,ERROR!";
        result = out.valueFrom(urifragment,"$shorten:"+urifragment,m_rqlink);
        assertEqual(result,UNDERSTRING,"$shorten:UNDERSTRING?MAXLEN,,left,,ERROR!");
    }


    /**
     * Verify that strings over the max length are shortened as
     * expected.
     * @since JWare/AntX 0.5
     **/
    public void testStringsOverMaxLengthShortenedDDD_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result = out.valueFrom(OVERSTRING,"$shorten:"+OVERSTRING,m_rqlink);
        assertEqual(result,EXACTLEFT,"$shorten:OVERSTRING");
        result = out.valueFrom(OVERSTRING+"?,,right","$shorten:"+OVERSTRING+"?,,right",m_rqlink);
        assertEqual(result,EXACTRIGHT,"$shorten:OVERSTRING?,,right");
        result = out.valueFrom("abcdefgh?5,,left","$shorten:abcdefgh?5,,left",m_rqlink);
        assertEqual(result,"...gh","$shorten:abcdefgh?5,,left");
        result = out.valueFrom("abcdefgh?5,,right","$shorten:abcdefgh?5,,right",m_rqlink);
        assertEqual(result,"ab...","$shorten:abcdefgh?5,,right");
    }


    /**
     * Verify can specify custom ellipses both left and right.
     * @since JWare/AntX 0.5
     **/
    public void testCustomEllipsesUsed_AntX05()
    {
        ValueURIHandler out = newOUT();
        String urifragment = ABC_ETC+"20,,left,,[snip]...";
        String result = out.valueFrom(urifragment,"$shorten:"+urifragment,m_rqlink);
        assertEqual(result,"[snip]...pqrstuvwxyz","$shorten:"+urifragment);
        urifragment = ABC_ETC+"25,,right,,..blablabla";
        result = out.valueFrom(urifragment,"$shorten:"+urifragment,m_rqlink);
        assertEqual(result,"abcdefghijklmn..blablabla","$shorten:"+urifragment);
    }


    /**
     * Verify that a custom ellipses string that is too long (will 
     * always force final string over maxlength) is ignored and 
     * replaced with "&#8230;" always.
     * @since JWare/AntX 0.5
     **/
    public void testTooLongCustomEllipsesIgnored_AntX05()
    {
        ValueURIHandler out = newOUT();
        String urifragment = "abcdefgh?5,,left,,[abbreviated]";
        String result = out.valueFrom(urifragment,"$shorten:"+urifragment,m_rqlink);
        assertEqual(result,"...gh","$shorten:"+urifragment);

        final String ellipses="[abbreviated]";
        final int maxlen= ellipses.length()-1;//12(=maxlength)
        urifragment = "abcdefghijklmnopqrstuvwxyz?"+maxlen+",,right,,"+ellipses;
        result= out.valueFrom(urifragment,"$shorten:"+urifragment,m_rqlink);
        assertEqual(result,"abcdefghi...","$shorten:"+urifragment);

        urifragment = "abcdefghijklmnopqrstuvwxyz?"+maxlen+",,left,,"+ellipses;
        result= out.valueFrom(urifragment,"$shorten:"+urifragment,m_rqlink);
        assertEqual(result,"...rstuvwxyz","$shorten:"+urifragment);
    }
}

/* end-of-ShortenStringURIHandlerTest.java */