/**
 * $Id: DefaultsValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.info;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.Defaults;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Simple value URI handler that returns default settings for various AntX fixture
 * elements. In particular the current Iteration's {@linkplain Defaults Defaults}
 * are checked. This handler is not installed automatically by the AntX runtime,
 * you must install it explicitly like the example below. Default value URIs are often
 * linked to either the <span class="src">$default:</span> or the
 * <span class="src">$df:</span> scheme.
 * <p/>
 * If the named default is not one of the known builtin default attributes, this
 * handler will look for a project/fixture property named like:
 * <span class="src">defaults.<i>default-name</i></span> where <i>default-name</i> 
 * is the fragment that comes after the <span class="src">$default:</span> prefix
 * in the value URI. The looked for property's name is <em>always</em> lowercased; for 
 * example, the URI <span class="src">$default:CopyLeft</span> will be matched against
 * a project property <span class="src">defaults.copyleft</span>.
 * <p/>
 * To allow you to specify inlined default values for missing property-based defaults,
 * this handler understands the alternate "<span class="src">@(propertyname)</span>"
 * format in its URI fragment query portion (the bits after the "&#63;"); see example
 * below. Note that inlined default values are limited to property-based defaults; they 
 * are never used for builtin fixture defaults. When an inlined default value is
 * used, it is used as specified (no lowercasing is applied).
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;assign var="_loop.failquick" value="${<b>$default:</b>haltiferror}"/&gt;
 *    &lt;property name="finetrace" value="${<b>$default:</b>assertions}"/&gt;
 *    &lt;attribute name="license" default="${<b>$default:</b>license?LGPL}"/&gt;
 *    &lt;parameter name="license" value="${<b>$default:</b>license?@(project.license)}"/&gt;
 *
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="default"
 *             value="com.idaremedia.antx.valueuri.info.DefaultsValueURIHandler"/&gt;
 *       &lt;parameter name="df"
 *             value="com.idaremedia.antx.valueuri.info.DefaultsValueURIHandler"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 * @see       DefaultsPropertyNameURIHandler
 **/

public class DefaultsValueURIHandler extends ValueURIHandlerSkeleton
{
    /**
     * Initializes a new defaults value uri handler.
     **/
    public DefaultsValueURIHandler()
    {
    }



    /**
     * Returns the embedded default value from the full uri if there.
     * Otherwise returns <i>null</i>.
     */
    public String getDefaultValue(String fullUri, Requester clnt)
    {
        if (fullUri!=null && fullUri.length()>4) {
            int i;
            if (fullUri.charAt(0)=='$' && (i=fullUri.indexOf(":"))>0) {
                String fallback = null;
                i= fullUri.lastIndexOf("?");
                if (i>0) {
                    if ((i+1)<fullUri.length()) {
                        fallback = fullUri.substring(i+1);
                        fallback = Tk.resolveString(clnt.getProject(),fallback,true);
                    } else {
                        fallback = "";
                    }
                }
                return fallback;
            }
        }
        return null;
    }



    /**
     * Returns the associated default setting if possible. Will
     * return <i>null</i> if default name unrecognized.
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        Defaults dflt = Iteration.defaultdefaults();
        String fragment = Tk.lowercaseFrom(uriFragment);

        // Ask the Iteration (or subclass) first!
        String value = dflt.valueFrom(fragment,clnt);
        if (value!=null) {
            return value;
        }

        // Determine if an inlined default was specified. We should
        // not assume the us-locale string's length is same as the original
        // so we renormalize the fragment minus the default if needed.
        
        final Project project = clnt.getProject();
        String fallback = null;
        int i= uriFragment.lastIndexOf("?");
        if (i>0) {
            if ((i+1)<uriFragment.length()) {
                fallback = uriFragment.substring(i+1);//NB:case used as passed-in!
                fallback = Tk.resolveString(project,fallback,true);
            } else {
                fallback = "";
            }
            fragment = Tk.lowercaseFrom(uriFragment.substring(0,i));
        }
        String property = dflt.defaultPropertiesPrefix(project)+fragment;
        value = Tk.getTheProperty(project,property);

        return value==null ? fallback : value;
    }
}

/* end-of-DefaultsValueURIHandler.java */