/**
 * $Id: DateTimeValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.datetime;

import  java.text.Format;
import  java.text.SimpleDateFormat;
import  java.util.Map;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.DateTimeFormat;
import  com.idaremedia.antx.helpers.DurationFormat;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Value URI handler that converts simple datetime queries into formatted
 * date/time strings. In addition to the symbolic format (the protocol's name),
 * you must specify the timestamp in one of three ways: use the string "now" for 
 * the  current system time, using piped result of another value uri, or declare the 
 * timestamp number value itself as a string.
 * <p/>
 * To specify a custom format string use the value URI fragment bits after the 
 * "&#63;" like: <span class="src">${$PS:now?'['h:mm:ss']$ '}</span>.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;echo message="Current time is: ${$shorttime:now}"/&gt;
 *    &lt;echo message="Current GMT time is: ${$gmtdatetime:now}"/&gt;
 * 
 *    &lt;echo message="My time prompt: ${$prompt:now?'['h:mm:ss']: '}"/&gt;
 * 
 *    &lt;assign var="starttime" op="now"/&gt;
 *    &lt;show msgid="start.work.at" msgarg1="${$var:starttime|$datetime:}"/&gt;
 *    ...<i>[work work work]</i>
 *    &lt;show msgid="ended.work.at" msgarg1="${$datetime:}"/&gt;
 *
 *    &lt;assign var="timing" op="now"/&gt;
 *    ...<i>[work work work]</i>
 *    &lt;assign var="timing" op="-now"/&gt;
 *    &lt;echo message="It Took: ${$var:timing|$duration:}"/&gt;
 * 
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="datetime"
 *             value="com.idaremedia.antx.valueuri.DateTimeValueURIHandler"/&gt;
 *       &lt;parameter name="shortdatetime"
 *             value="com.idaremedia.antx.valueuri.DateTimeValueURIHandler"/&gt;
 *       ...<i>[All other standard names and your own custom ones if desired]</i>
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 * @.caveat   You must use one of the expected scheme names to get the default
 *            formats; otherwise you must manually map your scheme name to this
 *            handler and <em>always</em> provide a format string.
 **/

public final class DateTimeValueURIHandler implements ValueURIHandler
{
    private static final Map LINKS = AntXFixture.newMap();
    static {
        //NB: long-winded and uglee but obvious. Also not 
        //    likely to change anytime soon...
        LINKS.put("$longdatetime:",DateTimeFormat.STANDARD);
        LINKS.put("$longtime:",DateTimeFormat.STANDARD_TIME);
        LINKS.put("$longdate:",DateTimeFormat.STANDARD_DATE);
        LINKS.put("$gmtdatetime:",DateTimeFormat.GMT);
        LINKS.put("$gmtdate:",DateTimeFormat.GMT_DATE);
        LINKS.put("$gmttime:",DateTimeFormat.GMT_TIME);
        LINKS.put("$shortdatetime:",DateTimeFormat.ABBREV);
        LINKS.put("$shorttime:",DateTimeFormat.ABBREV_TIME);
        LINKS.put("$shortdate:",DateTimeFormat.ABBREV_DATE);
        LINKS.put("$datetime:",DateTimeFormat.STANDARD);
        LINKS.put("$time:",DateTimeFormat.STANDARD_TIME);
        LINKS.put("$date:",DateTimeFormat.STANDARD_DATE);
        LINKS.put("$duration:",DurationFormat.INSTANCE);
        LINKS.put("$changelogdate:",DateTimeFormat.CHANGELOG);
        LINKS.put("$cvsdate:",DateTimeFormat.CHANGELOG);
        LINKS.put("$svndate:",DateTimeFormat.CHANGELOG);
    }



    /**
     * Initializes a new datetime value uri handler.
     **/
    public DateTimeValueURIHandler()
    {
    }
 
 

    /**
     * Returns the best fit formatter for the format directive
     * embedded in a given date value uri.
     * @param fullUri the full uri (including the '$protocol:' prefix)
     * @param clnt problem handler (non-null)
     * @return the formatter or <i>null</i> if no match found.
     **/
    public static Format defaultFormatter(String fullUri, Requester clnt)
    {
        Format dfmt= null;
        int i= fullUri.indexOf(':');
        if (i>0) {
            String which= fullUri.substring(0,++i);
            dfmt = (Format)LINKS.get(which);
            if (dfmt==null && (i=fullUri.indexOf("?",i))>0) {
                i++;
                if (i<fullUri.length()-1) {
                    String fmtstr = fullUri.substring(i);
                    fmtstr = Tk.resolveString(clnt.getProject(),fmtstr,true);
                    try {
                        dfmt = new SimpleDateFormat(fmtstr);
                    } catch(IllegalArgumentException mfX) {
                        clnt.problem(mfX.getMessage(),Project.MSG_WARN);
                    }
                }
            }
        }
        return dfmt;
    }



    /**
     * Returns a snapshot of the current set of uri scheme name to
     * format mappings. Never returns <i>null</i>. The returned map
     * is disassociated from this class; modifications to it are not
     * reflected back to this class.
     * @.impl Used for testing this class.
     **/
    public static Map copyOfMappings()
    {
        synchronized(LINKS) {
            return AntXFixture.newMapCopy(LINKS);
        }
    }



    /**
     * Tries to determine if the named scheme is a timestamp based
     * format or a duration format. Duration formats expect delta
     * for the timestamp parameter; timestamp format expect well
     * timestamps.
     * @param scheme name of uri scheme
     * @return Boolean.TRUE or Boolean.FALSE if can determine one
     *        way or other from name; otherwise returns <i>null</i>.
     **/
    public static Boolean isTimestampScheme(String scheme)
    {
        Boolean yes = null;
        if (scheme!=null) {
            scheme = Tk.lowercaseFrom(scheme);
            if (scheme.indexOf("date")>=0 || scheme.indexOf("time")>=0) {
                yes = Boolean.TRUE;
            } else if (scheme.indexOf("duration")>=0) {
                yes = Boolean.FALSE;
            }
        }
        return yes;
    }



    /**
     * Utility to format the given timestamp according a date value
     * uri format directive.
     * @param time the timestamp to use
     * @param fullUri the format informaton uri (non-null)
     * @param clnt problem handler (non-null)
     * @return formatted string or <i>null</i> if no formatter
     *      match found.
     **/
    public String format(long time, String fullUri, Requester clnt)
    {
        Format dfmt= defaultFormatter(fullUri,clnt);
        if (dfmt!=null) {
            return DateTimeFormat.format(time,dfmt);
        }
        return null;
    }



    /**
     * Returns a formatted datetime string for "now" uri. Any
     * other timestamp returns <i>null</i>.
     **/
    public String getDefaultValue(String fullUri, Requester clnt)
    {
        final long NOW = now();
        if (fullUri.endsWith(":now")) {
            return format(NOW,fullUri,clnt);
        }
        if (fullUri.indexOf(":now?")>1) {
            return format(NOW,fullUri,clnt);
        }
        return null;
    }



    /**
     * The uri fragment can be one of three things: either the string 
     * "now" for the current system time, or a <em>single</em> redirect
     * to a reference or variable that contains the timestamp, or the
     * timestamp value itself as a string.
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        String timestring = uriFragment;
        int i = uriFragment.indexOf('?');
        if (i>=0) {
            timestring = uriFragment.substring(0,i++);
        }
        timestring = Tk.resolveString(clnt.getProject(),timestring,true);

        if (timestring.length()==0 || "now".equals(timestring)) {
            return format(now(),fullUri,clnt);
        }

        long timestamp = Tk.longFrom(timestring,-1L);
        if (timestamp>0L) {
            return format(timestamp,fullUri,clnt);
        }

        return null;
    }


    /**
     * Tells this handler what "now" means. Never used from live uri
     * handling mechanism; only for testing and stubs.
     * @param timestamp the required value of "now" (>=0L)
     * @.impl Used for testing this class.
     **/
    public void setNow(long timestamp)
    {
        AntX.require_(timestamp>=0L,AntX.utilities+"DateTimeValueURIHandler:",
                      "setnow- valid timstamp");
        m_nowTS = timestamp;
    }


    private long now()
    {
        return m_nowTS<0L ? System.currentTimeMillis() : m_nowTS;
    }


    private long m_nowTS = -1L;
}

/* end-of-DateTimeValueURIHandler.java */