/**
 * $Id: TransformValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri;

import  java.util.Map;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.TransformHelper;
import  com.idaremedia.antx.parameters.ValueTransform;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Value URI handler that applies the builtin AntX {@linkplain ValueTransform} to
 * a given string. The allowed inputs are determined by what's legitimate for
 * the indicated scheme. Note that date/time value uris are usually handled by the
 * general date valueURI handler not this class.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;property name="build.id" value="${<b>$uppercase:</b>@(build.name)}"/&gt;
 *    &lt;property name="userscratch.url" value="${<b>$ospathurl:</b>@(TMPDIR)}"/&gt;
 * 
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="transform"
 *             value="com.idaremedia.antx.valueuri.TransformValueURIHandler"/&gt;
 *       &lt;parameter name="ospath"
 *             value="com.idaremedia.antx.valueuri.TransformValueURIHandler"/&gt;
 *       &lt;parameter name="lowercase"
 *             value="com.idaremedia.antx.valueuri.TransformValueURIHandler"/&gt;
 *       &#8230;<i>[All other transform shortcuts like uppercase, etc.]</i>
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 **/

public class TransformValueURIHandler extends ValueURIHandlerSkeleton
{
    private static final Map LINKS = AntXFixture.newMap();
    static {
        //NB: long-winded and uglee but obvious. Also not
        //    likely to change anytime soon...
        LINKS.put("$transform:",null);
        LINKS.put("$ospath:",ValueTransform.OSPATH);
        LINKS.put("$path:",ValueTransform.OSPATH);
        LINKS.put("$ospathurl:",ValueTransform.OSPATHURL);
        LINKS.put("$pathurl:",ValueTransform.OSPATHURL);
        LINKS.put("$uppercase:",ValueTransform.UPPERCASE);
        LINKS.put("$lowercase:",ValueTransform.LOWERCASE);
        LINKS.put("$trim:",ValueTransform.TRIM);
        LINKS.put("$stripws:",ValueTransform.STRIPWS);
        LINKS.put("$decimal:",ValueTransform.DECIMAL);
    }



    /**
     * Initializes a new transform value uri handler.
     **/
    public TransformValueURIHandler()
    {
        super();
    }



    /**
     * Returns the best fit value tranform for the format directive
     * embedded in a given value uri. You can use the general "$transform:"
     * scheme can name a transform that does not have a shorthand scheme.
     * For example: <span class="src">$transform:@(crdate)?longdatetime</span>.
     * @param fullUri the full uri (including the '$scheme:' prefix)
     * @param clnt problem handler (non-null)
     * @return the formatter or <i>null</i> if no match found.
     **/
    public static ValueTransform defaultTransform(String fullUri, Requester clnt)
    {
        ValueTransform valfmt= null;
        int i= fullUri.indexOf(':');
        if (i>0) {
            String which= fullUri.substring(0,++i);
            valfmt = (ValueTransform)LINKS.get(which);
            if (valfmt==null && (i=fullUri.indexOf("?",i))>0) {
                i++;
                if (i<fullUri.length()-1) {
                    String byname = Tk.resolveString(clnt.getProject(),
                        fullUri.substring(i),true);
                    valfmt = ValueTransform.from(byname);
                }
            }
        }
        return valfmt;
    }



    /**
     * Programmatic extension point that allows subclasses to add
     * own protocol names and value transform enum. Can also use to override
     * the default formats associated with our standard names.
     * @param name protocol marker string (non-null)
     * @param valfmt ValueTransform enum value (non-null)
     **/
    public static void addMapping(String name, ValueTransform valfmt)
    {
        AntX.require_(name!=null&&valfmt!=null, "TransformValueURIHandler:",
                     "addMaping- nonzro args");
        synchronized(LINKS) {
            LINKS.put(name,valfmt);
        }
    }



    /**
     * Tries to transform a script-supplied string as instructed. The
     * result is returned as a string. If an exception was thrown by
     * the transformation and the current iteration is not set to halt
     * valueuris problems, this method returns <i>null</i>.
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        ValueTransform vt = defaultTransform(fullUri,clnt);
        if (vt!=null) {
            int i = uriFragment.lastIndexOf('?');
            if (i>0) {
                uriFragment = Tk.resolveString(clnt.getProject(),
                    uriFragment.substring(0,i), true);
            } else {
                uriFragment = Tk.resolveString(clnt.getProject(),
                    uriFragment, true);
            }
            try {
                return TransformHelper.apply(vt,uriFragment,clnt.getProject());
            } catch(RuntimeException anyX) {
                clnt.problem(anyX.getMessage(),Project.MSG_WARN);
                if (Iteration.defaultdefaults().isHaltIfError("valueuris")) {
                    throw anyX;
                }
            }
        }
        return null;
    }
}

/* end-of-TransformValueURIHandler.java */