/**
 * $Id: SIDsValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri;

import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.SIDs;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Value URI handler that generates an AntX {@linkplain SIDs random value}. Very useful
 * in tests and SAMS artifact definition methods. There are three types of pseudo-random
 * values: string, int[eger], and long. The string and int[eger] options also support a
 * single parameter that is used as a prefix (in case of string) and an upper bound (in
 * case of int[eger]. The long value URI does not support any parameters. If you are 
 * generating a value to be used as a reference identifier in a particular project, use
 * the complementary {@linkplain NewRefIdValueURIHandler $newrefid:} handler instead. 
 * Be careful when using this value URI as part of an overlaid property's value&#8212;
 * the value <em>will change everytime</em> the property is dereferenced!
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;assign var="_localkey" value="${<b>$random:string?key</b>}"/&gt;
 *    &lt;assign var="_duration" value="${<b>$random:int?999999</b>}"/&gt;
 *    &lt;assign var="_branch" value="${<b>$random:boolean</b>}"/&gt;
 *
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="random"
 *             value="com.idaremedia.antx.valueuri.SIDsValueURIHandler"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 * @see       NewRefIdValueURIHandler
 **/

public final class SIDsValueURIHandler extends ValueURIHandlerSkeleton
{
    /**
     * Initalizes a new random value generator.
     **/
    public SIDsValueURIHandler()
    {
        super();
    }
    

    /**
     * Returns a random value based on the incoming instruction.
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        if (uriFragment.length()==0) {
            return SIDs.next();
        }
        String adjustment = "";
        int i = uriFragment.indexOf("?");
        if (i>0) {
            if (i<uriFragment.length()-1) {
                adjustment = uriFragment.substring(i+1);
            }
            uriFragment = uriFragment.substring(0,i);
        }
        uriFragment = Tk.lowercaseFrom(uriFragment);
        switch (uriFragment.charAt(0)) {
            case 's': {
                return SIDs.next(adjustment/*prefix*/);
            }
            case 'i': {
                int limit = Tk.NO_INT;
                if (adjustment.length()>0) {
                    limit = Tk.integerFrom(adjustment,Tk.NO_INT);
                } 
                if (limit!=Tk.NO_INT) {
                    return String.valueOf(SIDs.nextInt(limit));
                }
                return String.valueOf(SIDs.nextInt());
            }
            case 'l': {
                return String.valueOf(SIDs.nextLong());
            }
            case 'b': {
                int value = SIDs.nextInt(2);
                return String.valueOf(value!=0);
            }
        }
        return getDefaultValue(fullUri,clnt);
    }
}

/* end-of-SIDsValueURIHandler.java */