/**
 * $Id: ValueURIManagerTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.util.Iterator;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.PropertyHelper;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.FixtureIds;
import  com.idaremedia.antx.FixtureInitializer;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.apis.BuildError;
import  com.idaremedia.antx.parameters.FeedbackLevel;
import  com.idaremedia.antx.starters.SimpleManagerTask;

/**
 * A manager task that (un)installs an AntX value URI aware property helper hook. Once
 * installed this hook lets you use the "<span class="src">${$scheme:[fragment]}</span>"
 * shorthand string to retrieve arbitrary types of data. This task is a valid
 * antlib definition task, so you can include it in your antlib scriptlets. Usually
 * defined as <span class="src">&lt;manageuris&gt;</span>.
 * <p/>
 * <b>Example Usage:</b><pre>
 *   -- To Enable --
 *    &lt;manageuris action="enable"/&gt; -or-
 *    &lt;manageuris/&gt;
 *
 *   -- To Install A Set Of Handlers --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="foo" value="com.mysoft.FooUriReader"/&gt;
 *       &lt;parameter name="bar" value="com.mysoft.BarUriReader"/&gt;
 *    &lt;/manageuris&gt;
 *
 *   -- To Disable --
 *    &lt;manageuris action="disable"/&gt;
 *
 *   -- To Uninstall A Set Of Handlers --
 *    &lt;manageuris action="uninstall"&gt;
 *       &lt;parameter name="foo"/&gt;
 *    &lt;/manageuris&gt;
 *
 *   -- To Check Whether ValueURI Interpretation Is Enabled --
 *    &lt;manageuris action="check"/&gt; -or-
 *
 *    &lt;manageuris action="check"&gt;
 *       &lt;parameter name="updateproperty" name="valueuris.enabled"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 * @.impl     Direct access to project references map is intentional to avoid
 *            the spurious warnings about replaced references.
 * @see       com.idaremedia.antx.ValueURIHandler
 **/

public class ValueURIManagerTask extends SimpleManagerTask
    implements FixtureInitializer
{
    private static final String REFID= FixtureIds.VALUEURI_INTERPRETER;
    private static final String LABEL = "ValueUriInterpreter";


    /**
     * Initializes a new "enabling" manager task. You must set
     * the action option of this task to change its default
     * enable behavior.
     **/
    public ValueURIManagerTask()
    {
        super(AntX.fixture+"ValueURIManagerTask:","enable");
    }



    /**
     * Verifies that there are no value uri interpreters in the
     * current property helper hook chain w/o a matching reference
     * marker.
     * @param ph the first link in hook chain (non-null)
     **/
    private void verifyNoneInChain(PropertyHelper ph)
    {
        do {
            if (ph.getNext() instanceof Interpreter) {
                String error = Iteration.uistrs().get
                    ("fixture.supPh.stack.corrupted", LABEL);
                log(error, Project.MSG_ERR);
                throw new BuildError(error);
            }
            ph = ph.getNext();
        } while(ph!=null);
    }



    /**
     * Converts a fully qualified class name to a ValueURIHandler
     * class instance.
     * @param e the slot o' information
     * @return the class reference
     * @throws BuildException if unable to load class or class
     *      is incompatible.
     **/
    private Class loadHandlerClass(Map.Entry e)
    {
        String classname = (String)e.getValue();
        try {
            Class c = Class.forName(classname);
            if (!ValueURIHandler.class.isAssignableFrom(c)) {
                throw new ClassCastException();
            }
            return c;
        } catch (Exception anyX) {
            String error = Iteration.uistrs().get("task.bad.custimpl.class1",
                classname, ValueURIHandler.class.getName());
            log(error, Project.MSG_ERR);
            throw new BuildException(error, anyX, getLocation());
        }
    }



    /**
     * Will install a set of value uri handlers. Each handler is
     * defined by a generic parameter; the parameter's name is the
     * value uri prefix, and the value is the fully qualified class
     * name of the handler.
     * @param args list of value uri handlers to enable explicitly
     **/
    public void doInstall(Map args)
    {
        if (!args.isEmpty()) {
            Iterator itr= args.entrySet().iterator();
            Class oldc, newc;
            boolean warnreplace = isVerbose();

            while (itr.hasNext()) {
                Map.Entry e = (Map.Entry)itr.next();
                String prefix = (String)e.getKey();

                if (!FixtureExaminer.isBuiltinValueURIScheme(prefix)) {
                    newc = loadHandlerClass(e);
                    oldc = Iteration.replaceValueURIHandler(prefix,newc);
                    if (warnreplace && oldc!=null && oldc!=newc) {
                        String warning = Iteration.uistrs().get("valueuri.repl.handlr",
                            prefix, oldc.getName(), newc.getName());
                        log(warning, Project.MSG_WARN);
                    } else {
                        log("For ("+prefix+") installed value URI handler "+newc.getName(),
                            Project.MSG_VERBOSE);
                    }
                } else {
                    String warning = Iteration.uistrs().get("valueuri.resrved.handlr",prefix);
                    log(warning, Project.MSG_WARN);
                }
            }
        }
    }



    /**
     * Will install a property helper hook to interpret AntX value
     * uris. Does nothing if an interpreter is already installed for
     * the enclosing project (issues warning). If value uri handlers
     * are declared (as parameters), those handlers are added automatically.
     * @param args list of value uri handlers to also install
     **/
    public void doEnable(Map args)
    {
        Project p = getProject();
        Map ht = p.getReferences();
        boolean warn = false;

        synchronized(ht) {//NB:forces FxExaminer queries to wait too...
            Object o = ht.get(REFID);
            if (o==null) {
                Interpreter interpreter;
                PropertyHelper ph = PropertyHelper.getPropertyHelper(p);
                synchronized(ph) {
                    verifyNoneInChain(ph);
                    interpreter = new Interpreter(p);
                    interpreter.setNext(ph.getNext());
                    ph.setNext(interpreter);
                }
                ht.put(REFID,interpreter);
            } else {
                warn = true;
            }
        }
        if (!args.isEmpty()) {
            this.doInstall(args);
            warn = false;
        }
        if (warn && isVerbose()) {
            String warning = Iteration.uistrs().get("fixture.supPh.install.once",LABEL);
            log(warning, Project.MSG_WARN);
        }
    }



    /**
     * Will uninstall a set of named value uri handlers from the
     * current iteration. Does nothing if the args list is empty.
     * @param args list of value uri handlers to install
     **/
    public void doUninstall(Map args)
    {
        if (!args.isEmpty()) {
            Iterator itr= args.keySet().iterator();
            while (itr.hasNext()) {
                String prefix = (String)itr.next();
                if (!FixtureExaminer.isBuiltinValueURIScheme(prefix)) {
                    Iteration.removeValueURIHandler(prefix);
                } else {
                    String warning = Iteration.uistrs().get("valueuri.resrved.handlr",prefix);
                    log(warning, Project.MSG_WARN);
                }
            }
        }
    }



    /**
     * Will uninstall a property helper hook that interprets AntX
     * value uris. Does nothing if no such interpreter exists in
     * the current project (issues a warning).
     * @param ignoredArgs [ignored]
     **/
    public void doDisable(Map ignoredArgs)
    {
        Project p = getProject();
        Map ht = p.getReferences();
        boolean warn=false;

        synchronized(ht) {//NB:forces FxExaminer queries to wait too...
            Object o = ht.get(REFID);
            if (o instanceof Interpreter) {
                Interpreter interpreter = (Interpreter)o;
                ht.remove(REFID);

                PropertyHelper ph = PropertyHelper.getPropertyHelper(p);
                synchronized(ph) {
                    PropertyHelper prevPh=null;
                    do {
                        if (ph.getNext()==interpreter) { break; }
                        prevPh = ph;
                        ph = ph.getNext();
                    } while(ph!=null);
                    if (ph!=null) {
                        ph.setNext(interpreter.getNext());
                        if (prevPh!=null) {
                            prevPh.setNext(ph);
                        }
                    } else {
                        warn = true;
                    }
                    interpreter.setNext(null);
                }
                interpreter = null;
            }
            o = null;
        }

        if (warn && isVerbose()) {
            String warning= Iteration.uistrs().get("fixture.supPh.not.instald",LABEL);
            log(warning,Project.MSG_WARN);
        }
    }



    /**
     * Tests for the existence of a value uri interpreter in the
     * enclosing project. The check generates one of three possible
     * answers: <span class="src">yes</span>, <span class="src">no</span>,
     * and <span class="src">unknown</span>. An unknown is flagged if
     * <em>something</em> is declared as an interpreter but is not
     * known to this manager task.
     * <p/>
     * This action can be configured to save its answer string to a
     * project property or variable; set either the
     * "<span class="src">updateproperty</span>" or
     * "<span class="src">updatevariable</span>"
     * parameters to the name of the fixture elements to be set.
     * @param args custom update property instructions (non-null)
     **/
    public void doCheck(Map args)
    {
        Project p = getProject();
        Map ht = p.getReferences();
        String result = "no";

        synchronized(ht) {//NB:forces FxExaminer queries to wait too...
            Object o = ht.get(REFID);
            if (o instanceof Interpreter) {
                result = "yes";
            }
            else if (o!=null) {
                result = "unknown";
            }
        }

        boolean saved=false;

        Object o = args.get("updateproperty");
        if (o!=null) {
            String property = o.toString();
            FixtureExaminer.checkIfProperty(p,m_rqlink,property,true);
            p.setNewProperty(property,result);
            saved = true;
        }
        o = args.get("updatevariable");
        if (o!=null) {
            ExportedProperties.set(o.toString(),result);
            saved = true;
        }
        if (!saved) {
            int level = NoiseLevel.getDefault(p).getNativeIndex();
            String msg = Iteration.uistrs().get("valueuri.instaled",result);
            log(msg, level);
        }
    }



    /**
     * Our hook into the standard Ant project property framework.
     * This class must play nice with other installed hooks.
     *
     * @since     JWare/AntX 0.5
     * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version   0.5
     * @.safety   guarded (inherited)
     * @.group    impl,helper
     */
    private static class Interpreter extends PropertyHelper
    {
        private final Project m_project;

        Interpreter(Project project) {
            m_project = project;
        }

        /**
         * Filters the incoming value through our fixture examiner.
         **/
        public Object getPropertyHook(String ns, String name,
                                      boolean userOnly) {
            Object o;
            if ((ns==null || ns.length()==0) && !userOnly) {
                o= FixtureExaminer.findValue(m_project,name,null);
            } else {
                o= null;
            }
            if (o==null && getNext()!=null) {
                o = getNext().getPropertyHook(ns,name,userOnly);
            }
            return o;
        }

        /**
         * Always delegates to next in hook chain if available.
         **/
        public boolean setPropertyHook(String ns, String name, Object value,
                                       boolean inherited, boolean userOnly,
                                       boolean isNew) {
            if (getNext()!=null) {
                return getNext().setPropertyHook(ns,name,value,inherited,
                                                 userOnly,isNew);
            }
            return false;
        }
    }


    private boolean isVerbose()
    {
        //NB:undef => is quiet (don't warn)
        return !FeedbackLevel.isQuietish(getFeedbackLevel(),true,true);
    }
}

/* end-of-ValueURIManagerTask.java */