/**
 * $Id: EchoItemsTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  java.io.IOException;
import  java.io.OutputStream;
import  java.io.PrintStream;
import  java.util.Iterator;
import  java.util.List;
import  java.util.Properties;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.FlexValueSupport;
import  com.idaremedia.antx.parameters.PropertySource;
import  com.idaremedia.antx.starters.EchoThingTask;

/**
 * Test aid that echos properties, variables, etc. Also acts as stub 'true' condition
 * to facilitate debugging of complex rules. Example usage:<pre>
 *
 *   &lt;printenv property="ant.version"/&gt;
 *   &lt;printenv reference="my.classpath"/&gt;
 *   &lt;printenv variables="all" properties="all" if="build.debug"/&gt;
 *
 * OR (Nested as a dud-condition for debugging):
 *
 *   &lt;tally trueproperty="blastoff"&gt;
 *       &lt;tally ruleid="..." trueproperty="a.present"/&gt;
 *       &lt;print properties="all" if="tally.debug"/&gt;
 *       &lt;tally trueproperty="a.present"&gt;
 *          &lt;.../&gt;
 *       &lt;/tally&gt;
 *       &lt;print property="a.present" if="tally.debug"/&gt;
 *       &lt;.../&gt;
 *   &lt;/tally&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple (once configured)
 * @.group   impl,test,helper
 **/

public class EchoItemsTask extends EchoThingTask implements Condition, FlexValueSupport
{
    /**
     * Initializes a new EchoItemsTask instance.
     **/
    public EchoItemsTask()
    {
        super(AntX.starters);
    }


    /**
     * Initializes a new CV-labeled EchoItemsTask instance.
     * @param iam CV-label (non-null)
     **/
    protected EchoItemsTask(String iam)
    {
        super(iam);
    }


    /**
     * Initialize this task's project. Also auto-sets this task's
     * referenced thing to the name of the project if it has one.
     **/
    public void setProject(Project project)
    {
        super.setProject(project);

        if (project!=null && getThingRefId()==null) {
            String name = project.getName();
            if (name==null) {
                name = project.getProperty("ant.project.name");
            }
            if (name!=null) {
                setThingRefId(name);
            }
        }
    }


    /**
     * Sets the list of project properties to include in the output.
     * Use '<i>all</i>' to display all project properties. Use '<i>user</i>'
     * to display only user properties. Use a comma-delimited list for
     * specific properties.
     * @param nameList comma-delimited list of property names (non-null)
     **/
    public void setProperties(String nameList)
    {
        require_(nameList!=null,"setProps- nonzro list");
        m_propertiesList= nameList;
    }


    /**
     * Shortcut for setting a single property.
     **/
    public final void setProperty(String property)
    {
        setProperties(property);
    }


    /**
     * Returns list of project properties to be output. Returns
     * <i>null</i> if never set (nothing displayed).
     **/
    public final String getPropertiesNameList()
    {
        return m_propertiesList;
    }


    /**
     * Sets the by-prefix filter for echoed properties or variables.
     **/
    public void setPrefix(String prefix)
    {
        require_(prefix!=null && prefix.length()>0,"setPfx- valid str");
        m_prefix = prefix;
    }


    /**
     * Returns the by-prefix filter for echoed properties or variables.
     * Returns <i>null</i> if never set.
     **/
    public final String getPrefix()
    {
        return m_prefix;
    }


    /**
     * Sets the list of exported properties to include in the output.
     * Use '<i>all</i>' to display all exported properties. Use a
     * comma-delimited list for specific exported properties.
     * @param nameList comma-delimited list of variable names (non-null)
     **/
    public void setVariables(String nameList)
    {
        require_(nameList!=null,"setVars- nonzro list");
        m_variablesList= nameList;
    }


    /**
     * Shortcut for setting a single exported property.
     **/
    public final void setVariable(String variable)
    {
        setVariables(variable);
    }


    /**
     * Shortcut for setting a single exported property.
     **/
    public final void setVar(String variable)
    {
        setVariables(variable);
    }


    /**
     * Returns list of exported properties to be output. Returns
     * '<i>null</i>' if never set (nothing displayed).
     **/
    public final String getVariablesNameList()
    {
        return m_variablesList;
    }


    /**
     * Sets the list of project references to include in the output.
     * Use a comma-delimited list for specific references.
     * @param nameList comma-delimited list of references names (non-null)
     **/
    public void setReferences(String nameList)
    {
        require_(nameList!=null,"setRefs- nonzro list");
        m_refsList= nameList;
    }


    /**
     * Shortcut for setting a single reference.
     **/
    public final void setReference(String refid)
    {
        require_(refid!=null,"setRef- nonzro ref");
        setReferences(refid);
    }


    /**
     * Returns list of project references to be output. Returns
     * '<i>null</i>' if never set (nothing displayed).
     **/
    public final String getReferencesNameList()
    {
        return m_refsList;
    }


    /**
     * Sets the by-class filter for echoed references.
     **/
    public void setKindOf(Class claz)
    {
        require_(claz!=null,"setKindOf- nonzro claz");
        m_kindOfClass = claz;
    }


    /**
     * Returns the by-class filter for echoed references. Returns
     * <i>Object.class</i> if never set.
     **/
    public final Class getKindOfFilterClass()
    {
        return m_kindOfClass==null ? Object.class : m_kindOfClass;
    }


    /**
     * Sets whether reference printing will exclude out-of-scope
     * reference identifiers. Defaults <i>true</i> for backward
     * compatibility.
     * @since JWare/AntX 0.4
     **/
    public void setUnknowns(boolean yn)
    {
        m_includeUnknowns= yn;
    }


    /**
     * Returns <i>true</i> if this task will include to-be-determined
     * references in listings.
     * @since JWare/AntX 0.4
     **/
    public final boolean willIncludeUnknowns()
    {
        return m_includeUnknowns;
    }



    /**
     * Returns this task's project as the referenced thing.
     **/
    protected Object getReferencedThing(Class ofKind, String msgid)
    {
        return getProject();//that's-all-she-wrote
    }


    /**
     * Write the appropriate set of project properties to output stream.
     * @param P project from which properties read (non-null)
     * @param os output stream (non-null)
     * @see #setProperties
     * @return <i>true</i> if properties written
     * @throws IOException if any I/O error occurs on stream
     **/
    protected boolean echoProperties(final Project P, OutputStream os)
        throws IOException
    {
        String list= getPropertiesNameList();
        if (!Tk.isWhitespace(list)) {

            Properties allP= new Properties();

            PropertySource domain= PropertySource.from(list);
            if (domain!=null) {
                allP.putAll(FixtureExaminer.copyOfProperties(domain,P));
                if (getPrefix()!=null) {//oye...
                    Iterator itr= allP.keySet().iterator();
                    while (itr.hasNext()) {
                        String key = itr.next().toString();
                        if (!key.startsWith(m_prefix)) {
                            itr.remove();
                        }
                    }
                }
            } else {
                List wanted = Tk.splitList(list);
                for (int i=0,N=wanted.size();i<N;i++) {
                    String key = (String)wanted.get(i);
                    allP.setProperty(key, String.valueOf(P.getProperty(key)));
                }
                wanted=null;
            }

            String header= getMsg(newMsgGetter(uistrs().get("echo.label.properties")));
            allP.store(os,header);

            allP.clear();
            allP=null;
            return true;
        }
        return false;
    }


    /**
     * Write the appropriate set of exported properties to output stream.
     * @param P project information
     * @param os output stream (non-null)
     * @see #setVariables
     * @return <i>true</i> if properties written
     * @throws IOException if any I/O error occurs on stream
     **/
    protected boolean echoVariables(final Project P, OutputStream os)
        throws IOException
    {
        String list= getVariablesNameList();
        if (!Tk.isWhitespace(list)) {
            Properties allP;
            String ll= Tk.lowercaseFrom(list);

            if (Strings.ALL.equals(ll)) {
                allP= ExportedProperties.copy(null);
                if (getPrefix()!=null) {//oye...
                    Iterator itr= allP.keySet().iterator();
                    while (itr.hasNext()) {
                        String key = itr.next().toString();
                        if (!key.startsWith(m_prefix)) {
                            itr.remove();
                        }
                    }
                }
            } else {
                allP= ExportedProperties.copy(null);
                List wanted = Tk.splitList(list);
                allP.keySet().retainAll(wanted);
                if (wanted.size()>allP.size()) {//insert "null" strings
                    for (int i=0,N=wanted.size();i<N;i++) {
                        String key = (String)wanted.get(i);
                        if (allP.getProperty(key)==null) {
                            allP.setProperty(key,Strings.NULL);
                        }
                    }
                }
                wanted=null;
            }

            String header= getMsg(newMsgGetter(uistrs().get("echo.label.variables")));
            allP.store(os,header);

            allP.clear();
            allP=null;
            return true;
        }
        return false;
    }



    /**
     * Returns a marker string for unresolved references (still unknown).
     **/
    protected final String unresolvedString(Project P, String key)
    {
        return P.getName()+":UnresolvedReference@"+key;
    }



    /**
     * Write the appropriate set of project references to output stream.
     * Uses each targetted reference's 'toString' method to generate output.
     * @param P project information
     * @param os output stream (non-null)
     * @see #setReferences
     * @return <i>true</i> if properties written
     * @throws IOException if any I/O error occurs on stream
     **/
    protected boolean echoReferences(final Project P, OutputStream os)
        throws IOException
    {
        String list= getReferencesNameList();
        if (!Tk.isWhitespace(list)) {

            Properties allP= new Properties();

            Iterator keysitr;
            if (Strings.ALL.equals(Tk.lowercaseFrom(list))) {
                keysitr= P.getReferences().keySet().iterator(); //?MT-safe..copy?
            } else {
                keysitr= Tk.splitList(list).iterator();
            }
            while (keysitr.hasNext()) {
                String key = (String)keysitr.next();

                Object object = FixtureExaminer.trueReference(P,key);//@since AntX 0.4
                if (object==FixtureExaminer.IGNORED_REFERENCE) {
                    if (!willIncludeUnknowns()) {
                        continue;
                    }
                    object = unresolvedString(P,key);
                }
                if (m_kindOfClass==null/*allowAll*/ || object==null/*passthru*/ ||
                    (m_kindOfClass.isInstance(object))) {
                    allP.setProperty(key, Tk.stringFrom(object,P));
                }
            }

            String header= getMsg(newMsgGetter(uistrs().get("echo.label.references")));
            allP.store(os,header);

            allP.clear();
            allP=null;
            return true;
        }
        return false;
    }


    /**
     * Echoes the project items to this task's output stream.
     **/
    protected void echoThing()
    {
        OutputStream os = getOutputStream();
        try {
            boolean p= echoProperties(getProject(),os);
            boolean v= echoVariables(getProject(),os);
            boolean r= echoReferences(getProject(),os);

            if (!p && !v && !r) {
                if (getMsgId()!=null || getDefaultMsg()!=null) {
                    String s = getMsg();
                    new PrintStream(os).println(s);
                }
            }
            if (tryAntLog(os)) {
                log(getAntLogString(os),getMsgLevel().getNativeIndex());
            }

        } catch(IOException ioX) {
            String ermsg = uistrs().get("task.echo.unable");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());

        } finally {
            try { os.close(); } catch(Exception X){/*burp*/}
            os=null;
        }
    }


    /**
     * Echoes the list of items then always returns <i>true</i>.
     **/
    public final boolean eval()
    {
        execute();
        return true;
    }


    /**
     * Should always be able to execute if in legitimate project.
     **/
    protected final void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);
    }


    private String  m_propertiesList;
    private String  m_variablesList;
    private String  m_refsList;
    private Class   m_kindOfClass = Object.class;//NB:allow 'em all!
    private String  m_prefix;//NB:allow 'em all!
    private boolean m_includeUnknowns=true;//NB:show 'em all!
}

/* end-of-EchoItemsTask.java */
