/**
 * $Id: ValueTransform.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Enumeration that represents the type of value transformations a task might
 * implement. This enumeration is an extension of the more limited
 * <span class="src">{@linkplain CaseTransform}</span> class. The following list
 * explains what the various settings mean:<ul>
 *    <li><span class="src">none</span>: The value should be left as-is.</li>
 *    <li><span class="src">duration</span>: The value (a number string) should
 *         be converted to a human-readable duration string.</li>
 *    <li><span class="src">datetime</span>: The value (a number string) should
 *         be converted to an abbreviated human-readable date-time string.</li>
 *    <li><span class="src">longdatetime</span>: The value (a number string)
 *        should be converted to a full (or verbose) human-readable
 *        date-time string.</li>
 *    <li><span class="src">ospath</span>: The value (a path string) should
 *         be converted to a normalized platform-specific path.</li>
 *    <li><span class="src">ospathurl</span>: The value (a path string) should
 *         be converted to a normalized platform-specific path URL.</li>
 *    <li><span class="src">checkpoint</span>: The value (a number string) should
 *         be converted to a normalized GMT-based timestamp string.</li>
 *    <li><span class="src">lowercase</span>: The value should be lower-cased
 *         using the current locale.</li>
 *    <li><span class="src">uppercase</span>: The value should be UPPER-cased
 *        using the current locale.</li>
 *    <li><span class="src">trim</span>: The value should be trimmed of whitespace
 *        before and after (in-betweens not touched).</li>
 *    <li><span class="src">decimal</span>: The value (a number) should be converted
 *        to a simple fixed decimal format (at most three places after decimal).</li>
 *    <li><span class="src">stripws</span>: The value should be stripped of all 
 *        whitespace. Result is a <em>condensed</em> string.</li>
 * </ul>
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 * @see       TransformHelper
 **/

public class ValueTransform extends EnumSkeleton
{
    //@.impl ORDERING of static declarations is important! ------

    //1) Indices (in order)

    /** Index of {@linkplain #NONE NONE}. **/
    public static final int NONE_INDEX = 0;
    /** Index of {@linkplain #DATETIME DATETIME}. **/
    public static final int DATETIME_INDEX = NONE_INDEX+1;
    /** Index of {@linkplain #LONGDATETIME LONGDATETIME}. **/
    public static final int LONGDATETIME_INDEX = DATETIME_INDEX+1;
    /** Index of {@linkplain #DURATION DURATION}. **/
    public static final int DURATION_INDEX = LONGDATETIME_INDEX+1;
    /** Index of {@linkplain #OSPATH OSPATH}. **/
    public static final int OSPATH_INDEX = DURATION_INDEX+1;
    /** Index of {@linkplain #CHECKPOINT CHECKPOINT}. **/
    public static final int CHECKPOINT_INDEX = OSPATH_INDEX+1;
    /** Index of {@linkplain #LOWERCASE LOWERCASE}. **/
    public static final int LOWERCASE_INDEX = CHECKPOINT_INDEX+1;
    /** Index of {@linkplain #UPPERCASE UPPERCASE}. **/
    public static final int UPPERCASE_INDEX = LOWERCASE_INDEX+1;
    /** Index of {@linkplain #INVERTCASE INVERTCASE}. **/
    public static final int INVERTCASE_INDEX = UPPERCASE_INDEX+1;
    
    /** Index of {@linkplain #OSPATHURL OSPATHURL}.
     *  @since JWare/AntX 0.5
     */
    public static final int OSPATHURL_INDEX = INVERTCASE_INDEX+1;
    
    /** Index of {@linkplain #DECIMAL DECIMAL}.
     *  @since JWare/AntX 0.5
     */
    public static final int DECIMAL_INDEX = OSPATHURL_INDEX+1;

    /** Index of {@linkplain #TRIM TRIM}.
     *  @since JWare/AntX 0.5
     */
    public static final int TRIM_INDEX = DECIMAL_INDEX+1;

    /** Index of {@linkplain #STRIPWS STRIPWS}.
     *  @since JWare/AntX 0.5
     */
    public static final int STRIPWS_INDEX = TRIM_INDEX+1;

    /** The number of base value transform values. **/
    protected static final int BASE_VALUE_COUNT= STRIPWS_INDEX+1;


    //2) Values (in order)

    /** Values in same order as public indices. **/
    private static final String[] VALUES_= new String[] {
        "none", "datetime", "longdatetime", "duration", "ospath",
        "checkpoint", "lowercase", "uppercase", "invertcase",
        "ospathurl", "decimal", "trim", "stripws"
    };


    //3) Singletons (depend on Indices and Values already existing!)

    /** Singleton "<span class="src">none</span>" transform. **/
    public static final ValueTransform NONE=
        new ValueTransform(VALUES_[NONE_INDEX],NONE_INDEX);

    /** Singleton "<span class="src">datetime</span>" choice. **/
    public static final ValueTransform DATETIME=
        new ValueTransform(VALUES_[DATETIME_INDEX],DATETIME_INDEX);

    /** Singleton "<span class="src">longdatetime</span>" choice. **/
    public static final ValueTransform LONGDATETIME=
        new ValueTransform(VALUES_[LONGDATETIME_INDEX],LONGDATETIME_INDEX);

    /** Singleton "<span class="src">duration</span>" choice. **/
    public static final ValueTransform DURATION=
        new ValueTransform(VALUES_[DURATION_INDEX],DURATION_INDEX);

    /** Singleton "<span class="src">ospath</span>" choice. **/
    public static final ValueTransform OSPATH=
        new ValueTransform(VALUES_[OSPATH_INDEX],OSPATH_INDEX);

    /** Singleton "<span class="src">checkpoint</span>" choice. **/
    public static final ValueTransform CHECKPOINT=
        new ValueTransform(VALUES_[CHECKPOINT_INDEX],CHECKPOINT_INDEX);

    /** Singleton "<span class="src">lowercase</span>" transform. **/
    public static final ValueTransform LOWERCASE=
        new ValueTransform(VALUES_[LOWERCASE_INDEX],LOWERCASE_INDEX);

    /** Singleton "<span class="src">uppercase</span>" transform. **/
    public static final ValueTransform UPPERCASE=
        new ValueTransform(VALUES_[UPPERCASE_INDEX],UPPERCASE_INDEX);

    /** Singleton "<span class="src">invertcase</span>" transform. **/
    public static final ValueTransform INVERTCASE=
        new ValueTransform(VALUES_[INVERTCASE_INDEX],INVERTCASE_INDEX);


    /** Singleton "<span class="src">pathurl</span>" transform.
     *  @since JWare/AntX 0.5
     */
    public static final ValueTransform OSPATHURL=
        new ValueTransform(VALUES_[OSPATHURL_INDEX],OSPATHURL_INDEX);

    /** Singleton "<span class="src">decimal</span>" transform.
     *  @since JWare/AntX 0.5
     */
    public static final ValueTransform DECIMAL=
        new ValueTransform(VALUES_[DECIMAL_INDEX],DECIMAL_INDEX);

    /** Singleton "<span class="src">trim</span>" transform.
     *  @since JWare/AntX 0.5
     */
    public static final ValueTransform TRIM=
        new ValueTransform(VALUES_[TRIM_INDEX],TRIM_INDEX);

    /** Singleton "<span class="src">stripws</span>" transform.
     *  @since JWare/AntX 0.5
     */
    public static final ValueTransform STRIPWS=
        new ValueTransform(VALUES_[STRIPWS_INDEX],STRIPWS_INDEX);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public ValueTransform()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensures this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    private ValueTransform(String v, int i)
    {
        super(v);
    }



    /**
     * Returns a <em>copy</em> of the standard transform values
     * in order. Subclasses can use this method to pre-fill their
     * value arrays with the inherited list.
     * @param fillin [optional] array of strings to update with values.
     **/
    public static String[] copyOfDefaultValues(String[] fillin)
    {
        if (fillin==null) {
            fillin = new String[VALUES_.length];
        }
        System.arraycopy(VALUES_,0,fillin,0,VALUES_.length);
        return fillin;
    }



    /**
     * Returns copy of all possible source values as an ordered
     * string array. Note: ordering should be same as our
     * singleton indices.
     **/
    public String[] getValues()
    {
        return ValueTransform.copyOfDefaultValues(null);
    };



    /**
     * Helper that converts a scalar to a known ValueTransform.
     * Returns <i>null</i> if value does not match any of expected
     * source.
     * @param i the index to be matched
     **/
    public static ValueTransform from(int i)
    {
        if (i==NONE.index)        { return NONE; }
        if (i==OSPATH.index)      { return OSPATH; }
        if (i==OSPATHURL.index)   { return OSPATHURL; }
        if (i==DURATION.index)    { return DURATION; }
        if (i==DATETIME.index)    { return DATETIME; }
        if (i==DECIMAL.index)     { return DECIMAL; }
        if (i==LOWERCASE.index)   { return LOWERCASE; }
        if (i==UPPERCASE.index)   { return UPPERCASE; }
        if (i==TRIM.index)        { return TRIM; }
        if (i==STRIPWS.index)     { return STRIPWS; }
        if (i==CHECKPOINT.index)  { return CHECKPOINT; }
        if (i==LONGDATETIME.index){ return LONGDATETIME; }
        if (i==INVERTCASE.index)  { return INVERTCASE; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known
     * ValueTransform's index.
     * @param i the index to be matched
     * @param dflt the default ValueTransform if necessary
     **/
    public static ValueTransform from(int i, ValueTransform dflt)
    {
        ValueTransform choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known ValueTransform
     * singleton. Returns <i>null</i> if string unrecognized. String
     * can be either ValueTransform's symbolic name or its index.
     **/
    public static ValueTransform from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (Strings.DEFAULT.equals(s))   { return NONE; }/*safest*/
                if (DURATION.value.equals(s))    { return DURATION; }
                if (DATETIME.value.equals(s))    { return DATETIME; }
                if (OSPATH.value.equals(s))      { return OSPATH; }
                if ("path".equals(s))            { return OSPATH; }
                if (OSPATHURL.value.equals(s))   { return OSPATHURL; }
                if ("pathurl".equals(s))         { return OSPATHURL; }
                if (DECIMAL.value.equals(s))     { return DECIMAL; }
                if ("fixed".equals(s))           { return DECIMAL; }
                if (CHECKPOINT.value.equals(s))  { return CHECKPOINT; }
                if (LOWERCASE.value.equals(s))   { return LOWERCASE; }
                if (UPPERCASE.value.equals(s))   { return UPPERCASE; }
                if (TRIM.value.equals(s))        { return TRIM; }
                if (STRIPWS.value.equals(s))     { return STRIPWS; }
                if (LONGDATETIME.value.equals(s)){ return LONGDATETIME; }
                if (INVERTCASE.value.equals(s))  { return INVERTCASE; }
                if (NONE.value.equals(s))        { return NONE; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but with a
     * default value if supplied value does not match any known
     * ValueTransform's name.
     * @param s the symbolic name to be matched
     * @param dflt the default ValueTransform if necessary
     **/
    public static ValueTransform from(String s, ValueTransform dflt)
    {
        ValueTransform choice= from(s);
        return (choice==null) ? dflt : choice;
    }
}

/* end-of-ValueTransform.java */
