/**
 * $Id: MergeBehavior.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  com.idaremedia.antx.helpers.Tk;

/**
 * Enumeration that represents the various ways an iteration setting or a collection of
 * settings can be merged with a pre-existing set. The values are based on the standard
 * Ant "<span class="src">build.sysclasspath</span>" merge pattern. The following list 
 * explains what the various values mean:<ul>
 *    <li><span class="src">only</span>: The newer setting is used as if the existing
 *        value never existed.</li>
 *    <li><span class="src">ignore</span>: The newer setting is ignored as if it
 *        never existed.</li>
 *    <li><span class="src">first</span>: The newer setting should be combined with
 *        the existing setting while being given preference in script queries.</li>
 *    <li><span class="src">last</span>: The newer setting should be combined with
 *        the existing setting but preference will be given the existing settings
 *        in script queries.</li>
 * </ul>

 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 **/

public final class MergeBehavior extends EnumSkeleton 
{
    /** Index of {@linkplain #ONLY ONLY}. **/
    public static final int ONLY_INDEX = 0;
    /** Index of {@linkplain #IGNORE IGNORE}. **/
    public static final int IGNORE_INDEX = ONLY_INDEX+1;
    /** Index of {@linkplain #FIRST FIRST}. **/
    public static final int FIRST_INDEX = IGNORE_INDEX+1;
    /** Index of {@linkplain #LAST LAST}. **/
    public static final int LAST_INDEX = FIRST_INDEX+1;


    /** Singleton "<span class="src">only</span>" choice. **/
    public static final MergeBehavior ONLY=
        new MergeBehavior("only",ONLY_INDEX);

    /** Singleton "<span class="src">ignore</span>" choice. **/
    public static final MergeBehavior IGNORE=
        new MergeBehavior("ignore",IGNORE_INDEX);

    /** Singleton "<span class="src">first</span>" choice. **/
    public static final MergeBehavior FIRST=
        new MergeBehavior("first",FIRST_INDEX);

    /** Singleton "<span class="src">last</span>" choice. **/
    public static final MergeBehavior LAST=
        new MergeBehavior("last",LAST_INDEX);
    


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public MergeBehavior()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensures this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    private MergeBehavior(String v, int i)
    {
        super(v);
    }


    /**
     * Returns copy of all possible source values as an ordered
     * string array. Note: ordering should be same as our
     * singleton indices.
     **/
    public String[] getValues()
    {
        return new String[] {"only", "ignore", "first", "last"};
    };



    /**
     * Helper that converts a scalar to a known MergeBehavior.
     * Returns <i>null</i> if value does not match any of expected
     * source.
     * @param i the index to be matched
     **/
    public static MergeBehavior from(int i)
    {
        if (i==FIRST.index)  { return FIRST; }
        if (i==ONLY.index)   { return ONLY; }
        if (i==LAST.index)   { return LAST; }
        if (i==IGNORE.index) { return IGNORE; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known
     * MergeBehavior's index.
     * @param i the index to be matched
     * @param dflt the default MergeBehavior if necessary
     **/
    public static MergeBehavior from(int i, MergeBehavior dflt)
    {
        MergeBehavior choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known MergeBehavior
     * singleton. Returns <i>null</i> if string unrecognized. String
     * can be either MergeBehavior's symbolic name or its index.
     **/
    public static MergeBehavior from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (FIRST.value.equals(s))     { return FIRST; }
                if (ONLY.value.equals(s))      { return ONLY; }
                if (LAST.value.equals(s))      { return LAST; }
                if (IGNORE.value.equals(s))    { return IGNORE; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but with a
     * default value if supplied value does not match any known
     * MergeBehavior's name.
     * @param s the symbolic name to be matched
     * @param dflt the default MergeBehavior if necessary
     **/
    public static MergeBehavior from(String s, MergeBehavior dflt)
    {
        MergeBehavior choice= from(s);
        return (choice==null) ? dflt : choice;
    }
}

/* end-of-MergeBehavior.java */