/**
 * $Id: Handling.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Enumeration that represents how a component (task|macro) should handle a particular
 * kind of problem. The following list describes what each setting means:<ul>
 *    <li><span class="src">accept</span>: The value(s) should be accepted
 *        unconditionally.</li>
 *    <li><span class="src">reject</span>: The value should be rejected. Whether or
 *        not this leads to a script error is task dependent.</li>
 *    <li><span class="src">balk</span>: The value should trigger a script error.</li>
 *    <li><span class="src">ignore</span>: The value(s) should be ignored (as if
 *        it did not exist or was never presented).</li>
 *    <li><span class="src">inherit</span>: The handling behavior should be
 *        determined by the surrounding or "parent" context. What context is
 *        used is actually task dependent.</li>
 * </ul>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   api,helper
 **/

public final class Handling extends EnumSkeleton
{
    /** Index of {@linkplain #ACCEPT ACCEPT}. **/
    public static final int ACCEPT_INDEX = 0;
    /** Index of {@linkplain #IGNORE IGNORE}. **/
    public static final int IGNORE_INDEX = ACCEPT_INDEX+1;
    /** Index of {@linkplain #REJECT REJECT}. **/
    public static final int REJECT_INDEX = IGNORE_INDEX+1;
    /** Index of {@linkplain #BALK BALK}. **/
    public static final int BALK_INDEX = REJECT_INDEX+1;
    /** Index of {@linkplain #INHERIT INHERIT}. **/
    public static final int INHERIT_INDEX = BALK_INDEX+1;


    /** Singleton "<span class="src">accept</span>" choice. **/
    public static final Handling ACCEPT=
        new Handling("accept",ACCEPT_INDEX);

    /** Singleton "<span class="src">reject</span>" choice. **/
    public static final Handling REJECT=
        new Handling("reject",REJECT_INDEX);

    /** Singleton "<span class="src">ignore</span>" choice. **/
    public static final Handling IGNORE=
        new Handling("ignore",IGNORE_INDEX);

    /** Singleton "<span class="src">balk</span>" choice. **/
    public static final Handling BALK=
        new Handling("balk",BALK_INDEX);

    /** Singleton "<span class="src">inherit</span>" choice. **/
    public static final Handling INHERIT=
        new Handling("inherit",INHERIT_INDEX);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public Handling()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensures this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    private Handling(String v, int i)
    {
        super(v);
    }


    /**
     * Returns copy of all possible source values as an ordered
     * string array. Note: ordering should be same as our
     * singleton indices.
     **/
    public String[] getValues()
    {
        return new String[] {"accept", "ignore",
                             "reject", "balk", "inherit"};
    };



    /**
     * Helper that converts a scalar to a known Handling.
     * Returns <i>null</i> if value does not match any of expected
     * source.
     * @param i the index to be matched
     **/
    public static Handling from(int i)
    {
        if (i==IGNORE.index)  { return IGNORE; }
        if (i==BALK.index)    { return BALK; }
        if (i==ACCEPT.index)  { return ACCEPT; }
        if (i==INHERIT.index) { return INHERIT; }
        if (i==REJECT.index)  { return REJECT; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known
     * Handling's index.
     * @param i the index to be matched
     * @param dflt the default Handling if necessary
     **/
    public static Handling from(int i, Handling dflt)
    {
        Handling choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known Handling
     * singleton. Returns <i>null</i> if string unrecognized. String
     * can be either Handling's symbolic name or its index.
     **/
    public static Handling from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (IGNORE.value.equals(s))   { return IGNORE; }
                if (INHERIT.value.equals(s))  { return INHERIT; }
                if (BALK.value.equals(s))     { return BALK; }
                if (ACCEPT.value.equals(s))   { return ACCEPT; }
                if (REJECT.value.equals(s))   { return REJECT; }
                if (Strings.DEFAULT.equals(s)){ return BALK; }/*safest*/
                if (Strings.ENCLOSING.equals(s)) { return INHERIT; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but with a
     * default value if supplied value does not match any known
     * Handling's name.
     * @param s the symbolic name to be matched
     * @param dflt the default Handling if necessary
     **/
    public static Handling from(String s, Handling dflt)
    {
        Handling choice= from(s);
        return (choice==null) ? dflt : choice;
    }


//----------------------------------------------------------------------
// Common combinations for simple interpretations of choices:
//----------------------------------------------------------------------

    /**
     * Collapses a fine-grained handling into a simple "accept" or
     * "reject" choice. If the choice is either "ignore" or "accept"
     * it is collapsed to "accept". If the choice is either "balk" or
     * "reject" it is collapsed to "reject". Caller decides how
     * "inherit" and <i>null</i> are interpreted.
     * @param choice script value
     * @param inheritProxy returned if choice is "inherit"
     * @param nullProxy returned if choice is <i>null</i>
     **/
    public static Handling simplify(Handling choice,
                                    Handling inheritProxy,
                                    Handling nullProxy)
    {
        if (choice==null) {
            return nullProxy;
        }
        switch (choice.getIndex()) {
            case ACCEPT_INDEX:
            case IGNORE_INDEX: {
                return ACCEPT;
            }
            case REJECT_INDEX:
            case BALK_INDEX: {
                return REJECT;
            }
        }
        return inheritProxy;
    }


    /**
     * Collapses a fine-grained handling into a simple "accept" or
     * "reject" choice. Returns <i>null</i> if incoming choice is
     * <i>null</i>. See {@linkplain #simplify(Handling,Handling,Handling)
     * simplify(&#8230;)}.
     * @param choice script value (non-null)
     * @param inheritProxy returned if choice is "inherit"
     **/
    public static Handling simplify(Handling choice,
                                    Handling inheritProxy)
    {
        if (choice==null) {
            throw new IllegalArgumentException();
        }
        return simplify(choice,inheritProxy,null);
    }


    /**
     * Collapses a fine-grained handling choice into a simple "yes"
     * or "no" boolean choice. Returns <i>true</i> if choice is
     * determined to be "accept;" otherwise returns <i>false</i>
     * (including if incoming choice is <i>null</i>). Based on
     * {@linkplain #simplify(Handling,Handling) simplify(&#8230;)}.
     * @param choice script value (non-null)
     * @param inheritProxy used when choice is "inherit"
     **/
    public static boolean isYes(Handling choice,
                                Handling inheritProxy)
    {
        if (choice==null) {
            throw new IllegalArgumentException();
        }
        choice = simplify(choice,inheritProxy,null);
        return (choice==Handling.ACCEPT);
    }


    /**
     * Collapses a fine-grained handling into a simple "ignore" or
     * "no ignore" choice. This utility provides backward-compatibility
     * with the original AntX simple "ignore" or nothing handler
     * choices.
     * @param choice script value (non-null)
     * @param inheritProxy returned if choice is "inherit"
     **/
    public static Handling simplifyIgnoreOrNot(Handling choice,
                                               Handling inheritProxy)
    {
        if (choice==null) {
            throw new IllegalArgumentException();
        }
        switch (choice.getIndex()) {
            case REJECT_INDEX:
            case BALK_INDEX:
            case IGNORE_INDEX: {
                return IGNORE;
            }
            case ACCEPT_INDEX: {
                return ACCEPT;
            }
        }
        return inheritProxy;
    }
}

/* end-of-Handling.java */
