/**
 * $Id: CaseTransform.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  com.idaremedia.antx.helpers.Tk;

/**
 * Enumeration that represents the type of case transformations a task might implement.
 * The following list explains what the various settings mean:<ul>
 *    <li><span class="src">none</span>: The text should be left as-is.</li>
 *    <li><span class="src">lowercase</span>: The text should be lower-cased
 *         using the current locale.</li>
 *    <li><span class="src">uppercase</span>: The text should be UPPER-cased
 *        using the current locale.</li>
 *    <li><span class="src">invertcase</span>: The text's case should be
 *        inverted (UPPER to lower, lower to UPPER).</li>
 *    <li><span class="src">capitalize</span>: The text should be capitalized
 *        according to the task's definition.</li>
 * </ul>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   api,helper
 * @see      TransformHelper
 * @see      ValueTransform
 **/

public class CaseTransform extends EnumSkeleton
{
    //@.impl ORDERING of static declarations is important! ------

    //1) Indices (in order)

    /** Index of {@linkplain #NONE NONE}. **/
    public static final int NONE_INDEX = 0;
    /** Index of {@linkplain #LOWERCASE LOWERCASE}. **/
    public static final int LOWERCASE_INDEX = NONE_INDEX+1;
    /** Index of {@linkplain #UPPERCASE UPPERCASE}. **/
    public static final int UPPERCASE_INDEX = LOWERCASE_INDEX+1;
    /** Index of {@linkplain #INVERTCASE INVERTCASE}. **/
    public static final int INVERTCASE_INDEX = UPPERCASE_INDEX+1;
    /** Index of {@linkplain #CAPITALIZE CAPITALIZE}. **/
    public static final int CAPITALIZE_INDEX = INVERTCASE_INDEX+1;

    /** The number of base case transform values.
     *  @since JWare/AntX 0.4
     **/
    protected static final int BASE_COUNT = CAPITALIZE_INDEX+1;


    //2) Values (in order)

    /** Values in same order as public indices. **/
    private static final String[] VALUES_= new String[]
    {"none", "lowercase", "uppercase","invertcase","capitalize"};


    //3) Singletons (depend on Indices and Values already existing!)

    /** Singleton "<span class="src">none</span>" transform. **/
    public static final CaseTransform NONE=
        new CaseTransform(VALUES_[NONE_INDEX],NONE_INDEX);
    /** Singleton "<span class="src">lowercase</span>" transform. **/
    public static final CaseTransform LOWERCASE=
        new CaseTransform(VALUES_[LOWERCASE_INDEX],LOWERCASE_INDEX);
    /** Singleton "<span class="src">uppercase</span>" transform. **/
    public static final CaseTransform UPPERCASE=
        new CaseTransform(VALUES_[UPPERCASE_INDEX],UPPERCASE_INDEX);
    /** Singleton "<span class="src">invertcase</span>" transform. **/
    public static final CaseTransform INVERTCASE=
        new CaseTransform(VALUES_[INVERTCASE_INDEX],INVERTCASE_INDEX);
    /** Singleton "<span class="src">capitalize</span>" transform. **/
    public static final CaseTransform CAPITALIZE=
        new CaseTransform(VALUES_[CAPITALIZE_INDEX],CAPITALIZE_INDEX);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public CaseTransform()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensure it's initialized
     * as with default Ant Introspector helper thingy.
     **/
    private CaseTransform(String v, int i)
    {
        super(v);
    }



    /**
     * Returns a <em>copy</em> of the standard case transform values
     * in order. Subclasses can use this method to pre-fill their
     * value arrays with the inherited list.
     * @param fillin [optional] array of strings to update with values.
     * @since JWare/AntX 0.4
     **/
    public static String[] copyOfDefaultValues(String[] fillin)
    {
        if (fillin==null) {
            fillin = new String[VALUES_.length];
        }
        System.arraycopy(VALUES_,0,fillin,0,VALUES_.length);
        return fillin;
    }



    /**
     * Returns copy of all possible modification values as an ordered
     * string array. Note: ordering should be same as singletons indice.
     **/
    public String[] getValues()
    {
        return CaseTransform.copyOfDefaultValues(null);
    };



    /**
     * Helper that converts a scalar to a known CaseTransform. Returns <i>null</i>
     * if value does not match any of expected modifications.
     * @param i the index to be matched
     **/
    public static CaseTransform from(int i)
    {
        if (i==LOWERCASE.index)  { return LOWERCASE; }
        if (i==UPPERCASE.index)  { return UPPERCASE; }
        if (i==INVERTCASE.index) { return INVERTCASE; }
        if (i==CAPITALIZE.index) { return CAPITALIZE; }
        if (i==NONE.index)       { return NONE; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known CaseTransform's
     * index.
     * @param i the index to be matched
     * @param dflt the default CaseTransform if necessary
     **/
    public static CaseTransform from(int i, CaseTransform dflt)
    {
        CaseTransform choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known CaseTransform singleton.
     * Returns <i>null</i> if string unrecognized. String can be
     * either CaseTransform's symbolic name or its index.
     **/
    public static CaseTransform from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if ("default".equals(s))        { return NONE; }
                if (NONE.value.equals(s))       { return NONE; }
                if (LOWERCASE.value.equals(s))  { return LOWERCASE; }
                if (UPPERCASE.value.equals(s))  { return UPPERCASE;}
                if (INVERTCASE.value.equals(s)) { return INVERTCASE;}
                if (CAPITALIZE.value.equals(s)) { return CAPITALIZE;  }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(java.lang.String) from(String)} but
     * with a default value if value does not match any known
     * CaseTransform's name.
     * @param s the symbolic name to be matched
     * @param dflt the default CaseTransform if necessary
     **/
    public static CaseTransform from(String s, CaseTransform dflt)
    {
        CaseTransform choice= from(s);
        return (choice==null) ? dflt : choice;
    }
}

/* end-of-CaseTransform.java */
