/**
 * $Id: PropertiesIteratorTest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers.tests;

import  java.util.ArrayList;
import  java.util.Iterator;
import  java.util.Map;

import  junit.framework.TestSuite;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.Property;
import  org.apache.tools.ant.types.RegularExpression;

import  com.idaremedia.antx.ut.HTC;
import  com.idaremedia.antx.ut.HTCUtils;

import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.ownhelpers.InnerProperties;
import  com.idaremedia.antx.ownhelpers.PropertiesIterator;
import  com.idaremedia.antx.ownhelpers.UnresolvedProperty;
import  com.idaremedia.antx.parameters.PropertySource;

/**
 * Testsuite for {@linkplain PropertiesIterator}.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,test
 **/

public final class PropertiesIteratorTest extends HTC
{
    /** <i>PET</i> Test Category. **/
    public static final String TEST_CATEGORY="CLASS";


    /**
     * Create new PropertiesIteratorTest testcase.
     **/
    public PropertiesIteratorTest(String methodName)
    {
        super("PropertiesIterator::",methodName);
    }


    /**
     * Create full test suite for PropertiesIterator.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(PropertiesIteratorTest.class);
    }


    /**
     * Create baseline test suite for PropertiesIterator (same as full).
     **/
    public static TestSuite baseline()
    {
        return suite();
    }


    /**
     * Make this test (standalone) self-running.
     **/
    public static void main(String[] argv)
    {
        HTCUtils.quickCheck(suite());
    }


// ---------------------------------------------------------------------------------------------------------
// --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
// ---------------------------------------------------------------------------------------------------------

    protected void setUp() throws Exception
    {
        configureProjectFromResource("blank.xml");
    }


    private PropertiesIterator newOUT()
    {
        return new PropertiesIterator();
    }


    private PropertiesIterator newOUT(Project P)
    {
        PropertiesIterator out = newOUT();
        out.setProject(P);
        return out;
    }

    private PropertiesIterator newOUT(InnerProperties helper)
    {
        return new PropertiesIterator(helper);
    }

    private void disposeOUT(PropertiesIterator out)
    {
        out.dispose();
    }


// ---------------------------------------------------------------------------------------------------------
// ------------------------------------------- [ The Test Cases ] ------------------------------------------
// ---------------------------------------------------------------------------------------------------------

    public void checkBaseline()
    {
        //--Ensures setUp() works and can find our xml file!

        PropertiesIterator out = newOUT();

        assertIdent(out.getDomain(),PropertySource.ALL);
        assertFalse(out.isIgnoreCase());
        assertFalse(out.willCheckRE());

        disposeOUT(out);
    }


    public void testBaseline()
    {
        checkBaseline();
    }


    /**
     * Ensure we need to associate a live project with
     * iterator before it's usable.
     **/
    public void testFailNoProject_AntX04()
    {
        PropertiesIterator out = newOUT();
        try {
            out.hasNext();
            fail("Able to iterator unassociated iterator?!");
        } catch(Exception barfage) {
            assertExpected(barfage,"ProjectDependent:");
        }
        disposeOUT(out);
    }


    /**
     * Ensure you cannot remove properties via iterator.
     **/
    public void testFailRemoval_AntX04()
    {
        PropertiesIterator out = newOUT(getProject());
        try {
            assertTrue(out.hasNext());
            assertNotNil(out.next());
            out.remove();
            fail("Able to remove property via Iterator!?");
        } catch(UnsupportedOperationException Xpected) {
            /*burp*/
        }
        disposeOUT(out);
    }


    /**
     * Verify can iterate *ALL* properties in project.
     **/
    public void testIterateAllProperties_AntX04()
    {
        PropertiesIterator out = newOUT(getProject());
        Map mp = getProject().getProperties();
        final int N= mp.size();
        int i=0;
        while (out.hasNext()) {
            Object item = out.next();
            assertTrue((item instanceof Map.Entry),"Is Map.Entry @"+i);
            i++;
        }
        assertEqual(i,N,"PropertyCount");
        disposeOUT(out);
    }


    /**
     * Verify can iterate lite version of *ALL*.
     **/
    public void testIterateAllLiteProperties_AntX04()
    {
        PropertiesIterator out = newOUT();
        out.setProject(getProject());

        out.setDomain(PropertySource.from("all--"));
        assertIdent(out.getDomain(),PropertySource.ALLlite);

        Map nonJre = getProject().getProperties();
        Map jre = getProject().getProperties();

        final int fullCount= jre.size();

        FixtureExaminer.removeImmutableJavaProperties(nonJre);
        jre.keySet().removeAll(nonJre.keySet());

        final int nonJreCount= nonJre.size();
        assertTrue(nonJreCount<fullCount,
                   "JRE-bits-free map is smaller than original");

        int I=0;
        while (out.hasNext()) {
            Map.Entry mE = (Map.Entry)out.next();
            nonJre.remove(mE.getKey());
            assertFalse(jre.containsKey(mE.getKey()),"JRE bit");
            I++;
        }

        assertEqual(I,nonJreCount,"Iterated JRE-free bits");
        assertTrue(nonJre.isEmpty(),"JRE-bits removed as expected");

        jre= null;
        nonJre= null;
        disposeOUT(out);
    }


    /**
     * Verify can filter selection by regular expression.
     **/
    public void testREFilterAllDomain_AntX04()
    {
        Map mp = getProject().getProperties();
        Iterator itr= mp.keySet().iterator();
        ArrayList hits = new ArrayList();

        while (itr.hasNext()) {
            String key = (String)itr.next();
            if (key.startsWith("ant.")) {
                hits.add(key);
            }
        }
        assertFalse(hits.isEmpty(),"Unable to find any 'ant.' properties");
        assertTrue(hits.size()!=mp.size(),"Hits are subset of all");

        RegularExpression re= new RegularExpression();
        re.setPattern("^ant\\..*$");

        PropertiesIterator out = newOUT(getProject());
        out.setRE(re);
        assertTrue(out.willCheckRE(),"RE attached");

        while (out.hasNext()) {
            Map.Entry hit = (Map.Entry)out.next();
            assertTrue(hits.contains(hit.getKey()),"Result("+hit.getKey()+") expected");
            hits.remove(hit.getKey());
        }

        assertEqual(hits.size(),0,"Leftover Matches");

        disposeOUT(out);
    }


    /**
     * Verifies that unresolved properties are also returned
     * by default.
     **/
    public void testNoFilterUnresolvedByDefault_AntX04()
    {
        Project P = getProject();
        Map mp = P.getProperties();
        final int N = mp.size();

        Property setter = new Property();
        setter.setProject(P);

        setter.setName("__unresolved_1__");
        setter.setValue("${doinkdoinkdoinkdoinkdoink}");
        setter.execute();
        setter.setName("__unresolved_2__");
        setter.setValue("${OINKOINKOINKOINKOINK}");
        setter.execute();

        assertEqual(P.getProperty("__unresolved_1__"),
                    "${doinkdoinkdoinkdoinkdoink}","Unresolved_1");

        int i=0;
        PropertiesIterator out= newOUT();
        out.setProject(P);
        while (out.hasNext()) {
            out.next();
            i++;
        }

        assertEqual(i,N+2,"PropertyCount+Unresolved");

        disposeOUT(out);
    }


    /**
     * Verify can limit domain to CLI-based properties (and some
     * special Ant properites).
     **/
    public void testIterateCLIProperties_AntX04()
    {
        getProject().setUserProperty("ant.blablabla","hi!");

        Map mp = getProject().getProperties();
        final int N = mp.size();

        PropertiesIterator out = newOUT();
        out.setProject(getProject());

        out.setDomain(PropertySource.from("command"));
        assertIdent(out.getDomain(),PropertySource.COMMAND);

        int i=0,Nblablabla=0;
        while (out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String property = mE.getKey().toString();
            println(""+i+") "+property+"='"+mE.getValue()+"'");
            assertTrue(property.startsWith("ant."),"Property '"+property+"' is Ant property");
            if ("ant.blablabla".equals(property)) {
                Nblablabla++;
            }
            i++;
        }

        assertTrue(i<N,"CLI (much) lessthan ALL");
        assertEqual(Nblablabla,1,"InternalCommandCount");

        disposeOUT(out);
    }


    /**
     * Verify can limit domain to non-command/control properties.
     **/
    public void testIterateScriptOnlyProperties_AntX04()
    {
        getProject().setUserProperty("p.maybe","error!");
        getProject().setInheritedProperty("p.watup","errortoo!");

        PropertiesIterator out= newOUT(getProject());
        out.setDomain(PropertySource.SCRIPT);

        int N=0,LIMIT=20;
        while (out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String property = mE.getKey().toString();
            if (--LIMIT>0) {
                println(property+"='"+mE.getValue()+"'");
            }
            assertNotEqual("ant.file",property,"is 'ant.file'");
            assertNotEqual("p.maybe",property,"is 'p.maybe'");
            assertNotEqual("p.watup",property,"is 'p.watup'");
            N++;
        }
        println("Number SCRIPT properties: "+N);
        disposeOUT(out);
    }


    /**
     * Verify can limit domain to Java System properties.
     **/
    public void testIterateJavaRuntimeOnlyProperties_AntX04()
    {
        getProject().setNewProperty("_script.0", "broken");
        getProject().setUserProperty("_command.0", "broken too");
        getProject().setUserProperty("_control.0", "broken too too");

        PropertiesIterator out= newOUT(getProject());
        out.setDomain(PropertySource.JRE);

        int N=0,LIMIT=30;
        while (out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String property = mE.getKey().toString();
            if (--LIMIT>0) {
                println(property+"='"+mE.getValue()+"'");
            }
            assertNotEqual("ant.file",property,"is 'ant.file'");
            assertNotEqual("_script.0",property,"is 'script property'");
            assertNotEqual("_command.0",property,"is 'command property'");
            assertNotEqual("_control.0",property,"is 'control property'");
            N++;
        }
        println("Number of SYSTEM properties: "+N);
        disposeOUT(out);
    }


    /**
     * Verify can limit domain and apply a filter.
     **/
    public void testREFilterScriptProperties_AntX04()
    {
        getProject().setUserProperty("p.maybe","error!");
        getProject().setInheritedProperty("p.watup","errortoo!");

        PropertiesIterator out= newOUT(getProject());
        out.setDomain(PropertySource.SCRIPT);

        RegularExpression re= new RegularExpression();
        re.setPattern("^p\\.(true|false|maybe|watup)$");
        out.setRE(re);

        int i=0;
        while (out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String property = mE.getKey().toString();
            assertTrue("p.true".equals(property) || "p.false".equals(property),
                       "p.true =or= p.false");
            i++;
        }
        assertEqual(i,2,"MatchCount");

        disposeOUT(out);
    }


    /**
     * Verify can ignore properties with unresolved bits. Note that
     * the unresolved 'p.unresolved' property is defined in our test
     * build script.
     **/
    public void testIgnoreSimpleUnresolvedProperties_AntX04()
    {
        getProject().setUserProperty("p.doink","${doinkdoinkdoink}");

        PropertiesIterator out= newOUT(getProject());
        out.setCheckSimpleBrokenSubstitution();

        while (out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String property = mE.getKey().toString();
            if ("p.doink".equals(property) || "p.unresolved".equals(property)) {
                fail("Unresolved property("+property+","+mE.getValue()+
                     ") should be filtered");
            }
        }
        disposeOUT(out);
    }



    /**
     * Verify can ignore properties with unresolved bits. Note that
     * the unresolved 'p.unresolved' property is defined in our test
     * build script.
     **/
    public void testIgnoreAllUnresolvedProperties_AntX04()
    {
        getProject().setProperty("p.doink","hi_${doinkdoinkdoink}_");
        getProject().setProperty("p.superdoink","  ${bla} ${bla} ${doink}_");

        PropertiesIterator out= newOUT(getProject());
        out.setCheckBrokenSubstitution();

        while (out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String property = mE.getKey().toString();
            if ("p.doink".equals(property) || "p.unresolved".equals(property)
                || "p.superdoink".equals(property)) {
                fail("Unresolved property("+property+","+mE.getValue()+
                     ") should be filtered");
            }
        }
        disposeOUT(out);
    }


    /**
     * Verify can determine which properties are unresolved by assigning
     * our own proxy marker.
     **/
    public void testCanProxyUnresolvedProperties_AntX04()
    {
        getProject().setProperty("p.doink","hi_${doinkdoinkdoink}_");

        PropertiesIterator out= newOUT(getProject());
        out.setDomain(PropertySource.SCRIPT);
        out.setCheckBrokenSubstitution();
        out.setBrokenSubstitutionProxy(UnresolvedProperty.VALUE);

        ArrayList broken = new ArrayList();

        while (out.hasNext()) {
            Map.Entry mE = (Map.Entry)out.next();
            if (mE.getValue()==UnresolvedProperty.VALUE) {
                broken.add(mE);
            }
        }
        Iterator itr= broken.iterator();
        while (itr.hasNext()) {
            Map.Entry mE= (Map.Entry)itr.next();
            println(""+mE.getKey()+"='"+mE.getValue()+"'");
        }
        assertEqual(broken.size(),2,"BrokenCount");
        disposeOUT(out);
    }


    /**
     * Verify the 'ignorecase' option works for both prefix and regular
     * expression selection.
     **/
    public void testIgnoreCaseWorks_AntX04()
    {
        getProject().setProperty("BeGiN.one","ralph");
        getProject().setProperty("bEgIn.ONE","norton");
        getProject().setProperty("BEginError","doink!");

        PropertiesIterator out= newOUT(getProject());
        out.setDomain(PropertySource.SCRIPTlite);
        out.setIgnoreCase(true);

        out.setPrefix("BEgin.");
        int N=0;
        while (out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String name = (String)mE.getKey();
            assertFalse(name.startsWith("BEgin"),name+" not 'BEgin' prefix");
            N++;
        }
        assertEqual(N,2,"prefixMatchCount");

        out.reset();

        out.setPrefix(null);
        RegularExpression re= new RegularExpression();
        re.setPattern("^BEgin\\..*");
        out.setRE(re);

        N=0;
        while (out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String name = (String)mE.getKey();
            assertFalse(name.startsWith("BEgin"),name+" not 'BEgin' re");
            N++;
        }
        assertEqual(N,2,"reMatchCount");

        disposeOUT(out);
    }


    /**
     * Verify that initialization using an InnerProperties bean works
     * as expected. Also checks the "notLike" match variant.
     **/
    public void testInitByInnerProperties_AntX04()
    {
        Project P= getProject();
        P.setProperty("_fu.first", "first");
        P.setProperty("_fu.second", "second");
        P.setProperty("f_u.third", "third");

        InnerProperties bean = new InnerProperties();
        bean.setProject(P);
        bean.setDomain("script");
        bean.setNotLike("^_fu\\..*");

        PropertiesIterator out= newOUT(bean);
        assertTrue(out.isExclusion());
        assertFalse(out.willCheckPrefix());

        while(out.hasNext()) {
            Map.Entry mE= (Map.Entry)out.next();
            String name = (String)mE.getKey();
            assertFalse(name.startsWith("_fu."),name+" not '_fu.' re match");
        }

        disposeOUT(out);
    }
}

/* end-of-PropertiesIteratorTest.java */
