/**
 * $Id: FeedbackSink.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.EnumSkeleton;

/**
 * Enumeration that represents the possible types of (feedback) output sinks within
 * the Ant runtime. Includes elements like the standard Ant log and the System
 * stdio streams <i>err</i> and <i>out</i>.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   api,helper
 **/

public class FeedbackSink extends EnumSkeleton
{
    //@.impl ORDERING of static declarations is important! ------

    //1) Indices (in order)

    /** Index of {@linkplain #ALL ALL}. **/
    public static final int ALL_INDEX = 0; //@.impl must be zero!
    /** Index of {@linkplain #NONE NONE}. **/
    public static final int NONE_INDEX = ALL_INDEX+1;
    /** Index of {@linkplain #STDOUT STDOUT}. **/
    public static final int STDOUT_INDEX = NONE_INDEX+1;
    /** Index of {@linkplain #STDERR STDERR}. **/
    public static final int STDERR_INDEX = STDOUT_INDEX+1;
    /** Index of {@linkplain #STDIO STDIO}. **/
    public static final int STDIO_INDEX = STDERR_INDEX+1;
    /** Index of {@linkplain #ANTLOG ANTLOG}. **/
    public static final int ANTLOG_INDEX = STDIO_INDEX+1;
    /** Index of {@linkplain #EXTERNAL EXTERNAL}. **/
    public static final int EXTERNAL_INDEX = ANTLOG_INDEX+1;

    /** The number of base sink values. **/
    protected static final int BASE_COUNT = EXTERNAL_INDEX+1;


    //2) Values (in order)
   
    /** Values in same order as public indices. **/
    private static final String[] VALUES_= new String[]
    {"all", "none", "stdout","stderr","stdio", "antlog", "external"};


    //3) Singletons (depend on Indices and Values already existing!)
  
    /** Singleton "<span class="src">all</span>" choice. **/
    public static final FeedbackSink ALL=
        new FeedbackSink(VALUES_[ALL_INDEX],ALL_INDEX);
    /** Singleton "<span class="src">none</span>" choice. **/
    public static final FeedbackSink NONE=
        new FeedbackSink(VALUES_[NONE_INDEX],NONE_INDEX);

    /** Singleton "<span class="src">stdout</span>" choice. **/
    public static final FeedbackSink STDOUT=
        new FeedbackSink(VALUES_[STDOUT_INDEX],STDOUT_INDEX);
    /** Singleton "<span class="src">stderr</span>" choice. **/
    public static final FeedbackSink STDERR=
        new FeedbackSink(VALUES_[STDERR_INDEX],STDERR_INDEX);
    /** Singleton "<span class="src">stdio</span>" choice. **/
    public static final FeedbackSink STDIO=
        new FeedbackSink(VALUES_[STDIO_INDEX],STDIO_INDEX);


    /** Singleton "<span class="src">antlog</span>" choice. **/
    public static final FeedbackSink ANTLOG=
        new FeedbackSink(VALUES_[ANTLOG_INDEX],ANTLOG_INDEX);
    /** Singleton "<span class="src">external</span>" choice. **/
    public static final FeedbackSink EXTERNAL=
        new FeedbackSink(VALUES_[EXTERNAL_INDEX],EXTERNAL_INDEX);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public FeedbackSink()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensure this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    protected FeedbackSink(String v, int i)
    {
        super(v);
    }



    /**
     * Returns a <em>copy</em> of the standard feedback sink values
     * in order. Subclasses can use this method to pre-fill their
     * value arrays with the inherited list.
     * @param fillin [optional] array of strings to update with values.
     **/
    public static String[] copyOfDefaultValues(String[] fillin)
    {
        if (fillin==null) {
            fillin = new String[VALUES_.length];
        }
        System.arraycopy(VALUES_,0,fillin,0,VALUES_.length);
        return fillin;
    }



    /**
     * Returns copy of all possible feedback sink values as an ordered
     * string array. Note: ordering should be same as singletons indice.
     **/
    public String[] getValues()
    {
        return FeedbackSink.copyOfDefaultValues(null);
    }



    /**
     * Helper that converts a scalar to a known FeedbackSink. Returns
     * <i>null</i> if value does not match any of expected sinks.
     * @param i the index to be matched
     **/
    public static FeedbackSink from(int i)
    {
        if (i==ANTLOG.index)   { return ANTLOG; }
        if (i==STDIO.index)    { return STDIO; }
        if (i==STDOUT.index)   { return STDOUT; }
        if (i==STDERR.index)   { return STDERR; }
        if (i==EXTERNAL.index) { return EXTERNAL; }
        if (i==ALL.index)      { return ALL; }
        if (i==NONE.index)     { return NONE; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known FeedbackSink's
     * index.
     * @param i the index to be matched
     * @param dflt the default FeedbackSink if necessary
     **/
    public static FeedbackSink from(int i, FeedbackSink dflt)
    {
        FeedbackSink choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known FeedbackSink singleton.
     * Returns <i>null</i> if string unrecognized. String can be
     * either FeedbackSink's symbolic name or its index.
     **/
    public static FeedbackSink from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (STDIO.value.equals(s))    { return STDIO; }
                if (STDERR.value.equals(s))   { return STDERR; }
                if (STDOUT.value.equals(s))   { return STDOUT; }
                if (ANTLOG.value.equals(s))   { return ANTLOG; }
                if ("stdlog".equals(s))       { return ANTLOG; }
                if (EXTERNAL.value.equals(s)) { return EXTERNAL; }
                if (ALL.value.equals(s))      { return ALL; }
                if (NONE.value.equals(s))     { return NONE; }
                if (Strings.DEFAULT.equals(s))  { return STDIO; }
                if ("both".equals(s))         { return STDIO; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but with a
     * default value if supplied value does not match any known
     * FeedbackSink's name.
     * @param s the symbolic name to be matched
     * @param dflt the default FeedbackSink if necessary
     **/
    public static FeedbackSink from(String s, FeedbackSink dflt)
    {
        FeedbackSink choice= from(s);
        return (choice==null) ? dflt : choice;
    }



    /**
     * Utility method that verifies the given sink is one of the standard
     * system output streams (err|out). A <i>null</i> sink value always
     * returns <i>false</i>.
     * @param ss the feedback sink value
     **/
    public static final boolean isSystem(FeedbackSink ss)
    {
        if (ss!=null) {
            int i= ss.getIndex();
            if (i==FeedbackSink.STDIO_INDEX  ||
                i==FeedbackSink.STDOUT_INDEX || i==FeedbackSink.STDERR_INDEX) {
                return true;
            }
        }
        return false;
    }


    /**
     * Utility method that verifies the given sink is one of the internal
     * Ant process output streams (err|out|antlogs). A <i>null</i> sink
     * value always returns <i>false</i>.
     * @param ss the feedback sink value
     **/
    public static boolean isKnown(FeedbackSink ss)
    {
        if (ss!=null) {
            int i= ss.getIndex();
            if (i==FeedbackSink.STDIO_INDEX  || i==FeedbackSink.ANTLOG_INDEX ||
                i==FeedbackSink.STDOUT_INDEX || i==FeedbackSink.STDERR_INDEX) {
                return true;
            }
        }
        return false;
    }



    /**
     * Utility method that verifies the given sink is one of the general
     * sources of output streams (antlogs|stdio). A <i>null</i> sink
     * value always returns <i>false</i>.
     * @param ss the feedback sink value
     **/
    public static boolean isGrouping(FeedbackSink ss)
    {
        if (ss!=null) {
            int i= ss.getIndex();
            if (i==FeedbackSink.STDIO_INDEX || i==FeedbackSink.ANTLOG_INDEX) {
                return true;
            }
        }
        return false;
    }
}

/* end-of-FeedbackSink.java */
