/**
 * $Id: UISMConfigureTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.init;

import  java.io.File;
import  java.net.URL;
import  java.text.MessageFormat;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Path;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.apis.BundleStringManager;
import  com.idaremedia.apis.UIStringManager;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureOverlay;
import  com.idaremedia.antx.UISMContext;
import  com.idaremedia.antx.apis.BuildError;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.apis.Responses;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * Configuration task that adds another UIStringManager for use by nested AntX
 * resource bundle aware tasks in a shared project. The effect of a UISMConfigureTask
 * is like a logical 'or'-- if the nearest task's UIStringManager does not contain
 * the requested msgid, its preceding UIStringManager (another configure task or a
 * root UIStringManager) is asked to find the msgid. This delegation continues until
 * the msgid is located or there are no more UIStringManagers to ask. Note that the
 * default resource bundle is not automatically searched like the root bundle.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded (from UIStringManager interface once configured)
 * @.group   api,infra
 **/

public class UISMConfigureTask extends TaskSet
    implements FixtureOverlay, UISMSource, UIStringManager
{
    /**
     * Initializes a new UISMConfigureTask instance.
     **/
    public UISMConfigureTask()
    {
        super(AntX.uism);
    }


    /**
     * Initializes a new CV-tagged UISMConfigureTask instance.
     * @param iam CV-label (non-null)
     **/
    public UISMConfigureTask(String iam)
    {
        super(iam);
    }


    /**
     * Sets this task's project; updates other underlying
     * project-components.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_bundle.setProject(P);
    }


    /**
     * Initializes this taskset's problem handler instance.
     * @since JWare/AntX 0.4
     **/
    public void init()
    {
        super.init();
        m_boobooNotify = new Responses.LogUsing(this);
    }


    /**
     * Sets this tasks bundle as reference to an existing
     * msgs bundle declaration.
     * @param reference referenc to bundle (non-null)
     **/
    public void setBundleId(Reference reference)
    {
        m_bundle.setRefid(reference);
    }

// ---------------------------------------------------------------------------------------
// UISMSource implementation (inherit javadoc comments):
// ---------------------------------------------------------------------------------------

    public void setURL(String url)
    {
        m_bundle.setURL(url);
    }

    public URL getURL()
    {
        return m_bundle.getURL();
    }

    public void setFile(String filepath)
    {
        m_bundle.setFile(filepath);
    }

    public File getFile()
    {
        return m_bundle.getFile();
    }

    public void setResource(String rsrc)
    {
        m_bundle.setResource(rsrc);
    }

    public String getResource()
    {
        return m_bundle.getResource();
    }


// ---------------------------------------------------------------------------------------
// Resource loading implementation (subset of CustomLoaderEnabled):
// ---------------------------------------------------------------------------------------

    /**
     * Defines a simple lookup path for this taskset's bundle
     * resource.
     * @param classpath the path to be searched (non-null)
     * @see #setClassPathRef
     **/
    public void setClassPath(Path classpath)
    {
        m_bundle.setClassPath(classpath);
    }


    /**
     * Defines a custom lookup path for this taskset's bundle
     * resource by (re)using an existing path reference.
     * @param r reference to an existing path (non-null)
     **/
    public void setClassPathRef(Reference r)
    {
        m_bundle.setClassPathRef(r);
    }



    /**
     * Returns the custom lookup path this taskset will use when
     * looking for its resource bundle. Will returns <i>null</i>
     * if lookup path has not been customized.
     **/
    public Path getClassPath()
    {
        return m_bundle.getClassPath();
    }


    /**
     * Tells this taskset to use an existing classloader when
     * searching and/or loading bundle resource.
     * @param r reference to an existing ClassLoader (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setLoaderRef(Reference r)
    {
        m_bundle.setLoaderRef(r);
    }


    /**
     * Returns the identifier of the custom classloader this taskset
     * will use to lookup its bundle resource. Returns <i>null</i>
     * if no custom class loader.
     * @since JWare/AntX 0.4
     **/
    public String getLoaderRefId()
    {
        return m_bundle.getLoaderRefId();
    }


// ---------------------------------------------------------------------------------------
// UIStringManager implementation (inherit javadoc comments):
// ---------------------------------------------------------------------------------------

    /** Used as marker for strings that aren't in my immediate bundle. **/
    private static final String NOSTR= new String("==_\\\"");//NB:unlikely legit entry


    public String mget(String id, Object[] args, String defm)
    {
        BundleStringManager uism = getThisUISM();

        String msg = uism.mget(id,args,NOSTR);

        if (NOSTR.equals(msg)) {
            if (!ignoreAllInherited() && m_inheritedUISM!=null) {
                msg = m_inheritedUISM.mget(id,args,defm);
            } else if (defm!=null) {
                msg = defm;
            } else {
                msg = uism.getDefaultString();
            }
        }
        return msg;
    }


    public String mget(MessageFormat mf, String id, Object[] args, String defm)
    {
        BundleStringManager uism = getThisUISM();

        String msg = uism.mget(mf,id,args,NOSTR);
        if (NOSTR.equals(msg)) {
            if (!ignoreAllInherited() && m_inheritedUISM!=null) {
                msg = m_inheritedUISM.mget(mf,id,args,defm);
            } else if (defm!=null) {
                msg = defm;
            } else {
                msg = uism.getDefaultString();
            }
        }
        return msg;
    }


    public final String mget(String id, Object[] args)
    {
        return mget(id,args,null);
    }

    public final String get(String id)
    {
        return mget(id,(Object[])null,(String)null);
    }

    public final String dget(String id, String defm)
    {
        return mget(id,(Object[])null, defm);
    }

    public final String get(String id, Object arg1)
    {
        return mget(id, new Object[]{arg1}, null);
    }

    public final String dget(String id, Object arg1, String defm)
    {
        return mget(id, new Object[]{arg1}, defm);
    }

    public final String get(String id, Object arg1, Object arg2)
    {
        return mget(id, new Object[]{arg1,arg2}, null);
    }

    public final String dget(String id, Object arg1, Object arg2, String defm)
    {
        return mget(id, new Object[]{arg1,arg2}, defm);
    }

    public final String get(String id, Object arg1, Object arg2, Object arg3)
    {
        return mget(id, new Object[]{arg1,arg2,arg3}, null);
    }

    public final String dget(String id, Object arg1, Object arg2, Object arg3, String defm)
    {
        return mget(id, new Object[]{arg1,arg2,arg3}, defm);
    }

    public final String getDefaultString()
    {
        return getThisUISM().getDefaultString();
    }

// ---------------------------------------------------------------------------------------
// Taskset/controller management
// ---------------------------------------------------------------------------------------

    /**
     * Returns <i>true</i> if this task ignores its inherited UIStringManager
     * and returns only messages contained in its bundle. Is <i>false</i>
     * by default.
     **/
    protected final boolean ignoreAllInherited()
    {
        return m_noInherited;
    }


    /**
     * Set whether this task will use the UIStringManager hierarchy or
     * return only messages contained in its bundle.
     * @param passthru <i>false</i> if this task should ignore the
     *        the UIStringManager hierarchy.
     **/
    public void setInheritance(boolean passthru)
    {
        m_noInherited = !passthru;
    }


    /**
     * Returns this task's underlying UIStringManager. Never
     * returns <i>null</i>.
     * @.safety single
     **/
    private BundleStringManager getThisUISM()
    {
        if (m_UISM==null) {
            m_UISM = new BundleStringManager
                (m_bundle.newPropertyBundle(m_boobooNotify),null);
        }
        return m_UISM;
    }


    /**
     * Installs this UIStringManager as the frontmost within the current
     * thread's iteration environment before running nested tasks. When tasks
     * have finished (or failed), uninstalls self from UISM context stack.
     * @throws BuildException if inner tasks do, or we cannot install UIStringManager
     * @throws BuildError if iteration UIStringManager stack is fatally corrupted
     * @see UISMContext
     **/
    protected void performNestedTasks()
        throws BuildException
    {
        synchronized(m_runlock) {
            boolean installed=false;
            try {
                m_inheritedUISM = UISMContext.installStringManager(this,m_boobooNotify);
                installed= true;
                getThisUISM();
                super.performNestedTasks();

            } finally {
                if (installed) {
                    m_inheritedUISM = null;
                    UIStringManager uism = UISMContext.getStringManager();
                    if (uism!=this) {
                        String ME= uistrs().dget("task.uism.whoami","UIStringManager");
                        String ermsg = uistrs().get("context.stack.corrupted",ME);
                        log(ermsg, Project.MSG_ERR);
                        throw new BuildError(ermsg,getLocation());
                    }
                    UISMContext.unwindStringManager(null);
                }//installed
            }
        }
    }


    private boolean m_noInherited;//NB:=>use inheritance hierarchy
    private UIStringManager m_inheritedUISM;//NB:on execution
    private UISMBundle m_bundle= new UISMBundle();//NB:delegatee for configuration
    private BundleStringManager m_UISM;//NB:inited on 1st use
    private Object m_runlock = new int[0];//NB:prevent nested execution
    private ProblemHandler m_boobooNotify;//NB:inited by init()
}

/* end-of-UISMConfigureTask.java */
