/**
 * $Id: CopyMsgTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.init;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ExportScope;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.parameters.FlexValueSupport;
import  com.idaremedia.antx.starters.MsgTask;

/**
 * Helper task that copies a resource bundle based message to a regular Ant property
 * or an AntX exported property. Optionally applies runtime arguments to templated
 * strings.
 * <p>
 * Examples: <pre>
 *   &lt;copymsg msgid="msg.helloworld" property="default.greeting" msgarg1="${TODAY}"/&gt;
 *   &lt;copymsg msgid="defaults.copyleft" variable="copyleft"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 **/

public final class CopyMsgTask extends MsgTask
    implements FlexValueSupport, AntLibFriendly
{
    /**
     * The default scope used by all CopyMsgTasks.
     **/
    public static final ExportScope DEFAULT_SCOPE= ExportScope.THREAD;


    /**
     * Initializes a new CopyMsgTask.
     **/
    public CopyMsgTask()
    {
        super(AntX.uism);
    }


    /**
     * Setup this copy task to update the specified property
     * with generated message.
     * @param property name of property (non-null)
     **/
    public void setProperty(String property)
    {
        require_(property!=null, "setProp- nonzro name");
        checkNewCopyTarget();
        m_value.set(property);
        m_value.setIsProperty(true);
    }


    /**
     * Setup this copy task to update the specified exported
     * property with generated message.
     * @param property name of exported property (non-null)
     **/
    public void setVariable(String property)
    {
        require_(property!=null, "setVar- nonzro name");
        checkNewCopyTarget();
        m_value.set(property);
        m_value.setIsExported(true);
    }


    /**
     * Sets a variable property's scope. Defaults to the
     * <i>THREAD</i> scope if undefined.
     **/
    public void setScope(ExportScope scope)
    {
        require_(scope!=null,"setScop- nonzro");
        m_scope = scope;
    }


    /**
     * Returns this task's to-be-created variable's scope.
     * Never returns <i>null</i>; defaults to <i>THREAD</i>.
     **/
    public final ExportScope getScope()
    {
        return m_scope;
    }


    /**
     * Setup this copy task to update the specified project
     * reference with generated message.
     * @param refid name of (new) reference (non-null)
     **/
    public void setReference(String refid)
    {
        require_(refid!=null, "setRef- nonzro refid");
        checkNewCopyTarget();
        m_value.set(refid);
        m_value.setIsReference(true);
    }


    /**
     * Copies this item's generated message to its target.
     **/
    public void execute()
    {
        verifyCanExecute_("execute");

        if (testIfCondition() && testUnlessCondition()) {
            String what = getMsg();
            String key  = m_value.get();
            Project P   = getProject();

            if (m_value.isProperty()) {
                if (ExportScope.ALL.equals(getScope())) {
                    P.setInheritedProperty(key,what);
                } else {
                    P.setNewProperty(key,what);
                }
            }
            else if (m_value.isExported())  {
                ExportScope.setTheProperty(getScope(),P, key,what,false);
            }
            else if (m_value.isReference()) {
                P.addReference(key,what);
            }
        }
    }


    /**
     * Verifies that one of the three target holders has been
     * specified.
     * @throws BuildException if a target has not been specified
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        String ermsg = null;
        if (m_value.isLiteral()) {
            ermsg = uistrs().get("task.needs.this.attr", getTaskName(),
                                 "property|reference|variable");
        } else if (getMsgId()==null && getDefaultMsg().length()==0) {
            ermsg = uistrs().get("task.needs.this.attr", getTaskName(),
                                 "msgid|message");
        }
        if (ermsg!=null) {
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    /**
     * Ensures that a new 'set' isn't going to overwrite a previously
     * set copy target (only one permitted).
     * @throws BuildException if a copy target has already been specified
     **/
    private void checkNewCopyTarget()
    {
        if (m_copytargets>0) {
            String ermsg = uistrs().get("task.too.many.flex.attrs");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        m_copytargets++;
    }


    private FlexString m_value  = new FlexString();
    private ExportScope m_scope = ExportScope.DEFAULT_SCOPE;
    private int m_copytargets;
}

/* end-of-CopyMsgTask.java */
