/**
 * $Id: Tk.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.lang.reflect.Field;
import  java.lang.reflect.Method;
import  java.lang.reflect.Modifier;
import  java.io.BufferedReader;
import  java.io.File;
import  java.io.InputStream;
import  java.io.InputStreamReader;
import  java.io.OutputStream;
import  java.net.URL;
import  java.util.ArrayList;
import  java.util.Collections;
import  java.util.List;
import  java.util.Locale;
import  java.util.Map;
import  java.util.StringTokenizer;

import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.Property;
import  org.apache.tools.ant.types.EnumeratedAttribute;
import  org.apache.tools.ant.util.LoaderUtils;

import  com.idaremedia.apis.FixtureStarter;

import  com.idaremedia.antx.apis.Requester;

/**
 * Collection of independent, common utility algorithms. Snatched from JWare/foundation.
 * <b>These helper methods <em>must</em> remain independent of other AntX classes.</b>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 **/

public final class Tk
{
    /** A marker for an undefined integer value. **/
    public static final int NO_INT= Integer.MAX_VALUE;

    /** A marker for an undefined long, timestamp, or date value. **/
    public static final long NO_NUM= Long.MAX_VALUE;


    /** A marker for an undefined long, timestamp, or date value
     *  as a <span class="src">Long</span> object.
     *  @since JWare/AntX 0.5
     **/
    public static final Long NO_NUM_NUM= new Long(Long.MAX_VALUE);



    /**
     * Checks if the given string is all whitespace. Note: if the given
     * string is either <i>null</i> or the empty string it's considered
     * white space.
     * @param s the string to check
     * @return <i>true</i> if whitespace (including <i>null</i> and empty)
     **/
    public static boolean isWhitespace(String s)
    {
        int n=0;
        if (s==null || (n=s.length())==0) {
            return true;
        }
        for (int i=0; i<n;i++) {
            if (!Character.isWhitespace(s.charAt(i))) {
                return false;
            }
        }
        return true;
    }



    /**
     * Strips the given string of all whitespace characters (as defined
     * by java method <span class="src">Character.isWhitespace</span>).
     * Returns original string if no whitespace detected.
     * @param s the string to check
     * @return new stripped string of original if no whitespace
     **/
    public static String stripWhitespace(String s)
    {
        StringBuffer sb=null;

        int n=0;
        if (s==null || (n=s.length())==0) {
            return s;
        }

        char c;
        for (int i=0; i<n;i++) {
            c = s.charAt(i);
            if (!Character.isWhitespace(c)) {
                if (sb==null) {
                    sb= new StringBuffer(n);
                }
                sb.append(c);
            }
        }
        return (sb!=null) ? sb.toString() : s;
    }



    /**
     * Helper that extracts a list of individual names from a delimited
     * list of names. Caller assumes ownership of returned (single-threaded)
     * list.
     * @param delimitedList delimited list
     * @param delim the delimiter string
     **/
    public final static List splitList(String delimitedList, String delim)
    {
        StringTokenizer st= new StringTokenizer(delimitedList,delim);
        int N= st.countTokens();
        List l= new ArrayList(Math.max(N,4));
        while (st.hasMoreTokens()) {
            l.add(st.nextToken());
        }
        return l;
    }


    /**
     * Variant of {@linkplain #splitList(String,String) splitList(&#8230;)}
     * that expects a comma-delimited list.
     * @param delimitedList the comma-delimited list
     **/
    public final static List splitList(String delimitedList)
    {
        return splitList(delimitedList,",");
    }


    /**
     * Using specified string, returns best-guess conversion for
     * an assertable's CV label.
     **/
    public static String cvlabelFrom(String iam)
    {
        if (iam==null) {
            iam="";
        } else if (iam.endsWith(".")) {
            iam = iam.substring(0,iam.length()-1)+":";
        } else if (!iam.endsWith(":")){
            iam += ":";
        }
        return iam;
    }


    /**
     * Converts a string to its US lowercase equivalent. Normalizes
     * to lowercase the standard way Ant does. If incoming string
     * is <i>null</i>, this method returns <i>null</i>.
     * @param s the string to lowercase
     **/
    public static String lowercaseFrom(String s)
    {
        if (s==null) {
            return null;
        }
        return String.valueOf(s).toLowerCase(Locale.US);
    }



    /**
     * Converts a string to its US lowercase equivalent. Normalizes
     * to lowercase the standard way Ant does. If incoming string
     * is <i>null</i> this method returns the string
     * "<span class="src">null</span>".
     * @param s the string to lowercase
     * @since JWare/AntX 0.4
     **/
    public static String lowercaseFromNoNull(String s)
    {
        return String.valueOf(s).toLowerCase(Locale.US);
    }



    /**
     * Returns a generic system identity string for given object.
     * @param thing the thing to be stringified (can be <i>null</i>)
     * @since JWare/AntX 0.4
     **/
    public static String identityStringFrom(Object thing)
    {
        if (thing==null) {
            return Strings.NULL;
        }
        return thing.getClass().getName()
            +"@"
            +System.identityHashCode(thing);
    }



    /**
     * Tries to obtain the result of an object's
     * <span class="src">toString()</span> handling. If the object's
     * handling generates an exception (often the case in Ant 1.6+)
     * this method returns a generic system identity string.
     * @param thing the thing to be stringified (can be <i>null</i>)
     * @param P [optional] project for problem logging purposes
     * @since JWare/AntX 0.4
     **/
    public static String stringFrom(Object thing, Project P)
    {
        try {
            return String.valueOf(thing);
        } catch(RuntimeException rtX) {
            if (P!=null) {
                P.log(rtX.getMessage(),Project.MSG_INFO);
            }
            return identityStringFrom(thing);
        }
    }


    /**
     * Converts given object to an <code>int</code> or a default value
     * <code>defi</code> if given object is <i>null</i> or not parseable.
     * Understands various string representations like octal,
     * hexidecimal, etc.
     *
     * @param o the conversion source
     * @param defi the default value
     * @return the value as an integer or <code>defi</code>
     **/
    public static int integerFrom(Object o, int defi)
    {
        int i= defi;
        if (o!=null) {
            try {
                if (o instanceof Number) {
                    i= ((Number)o).intValue();
                } else if (o instanceof Boolean) {
                    i= ((Boolean)o).booleanValue()==true ? 1 : 0;
                } else {
                    i= Integer.decode(o.toString()).intValue();
                }
            }
            catch (NumberFormatException nx) {/*burp*/}
        }
        return i;
    }


    /**
     * Converts given object to a <code>long</code> or a default value
     * <code>defl</code> if given object is <i>null</i> or not parseable.
     *
     * @param o the conversion source
     * @param defl the default value
     * @return the value as a long or <code>defl</code>
     **/
    public static long longFrom(Object o, long defl)
    {
        long l= defl;
        if (o!=null) {
            try {
                if (o instanceof Number) {
                    l= ((Number)o).longValue();
                } else if (o instanceof Boolean) {
                    l= ((Boolean)o).booleanValue()==true ? 1 : 0;
                } else {
                    l= Long.decode(o.toString()).longValue();
                }
            }
            catch (NumberFormatException nx) {/*burp*/}
        }
        return l;
    }



    /**
     * Converts a string value to a number (long|float) or returns a symbolic
     * {@linkplain Tk#NO_NUM_NUM NO_NUM_NUM} if string could not be converted.
     * @param valu the numeric value as a string (all of string is read).
     * @since JWare/AntX 0.5
     **/
    public static Number numberFrom(String valu)
    {
        Number n = Tk.NO_NUM_NUM;
        long l = Tk.longFrom(valu,Tk.NO_NUM);
        if (l!=Tk.NO_NUM) {
            n = new Long(l);
        } else {
            try {
                n = Double.valueOf(valu);
            } catch(NumberFormatException nfx) {/*burp*/}
        }
        return n;
    }



    /**
     * Wrapper for Project.toBoolean that handles <i>null</i>
     * string values.
     * @param s string to be evaluated (can be <i>null</i>)
     **/
    public static boolean booleanFrom(String s)
    {
        Boolean B = string2PosBool(s);
        return (B==null) ? false : B.booleanValue();
    }


    /**
     * Converts a string form (possibly <i>null</i>) to a positive <i>Boolean</i>
     * value. Usually used by interactive frontends in testing. Considers any
     * of the following to be <i>true</i> (case is ignored): "<i>true</i>",
     * "<i>on</i>", "<i>1</i>", "<i>ok</i>", or "<i>yes</i>". No matches
     * return <i>false</i>.
     * @param s stringified value
     * @return <i>null</i> if <code>s</code> is <i>null</i> or whitespace.
     *         Returns <i>false</i> if s not recognized as a positive boolean
     **/
    public static final Boolean string2PosBool(String s)
    {
        if (Tk.isWhitespace(s)) {
            return null;
        }
        s= Tk.lowercaseFrom(s);
        if (s.length()>4) {//=="true".length()
            return Boolean.FALSE;
        }
        if (s.equals("true")) {
            return Boolean.TRUE;
        }
        if (s.equals("1")) {
            return Boolean.TRUE;
        }
        if (s.equals("on"))  {
            return Boolean.TRUE;
        }
        if (s.equals("yes"))  {
            return Boolean.TRUE;
        }
        if (s.equals("t")) {
            return Boolean.TRUE;
        }
        if (s.equals("ok")) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }


    /**
     * Converts a string form (possibly <i>null</i>) to a negative <i>Boolean</i>
     * value. Usually used by interactive frontends in testing. Considers any
     * of the following to be <i>false</i> (case is ignored): "<i>false</i>",
     * "<i>off</i>", "<i>0</i>", "<i>none</i>", or "<i>no</i>".
     * @param s stringified value
     * @return <i>null</i> if <code>s</code> is <i>null</i> or whitespace.
     *         Returns <i>true</i> if is not recognized as a negative boolean
     **/
    public static final Boolean string2NegBool(String s)
    {
        if (Tk.isWhitespace(s)) {
            return null;
        }
        s= Tk.lowercaseFrom(s);
        if (s.length()>5) {//=="false".length()
            return Boolean.TRUE;
        }
        if (s.equals("false")) {
            return Boolean.FALSE;
        }
        if (s.equals("0"))   {
            return Boolean.FALSE;
        }
        if (s.equals("off"))  {
            return Boolean.FALSE;
        }
        if (s.equals("no"))  {
            return Boolean.FALSE;
        }
        if (s.equals("none"))  {
            return Boolean.FALSE;
        }
        if (s.equals("f")) {
            return Boolean.FALSE;
        }
        if (s.equals("notok")) {
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }


    /**
     * Returns the leaf of a class's name. If the leaf partion of the
     * class name includes the '$' character (as for inner or anonymous
     * classes), all occurances of  '$' are replaced by a '-'
     * character.
     * @param cls the class (non-null)
     **/
    public static String leafNameFrom(Class cls)
    {
        if (cls==null) {
            throw new IllegalArgumentException("leafFrom- NULL class");
        }
        String cn = cls.getName();
        int i= cn.lastIndexOf('.');
        if (i != -1) {
            return cn.substring(i+1).replace('$','-');
        }
        return cn;
    }


    /**
     * Helper to initialize an enumeration value at construction w/o
     * dealing with check build exception.
     * @return <i>true</i> if no barfage on <i>setValue</i>
     **/
    public static boolean initEnum(EnumeratedAttribute e, String ini)
    {
        try {
            e.setValue(ini);
        } catch(Exception x) {
            return false;
        }
        return true;
    }



    /**
     * Helper to convert a file URL to a File object's name.
     **/
    public static String toFilename(URL url)
    {
        if (!url.getProtocol().equals("file")) {
            if (url.getProtocol().equals("jar")) {
                String fn = url.getPath();
                if (fn.startsWith("file:")) {
                    return fn.substring(5).replace
                        ('/', File.separatorChar).replace('|', ':');
                }
            }
            return null;
        }
        return url.getPath().replace('/', File.separatorChar).replace('|', ':');
    }



    /**
     * Helper to convert a file URL to a Java/IO File object.
     **/
    public static File toFile(URL url)
    {
        String fn = toFilename(url);
        return (fn==null) ? null : new File(fn);
    }



    /**
     * Quietly closes an opened input stream. Any I/O or security
     * exceptions are ignored.
     * @param input the stream to close (non-null)
     * @return <i>true</i> if no problems encountered
     * @since JWare/AntX 0.5
     **/
    public static boolean closeQuietly(InputStream input)
    {
        try {
            input.close();
            return true;
        } catch(Exception ioX) {
            return false;
        }
    }
    


    /**
     * Quietly closes an opened output stream. Any I/O or security
     * exceptions are ignored.
     * @param output the stream to close (non-null)
     * @return <i>true</i> if no problems encountered
     * @since JWare/AntX 0.5
     **/
    public static boolean closeQuietly(OutputStream output)
    {
        try {
            output.close();
            return true;
        } catch(Exception ioX) {
            return false;
        }
    }



    /**
     * Copies the configuration of one property to another. Often used
     * when calling a target and caller needs to pass on custom property
     * information to target.
     * @param source source property (non-null)
     * @param dest destination property (non-null)
     **/
    public static void transferProperty(Property source, Property dest)
    {
        if (source==null || dest==null) {
            throw new IllegalArgumentException("transferProperty- NULL source or dest");
        }

        if (source.getName()!=null) {//possible if file|rez|prefix
            dest.setName(source.getName());
        }
        if (source.getRefid()!=null) {
            dest.setRefid(source.getRefid());
        }
        if (source.getValue()!=null) {//NB:also covers locations
            dest.setValue(source.getValue());
        }
        if (source.getFile()!=null) {
            dest.setFile(source.getFile());
        }
        if (source.getUrl()!=null) {
            dest.setUrl(source.getUrl());//copy?
        }
        if (source.getResource()!=null) {
            dest.setResource(source.getResource());
        }
        if (source.getEnvironment()!=null) {
            dest.setEnvironment(source.getEnvironment());
        }
        if (source.getPrefix()!=null) {
            dest.setPrefix(source.getPrefix());
        }
        if (source.getClasspath()!=null) {
            dest.setClasspath(source.getClasspath());
        }

    }



    /**
     * Retrieves a property from either an existing project or the default
     * Java system object if necessary. The System is only checked if the
     * project is <i>null</i> or does not contain the named property.
     * @param P [optional] executing project
     * @param property the property's name
     * @return property value (can be <i>null</i>)
     **/
    public static final String getTheProperty(Project P, String property)
    {
        String value=null;

        if (P!=null) {
            value = P.getProperty(property);
        }
        if (value==null) {
            value = FixtureStarter.getProperty(property);
        }
        return value;
    }



    /**
     * Helper to set a property based on an existing project. If passthru
     * is <i>true</i> then sets property in velcro mode-- the setting
     * sticks forever (including to sub-antcalls). Tries to set a System
     * property if project is <i>null</i>.
     * @param P [optional] executing project
     * @param property the property's name (non-null)
     * @param value the property's value (non-null)
     * @throws java.lang.SecurityException if not permission to set System
     *         properties
     **/
    public static final void setTheProperty(Project P, String property,
                                            String value, boolean passthru)
    {
        if (P!=null) {
            if (passthru) {
                P.setUserProperty(property,value);//like velcro...
            } else {
                P.setNewProperty(property, value);
            }
        } else {
            System.setProperty(property,value);//NB:Need proper perms
            FixtureStarter.unsetProperty(property);//NB:do *after* system-set OK
        }
    }


    /**
     * Makes a Location string human-readable (manageable).
     * @param l the location to be stringified (non-null)
     **/
    public static String shortStringFrom(Location l)
    {
        if (l==null || Location.UNKNOWN_LOCATION.equals(l)) {
            return Strings.UNDEFINED;
        }
        String fileline = l.toString();
        int i= fileline.lastIndexOf(':');
        if (i>0) {
            fileline= fileline.substring(0,i);
        }
        int N= fileline.length();
        if (N>41) {
            fileline= "..."+fileline.substring(N-38);
        }
        return fileline;
    }


    /**
     * Makes a Location string human-readable (manageable) if 
     * application's defaults allow it.
     * @param prefer <i>true</i> if shortening is allowed
     * @param l the location to be stringified (non-null)
     * @since JWare/AntX 0.5
     **/
    public static String shortStringFrom(boolean prefer, Location l)
    {
        if (prefer) {
            return shortStringFrom(l);
        }
        if (Location.UNKNOWN_LOCATION.equals(l)) {
            return Strings.UNDEFINED;
        }
        return l.toString();
    }


    /**
     * Resolves any property references in given string iff any
     * exist. If no references (or string is <i>null</i>), returns
     * the same string.
     * @param P [optional] the project from which properties read
     * @param value the template value (non-null)
     * @throws NullPointerException if project is <i>null</i>
     * @since JWare/AntX 0.4
     **/
    public static String resolveString(Project P, String value)
    {
        if (value==null || P==null) {
            return value;
        }
        if (value.indexOf("${")>=0) {
            value = P.replaceProperties(value);//expensiv:mem+time
        }
        return value;
    }


    /**
     * Resolves any property or attribute references in given
     * string iff any exist. If no references (or string is
     * <i>null</i>), returns the same string. Basically tries to
     * avoid overhead of <span class="src">Project.replaceProperties</span>
     * in cases where there are a lot of properties to test.
     * @param P [optional] the project from which properties read
     * @param value the template value (non-null)
     * @param altForm <i>true</i> if we should look for "@{" marker
     *        as well as standard "${" marker (like in macros)
     * @throws NullPointerException if project is <i>null</i>
     * @since JWare/AntX 0.4
     **/
    public static String resolveString(Project P, String value,
                                       boolean altForm)
    {
        if (value==null || P==null) {
            return value;
        }
        //1. Check for the AntX 0.5 value-URI preferred form first.
        if (altForm) {
            //Like Ant's property reference this is not nestable!
            if (value.indexOf("@(")>=0) {
                StringBuffer sb = new StringBuffer(value.length());
                char[] cs = getChars(value);
                int i,j=0,n;
                do {
                    i= value.indexOf("@(",j);
                    if (i>=0) {
                        n= value.indexOf(')',i+2);
                        if (n<0) {
                            return value;//*=> is BROKEN; like Ant*
                        }
                        sb.append(cs,j,i-j);
                        sb.append("${");
                        i += 2;
                        sb.append(cs,i,n-i);
                        sb.append("}");
                        j= n+1;
                    } else {
                        sb.append(cs,j,cs.length-j);
                        break;
                    }
                } while(true);
                value = sb.toString();
            }
        }
        //2. Check for the AntX 0.4 form second.
        if (altForm && value.indexOf("@{",0)>=0) {//simple version
            StringBuffer sb = new StringBuffer(value.length());
            char[] cs = getChars(value);
            int i=0,j=0;
            do {
                i= value.indexOf("@{",j);
                if (i>=0) {
                    sb.append(cs,j,i-j);
                    sb.append("${");
                    j= i+2;
                } else {
                    sb.append(cs,j,cs.length-j);
                    break;
                }
            } while (true);
            value = P.replaceProperties(sb.toString());

        } else {
            if (value.indexOf("${",0)>=0) {
                value = P.replaceProperties(value);
            }
        }
        return value;
    }


    /**
     * Returns character buffer containing contents of a string.
     * @since JWare/AntX 0.4
     * @throws NullPointerException if string is <i>null</i>
     **/
    public static char[] getChars(String s)
    {
        char[] cs = new char[s.length()];
        s.getChars(0,s.length(),cs,0);
        return cs;
    }



    /**
     * Returns a readonly variant of the incoming map, returning
     * the same map if it is already unmodifiable. If the given
     * map is <i>null</i>, this method returns <i>null</i> unless
     * told not to explicitly. This method lessens object creation
     * when you need to generate lots of readonly views of maps
     * that might already be readonly.
     * @param in the map to examine (and possibly wrap)
     * @param noNull <i>true</i> if null inputs should be mapped
     *        to the standard empty readonly map.
     * @since JWare/AntX 0.4
     **/
    public static Map readonlyFrom(Map in, boolean noNull)
    {
        if (in==null) {
            return noNull ? Collections.EMPTY_MAP : null;
        }
        if (in==Collections.EMPTY_MAP) {
            return in;
        }
        if (in.getClass()==READONLYMAP.getClass()) {
            return in;
        }
        return Collections.unmodifiableMap(in);
    }



    /**
     * Tries to run a declared method on a given object. The specified
     * method should be unique (from combination of name and parameters).
     * Caller must have permission to set method accessibility. This method
     * assumes the void signature if either <i>signature</i> or
     * <i>argv</i> is <i>null</i>. This method does not allow you to run
     * private <em>inherited</em> methods; however, you can run non-public
     * methods on the object's immediate class.
     * @param object object (non-null)
     * @param methodname name of method (non-null)
     * @param signature [optional] method signature (ordered,non-null)
     * @param argv [optional] parameter list (ordered,non-null)
     * @throws Exception if unable to invoke specified method
     * @since JWare/AntX 0.4 (from JWare/core 0.7)
     **/
    public final static Object invokeit(Object object, String methodname,
                                        Class[] signature, Object[] argv)
        throws Exception
    {
        Class objectclass = object.getClass();
        Method method=null;
        try {
            method = objectclass.getMethod(methodname,signature);//NB:90+%
        } catch(NoSuchMethodException nsmx) {
            try {
                method = objectclass.getDeclaredMethod(methodname,signature);//NB:95+%
            } catch(NoSuchMethodException nsdmx) {
                Class parent = objectclass.getSuperclass();//NB:98+%
                while (parent!=null) {
                    try {
                        method = parent.getDeclaredMethod(methodname,signature);
                        if (Modifier.isPrivate(method.getModifiers())) {//?package too?
                            method=null;
                        }
                        break;
                    } catch(NoSuchMethodException igx) {/*burp*/}
                    parent = parent.getSuperclass();
                }
                if (method==null) {
                    throw nsmx;
                }
            }//2nd
        }//1st

        boolean is = method.isAccessible();
        if (!is) {
            method.setAccessible(true);
        }
        try {
            return method.invoke(object,argv);
        } finally {
            if (!is) {
                method.setAccessible(false);
            }
        }
    }



    /**
     * Utility to extract a named field's information from a class.
     * Caller must have required security permissions for this method to work.
     * The returned field may or may not be accessible by caller. Does not
     * work for interface-bound fields.
     * @param clazz class under test (non-null)
     * @param fieldname field interested in finding (non-null)
     * @return Field information
     * @throws NoSuchFieldException if field not found in class or superclasses.
     * @throws IllegalArgumentException if either clazz or fieldname invalid.
     * @since JWare/AntX 0.5
     **/
    public static Field findField(Class clazz, final String fieldname)
        throws NoSuchFieldException
    {
        if (clazz==null || isWhitespace(fieldname)) {
            throw new IllegalArgumentException("findField- invalid class or fieldname");
        }
        do {
            try {
                return clazz.getDeclaredField(fieldname);
            } catch(NoSuchFieldException nsfx) {/*burp*/}
            
            clazz = clazz.getSuperclass();
            if (clazz==null) {
                throw new NoSuchFieldException(fieldname);
            }
        } while(true);
    }



    /**
      * Checks if the two given strings are semantically equal, optionally
      * taking case into account. This method does call <span class="src">a</span>
      * object's <span class="src">equals</span> method. Two <i>null</i> objects
      * are considered equal.
      * @param a first string (primary)
      * @param b second string
      * @param caseins <i>true</i> if comparision should ignore case
      * @return <i>true</i> if equal strings
      **/
     public static boolean equalStrings(String a, String b, boolean caseins)
     {
         if (a==null) {
             return b==null;
         }
         if (b==null) {
             return false;
         }
         if (a==b) {
             return true;
         }
         return caseins ? a.equalsIgnoreCase(b) : a.equals(b);
     }


     /**
      * Same as {@linkplain #equalStrings(String, String, boolean) equalStrings}
      * but with default casesensitive string comparision.
      **/
     public static boolean equalStrings(String a, String b)
     {
         return equalStrings(a,b,false);
     }





    /**
     * Helper for obtaining a hash value of object which could be <i>null</i>.
     * Note that because the JDK has no proper implementation of
     * <span class="src">equals</span> or <span class="src">hashCode</span> for
     * Java arrays, we normalize array hashValues to allow the hashCode of
     * equal arrays (manually determined) to match (thereby not violating
     * the JLS).
     **/
    public static int hashOf(Object o)
    {
        if (o==null) {
            return 191;//?!better choice?!
        }
        if (o.getClass().isArray()) {
            return o.getClass().getComponentType().hashCode();
        }
        return o.hashCode();
    }



    /**
     * Looks for an application supplied implementation for a named SPI. This
     * method first checks the system properties for a property of same name
     * as SPI class. If not found, this method does the usua
     * <span class="src">META-INF/services/{spi-class-name}</span> lookup.
     * @param spiName class name of SPI to implement (non-null)
     * @param clnt problem handler (non-null)
     * @return fully qualified class name of matching implementation or <i>null</i>
     *    if no match found.
     * @since JWare/AntX 0.5
     **/
    public final static String loadSpiImpl(String spiName, Requester clnt)
    {
        String classname = System.getProperty(spiName);
        if (!Tk.isWhitespace(classname)) {
            return classname;
        }

        String serviceId = "META-INF/services/"+spiName;
        try {
            ClassLoader cL = LoaderUtils.getContextClassLoader();
            InputStream is = null;
            if (cL!=null) {
                is = cL.getResourceAsStream(serviceId);
            }
            if (is==null) {
                is = ClassLoader.getSystemResourceAsStream(serviceId);
            }
            // [From Ant] This code is needed by EBCDIC and other
            // unicode deficient system although service provider
            // guidelines seez should be UTF-16 encoding (ssmc)
            if (is!=null) {
                InputStreamReader isr;
                try {
                    isr = new InputStreamReader(is,"UTF-8");
                } catch (java.io.UnsupportedEncodingException e) {
                    isr = new InputStreamReader(is);
                }
                BufferedReader rd= new BufferedReader(isr);
                classname = rd.readLine();
                rd.close();

                if (!Tk.isWhitespace(classname)) {
                    return classname;
                }
            }
        } catch (Exception anyX) {
            clnt.problem(anyX.getLocalizedMessage(),Project.MSG_WARN);
        }

        return null;
    }



    /**
     * Like {@linkplain #loadSpiImpl(String,Requester)
     * loadSpiImpl(String,&#8230;)} using the given SPI abstract class's
     * name as lookup key.
     * @param spiClass SPI class to implement (non-null)
     * @param clnt problem handler (non-null)
     * @return fully qualified class name of matching implementation or <i>null</i>
     *    if no match found.
     * @since JWare/AntX 0.5
     **/
    public final static String loadSpiImpl(Class spiClass, Requester clnt)
    {
        return loadSpiImpl(spiClass.getName(),clnt);
    }



    /** Prevent This. **/
    private Tk()
    {
    }



    /** So we can determine what the JRE's standard
     *  "unmodifiableMap" class is.
     **/
    private static final Map READONLYMAP=
        Collections.unmodifiableMap(Collections.EMPTY_MAP);
}

/* end-of-Tk.java */
