/**
 * $Id: DurationFormat.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.io.PrintWriter;
import  java.text.FieldPosition;
import  java.text.Format;
import  java.text.ParsePosition;
import  java.util.Date;

/**
 * Helper class that formats a duration (milliseconds since January 1, 1970) as a
 * simple ASCII string: <code><i>X</i>hr <i>Y</i>min <i>Z</i>sec <i>00</i>ms</code>.
 * Does not support parsing of input. Safe for concurrent use. Only generates
 * output for non-zero fields.
 *
 * @since    JWare/AntX 0.4 (copied from JWare/core 0.6)
 * @author   ssmc, &copy;1997-2002 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  <span id="clsv">0.5</span>
 * @.safety  multiple
 * @.group   impl,helper
 **/

public class DurationFormat extends Format
{
    static final long MS_PER_HOUR = 60L  /*min-hr*/  * 60L  /*sec-min*/ * 1000L/*ms-sec*/;
    static final long MS_PER_MIN  = 60L  /*sec-min*/ * 1000L/*ms-sec*/;
    static final long MS_PER_SEC  = 1000L/*ms-sec*/;

    /**
     * Used to mean zero (0) duration.
     **/
    public static final String NO_DURATION = "--";



    /**
     * Creates a standard Java Format adapter for duration formatting.
     **/
    public DurationFormat()
    {
        super();
        SPC="";
        hash_= getClass().getName().hashCode()+SPC.hashCode();
    }



    /**
     * Creates a standard Java Format adapter for duration formatting.
     **/
    public DurationFormat(String spc)
    {
        super();
        if (spc==null) { throw new IllegalArgumentException("non-null spacer"); }
        SPC=spc;
        hash_= getClass().getName().hashCode()+SPC.hashCode();
    }



    /**
     * Formats a duration (milliseconds since January 1, 1970) as an ASCII
     * string: <code><i>X</i>hr <i>Y</i>min <i>Z</i>sec <i>00</i>ms</code>.
     * @param duration the duration to format
     * @param inc_ms true if milliseconds should be included
     * @param sb string buffer to be updated (non-null)
     * @return the updated string buffer (never null)
     **/
    public StringBuffer format(long duration, boolean inc_ms, StringBuffer sb)
    {
        if (duration>0) {
            int nf=0;
            long field = duration / MS_PER_HOUR;
            long ms_gone = field * MS_PER_HOUR;
            if (field>0) {
                sb.append(field);
                sb.append("hr");
                nf++;
            }
            field = (duration - ms_gone) / MS_PER_MIN;
            ms_gone += field * MS_PER_MIN;
            if (field>0) {
                if (nf>0) {sb.append(SPC);}
                sb.append(field);
                sb.append("min");
                nf++;
            }
            field = (duration - ms_gone) / MS_PER_SEC;
            ms_gone += field * MS_PER_SEC;
            if (field>0) {
                if (nf>0) {sb.append(SPC);}
                sb.append(field);
                sb.append("sec");
                nf++;
            }
            field = duration - ms_gone;
            if (inc_ms && field>0) {
                if (nf>0) {sb.append(SPC);}
                sb.append(field);
                sb.append("ms");
                nf++;
            }
        } else {
            sb.append(NO_DURATION);
        }
        return sb;
    }



    /**
     * Formats a duration (milliseconds since January 1, 1970) as an ASCII
     * string: <code><i>X</i>hr <i>Y</i>min <i>Z</i>sec <i>00</i>ms</code>.
     * @param duration the duration to format
     * @param inc_ms true if milliseconds should be included
     * @param w print writer used to output string (non-null)
     **/
    public void format(long duration, boolean inc_ms, PrintWriter w)
    {
        if (duration>0) {
            int nf=0;
            long field = duration / MS_PER_HOUR;
            long ms_gone = field * MS_PER_HOUR;
            if (field>0) {
                w.print(field);
                w.print("hr");
                nf++;
            }
            field = (duration - ms_gone) / MS_PER_MIN;
            ms_gone += field * MS_PER_MIN;
            if (field>0) {
                if (nf>0) {w.print(SPC);}
                w.print(field);
                w.print("min");
                nf++;
            }
            field = (duration - ms_gone) / MS_PER_SEC;
            ms_gone += field * MS_PER_SEC;
            if (field>0) {
                if (nf>0) {w.print(SPC);}
                w.print(field);
                w.print("sec");
                nf++;
            }
            field = duration - ms_gone;
            if (inc_ms && field>0) {
                if (nf>0) {w.print(SPC);}
                w.print(field);
                w.print("ms");
                nf++;
            }
        } else {
            w.print(NO_DURATION);
        }
    }



    /**
     * Convenience that allocates transient StringBuffer internally.
     **/
    public final String format(long duration, boolean inc_ms)
    {
        return format(duration,inc_ms,new StringBuffer(50)).toString();
    }



    /**
     * If given object is a Number (any), formats the long value of this
     * number.
     * @throws IllegalArgumentException if object not a number or date
     **/
    public StringBuffer format(Object obj, StringBuffer sb, FieldPosition fp)
    {
        long duration= -1L;
        if (obj instanceof Number) {
            duration = ((Number)obj).longValue();
        }
        else if (obj instanceof Date) {
            duration = ((Date)obj).getTime();
        }
        else {
            throw new IllegalArgumentException("Unable to convert non-Number");
        }
        return format(duration,true,new StringBuffer(50));
    }



    /**
     * Unsupported parse method; DurationFormat only works for output.
     * @throws UnsupportedOperationException always
     **/
    public Object parseObject(String str, ParsePosition pp)
    {
        throw new UnsupportedOperationException();
    }



    /**
     * Unsupported parse method; DurationFormat only works for output.
     * @throws UnsupportedOperationException always
     **/
    public Object parseObject(String str)
    {
        throw new UnsupportedOperationException();
    }



    /**
     * Determine if incoming object is equivalent DurationFormat.
     **/
    public boolean equals(Object o)
    {
        if (o==this) {
            return true;
        }
        if (o==null) {
            return false;
        }
        if (o.getClass()==getClass()) {
            DurationFormat df = (DurationFormat)o;
            return SPC.equals(df.SPC);
        }
        return false;
    }



    /**
     * Returns hash value for this DurationFormat.
     **/
    public int hashCode()
    {
        return hash_;
    }



    /**
     * VM-shareable singleton that separates fields with '&#46;' (period).
     **/
    public static final DurationFormat INSTANCE = new DurationFormat(".");



    /**
     * Standard API for retrieving VM-shareable singleton {@linkplain #INSTANCE}.
     **/
    public static final DurationFormat getInstance()
    {
        return INSTANCE;
    }



    private final String SPC;
    private final int hash_;
}

/* end-of-DurationFormat.java */
