/**
 * $Id: IffOs.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.go;

import  java.util.List;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Os;

import  com.idaremedia.antx.helpers.Tk;

/**
 * Implementation of the general <i>if-os</i> test for all conditional components.
 * The <span class="src">IffOs</span> criteria passes if the current Ant runtime's
 * operating system definition matches the given selection criteria. The selection
 * criteria is based on the standard Ant <span class="src">&lt;os&gt;</span> condition.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 * @.pattern GoF.Strategy
 * @see      Go
 * @see      IffAnt
 * @see      com.idaremedia.antx.parameters.PlatformConditional PlatformConditional
 **/

public final class IffOs
{
    /**
     * Returns <i>true</i> if the current Ant runtime's operating
     * system definition matches the selection criteria. The general form
     * of the input string is:
     * <span class="src">[family|*],[name|*],[arch|*],[version|*]</span>.
     * Allows selectors like: <pre>
     *    ifOs="windows,*,*,5.0.0.1"
     *    ifOs="unix,*,SunOS"
     * </pre>
     * @param string comma-delimited list of selection fields in order
     * @param P project from which properties read
     * @param ifNot <i>true</i> if test should be for no-match
     **/
    public static boolean pass(String string, Project P, boolean ifNot)
    {
        if (Tk.isWhitespace(string)) {
            return false;
        }

        string = Tk.resolveString(P,string);

        List l= Tk.splitList(string);
        String family = l.get(0).toString();

        String next;
        String name = null;
        if (l.size()>1) {
            next = l.get(1).toString();
            if (!ignore(next)) {
                name= next;
            }
        }
        String arch = null;
        if (l.size()>2) {
            next= l.get(2).toString();
            if (!ignore(next)) {
                arch = next;
            }
        }
        String vers = null;
        if (l.size()>3) {
            next = l.get(3).toString();
            if (!ignore(next)) {
                vers = next;
            }
        }
        boolean is= false;
        try {
            is=Os.isOs(family,name,arch,vers);
        } catch(Exception unknownOS) {
            /*burp => unknown => isNot */
        }
        return ifNot ? !is : is;
    }


    /**
     * Execute test for an "if-os" conditional parameter.
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  single
     * @.group   impl,helper
     **/
    public static final class Is extends Go.TestSkeleton {
        public Is() {
        }
        public Is(String choice) {
            super(choice);
        }
        public boolean pass(Project P) {
            verifyInited();
            return IffOs.pass(getParameter(),P,false);
        }
        public String getParameterName() {
            return "ifOS";
        }
    }



    /**
     * Execute test for an "unless-os" conditional parameter.
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  single
     * @.group   impl,helper
     **/
    public static final class IsNot extends Go.TestSkeleton {
        public IsNot() {
        }
        public IsNot(String choice) {
            super(choice);
        }
        public boolean pass(Project P) {
            verifyInited();
            return IffOs.pass(getParameter(),P,true);
        }
        public String getParameterName() {
            return "unlessOS";
        }
    }


    private static boolean ignore(String s)
    {
        return Tk.isWhitespace(s) || "*".equals(s.trim());
    }


    /**
     * Prevent; only helpers public.
     **/
    private IffOs() {
    }
}

/* end-of-IffOs.java */
