/**
 * $Id: MatchCondition.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.match;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.condition.solo.RulesTk;
import  com.idaremedia.antx.helpers.GenericParameters;
import  com.idaremedia.antx.ownhelpers.ScopedProperties;

/**
 * Switch choice that matches value-under-test and a context against a named
 * condition definition. If the condition evaluates true, the choice is matched.
 * If the enclosing switch task has a non-null switch value, the value is overlayed
 * under the property name "<span class="src">switch.value</span>" before the condition
 * is called. This value takes precedence over any property named in the evaluation
 * context.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,infra
 **/

public class MatchCondition extends ChoiceTask
{
    /**
     * Creates new nested MatchCondition instance.
     **/
    public MatchCondition()
    {
        super(AntX.flow+"switch");
    }



    /**
     * Initializes new condition choice task with 
     * custom  delay configuration.
     * @param iam CV-label
     * @param delayConfiguration <i>true</i> if delay 
     *          nested  task configuration
     **/
    protected MatchCondition(String iam, boolean delayConfiguration)
    {
        super(iam,delayConfiguration);
    }



    /**
     * Initializes the enclosing project of this component. 
     * Updates any internal project-components too.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_conditionId.setProject(P);
    }



    /**
     * Returns the raw condition specifier. This value can be 
     * <i>null</i>, a property's name, a variable's name, or a
     * reference.
     **/
    public String getFlexValue()
    {
        return m_conditionId.get();
    }



    /**
     * Returns <i>true</i> if no condition id has been set.
     **/
    public boolean isUndefined()
    {
        return m_conditionId.isUndefined();
    }



    /**
     * Returns the condition's id used by this choice's
     * comparision. Returns <i>null</i> if value never defined
     * or determined value doesn't exist; for example, if the
     * named reference doesn't exist in project.
     **/
    public String getVUT()
    {
        return m_conditionId.getValue(getProject());
    }



    /**
     * Initializes the condition id holder as a direct literal.
     * @param id the condition id as flex value (non-null)
     **/
    private void setFlexValue(String id)
    {
        m_conditionId.set(id);
        m_conditionId.setIsLiteral();
    }



    /**
     * Sets the name of the condition directly. The item must 
     * exist when this choice is evaluated.
     * @param condition the condition's name (non-null)
     **/
    public void setCriteria(Reference condition)
    {
        require_(condition!=null,"setTest- nonzro reference");
        setFlexValue(condition.getRefId());
    }



    /**
     * Synonym for {@linkplain #setCriteria setCriteria()} that
     * matches the standard AntX condition value URI protocol.
     * @param condition the condition's name (non-null)
     **/
    public final void setTest(Reference condition)
    {
        setCriteria(condition);
    }



    /**
     * Returns this choice's condition reference id if it is
     * known. Will return <i>null</i> if value is not defined as
     * a literal (known) value.
     **/
    public final String getCriteria()
    {
        return m_conditionId.isLiteral() ? m_conditionId.get() : null;
    }



    /**
     * Sets the property from which condition's reference is read.
     * @param property the property's name (non-null)
     **/
    public void setProperty(String property)
    {
        require_(property!=null,"setProperty- nonzro name");
        setFlexValue(property);
        m_conditionId.setIsProperty(true);
    }



    /**
     * Sets the variable from which condition's reference is read.
     * @param variable the variable's name (non-null)
     **/
    public void setVariable(String variable)
    {
        require_(variable!=null,"setVariable- nonzro name");
        setFlexValue(variable);
        m_conditionId.setIsExported(true);
    }



    /**
     * Sets the reference from which condition's reference is read.
     * @param reference the reference's name (non-null)
     **/
    public void setReference(String reference)
    {
        require_(reference!=null,"setReference- nonzro name");
        setFlexValue(reference);
        m_conditionId.setIsReference(true);
    }



    /**
     * Installs all of our context properties and switch value
     * as an overlay before calling condition.
     **/
    private void installOverlay(String vut, Reference context)
    {
        Project P= getProject();
        
        //NB: Ordering is important; let the switch.value prevail!
        if (context!=null) {
            Object o = P.getReference(context.getRefId());
            if (o instanceof GenericParameters) {
                m_overlay = new ScopedProperties(P,true);
                m_overlay.put((GenericParameters)o);
            }
        }
        if (vut!=null) {
            if (m_overlay==null) {
                m_overlay = new ScopedProperties(P,true);
            }
            m_overlay.put("switch.value",vut);
        }
        if (m_overlay!=null) {
            m_overlay.install();
        }
    }




    /**
     * Undoes the effect of {@linkplain #installOverlay}.
     **/
    private void uninstallOverlay()
    {
        if (m_overlay!=null) {
            m_overlay.uninstall(null);
            m_overlay=null;
        }
    }



    /**
     * Ensures we are defined and the referred-to object is a
     * condition-compatible object.
     * @throws BuildException if undefined.
     * @throws BuildException if reference is not conditon compatible.
     */
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);
        
        String testId = m_conditionId.getValue();
        RulesTk.verifyTest(testId,m_rqlink);
    }



    /**
     * Evaluates this choice's condition for a positive result. Returns
     * <i>true</i> if this choice's criteria are met by the incoming
     * value and evaluation context.
     * @param vut value under test (vut-vut)
     * @param context [optional] evaluation context
     * @throws BuildException if condition reference not named.
     **/
    public boolean eval(String vut, Reference context)
        throws BuildException
    {
        verifyCanExecute_("eval");
        installOverlay(vut, context);
        try {
            String testId = m_conditionId.getValue();
            return RulesTk.evalTest(testId, m_rqlink);
        } finally {
            uninstallOverlay();
        }
    }


    private FlexString m_conditionId = new FlexString();
    private ScopedProperties m_overlay;//only-for eval
    private Requester m_rqlink = new Requester.ForComponent(this);
}

/* end-of-MatchCondition.java */