/**
 * $Id: ConditionalTaskSet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.condition.solo.RulesTk;
import  com.idaremedia.antx.parameters.FlexConditional;

/**
 * Taskset that has builtin support for conditional execution. Implements basic support
 * for both the standard '<i>if</i>' and '<i>unless</i>' control options as well as
 * several other AntX-specific variants like '<i>ifAnt</i>'. See the
 * {@linkplain FlexConditional} interface for the complete list of supported
 * execution conditions.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;do test="some-condition-id"&gt;
 *        <i>[Your tasks here&#8230;]</i>
 *    &lt;/do&gt;
 *
 *    &lt;do ifAntLike=".*\1\.6.*"&gt;
 *        <i>[Your tasks here&#8230;]</i>
 *    &lt;/do&gt;
 *
 *    &lt;do ifAllTrue="a,b,c" unless="no-alpabet"&gt;
 *        <i>[Your tasks here&#8230;]</i>
 *    &lt;/do&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.5 (Pushed down from antx.starters.ConditionalTaskSet)
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 **/

public class ConditionalTaskSet extends com.idaremedia.antx.starters.ConditionalTaskSet
    implements FlexConditional
{
    /**
     * Initializes a new empty conditional taskset.
     **/
    public ConditionalTaskSet()
    {
        super();
    }


    /**
     * Initializes a new CV-labeled empty conditional taskset.
     * @param iam CV-label (non-null)
     **/
    protected ConditionalTaskSet(String iam)
    {
        super(iam);
    }



    /**
     * Initializes new configuration-delaying taskset.
     * @param iam CV-label (non-null)
     * @param delayConfigure <i>true</i> if nested tasks configured
     *        when they're actually executed
     **/
    protected ConditionalTaskSet(String iam, boolean delayConfigure)
    {
        super(iam,delayConfigure);
    }



    /**
     * Adds the if-condition-is-true to this taskset.
     * A property value is considered <i>true</i> if it
     * is one of: "true","on", or "yes".
     **/
    public void setIfTrue(String property)
    {
        m_guard.setIfTrue(property);
    }

    public void setTrue(String booleanString)
    {
        m_guard.setTrue(booleanString);
    }

    public void setFalse(String booleanString)
    {
        m_guard.setFalse(booleanString);
    }


    public void setIfAll(String properties)
    {
        m_guard.setIfAll(properties);
    }

    public void setIfAllTrue(String properties)
    {
        m_guard.setIfAllTrue(properties);
    }

    public void setIfOS(String choice)
    {
        m_guard.setIfOS(choice);
    }

    public void setIfAntLike(String version)
    {
        m_guard.setIfAntLike(version);
    }



    /**
     * Adds the unless-condition-is-true to this taskset.
     * A property value is considered <i>true</i> if it is
     * one of: "true", "on", or "yes".
     **/
    public final void setUnlessTrue(String property)
    {
        m_guard.setUnlessTrue(property);
    }


    public void setUnlessAll(String properties)
    {
        m_guard.setUnlessAll(properties);
    }

    public void setUnlessAllTrue(String properties)
    {
        m_guard.setUnlessAllTrue(properties);
    }

    public void setUnlessOS(String choice)
    {
        m_guard.setUnlessOS(choice);
    }

    public void setUnlessAntLike(String version)
    {
        m_guard.setUnlessAntLike(version);
    }


    /**
     * Assigns this taskset a named condition execution guard. If the
     * referenced condition evaluates <i>true</i> this taskset's
     * contents will be performed.
     * @param testRef reference to test definition (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setCriteria(Reference testRef)
    {
        require_(testRef!=null,"setTest: nonzro test refid");
        m_conditionId = testRef;
    }



    /**
     * Synonym for {@linkplain #setCriteria(Reference) setCriteria}.
     * @param testRef reference to test definition (non-null)
     * @since JWare/AntX 0.5
     **/
    public final void setTest(Reference testRef)
    {
        setCriteria(testRef);
    }



    /**
     * Returns <i>true</i> if either this taskset's named condition test
     * passes or all of its individual if tests pass.
     * @return <i>true</i> if all execution conditions met
     * @since JWare/AntX 0.5
     **/
    public boolean testIfCondition()
    {
        if (m_conditionId!=null) {
            String testId = m_conditionId.getRefId();
            boolean passIf = true;
            if (!RulesTk.evalTest(testId, m_rqlink)) {
                m_guard.setLastFailure("$test:"+testId);
                passIf = false;
            }
            log("Conditional execution (test="+testId+") is "+passIf, Project.MSG_DEBUG);
            return passIf;
        }
        return m_guard.testIfCondition();
    }



    /**
     * Ensures that for conditional execution either a single named
     * condition has been defined or a combination of if/unless options.
     * @throws BuildException if a test condition has been named well
     *   as other if/unless conditions.
     * @throws BuildExceptionif a referred-to test object is not a
     *   valid condition object.
     * @since JWare/AntX 0.5
     */
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (m_conditionId!=null) {
            if (!m_guard.isEmpty()) {
                String e = getAntXMsg("flow.only.test.param",m_conditionId.getRefId());
                log(e, Project.MSG_ERR);
                throw new BuildException(e, getLocation());
            }
            RulesTk.verifyTest(m_conditionId.getRefId(), m_rqlink);
        }
    }



    private Requester m_rqlink = new Requester.ForComponent(this);
    private Reference m_conditionId;
}

/* end-of-ConditionalTaskSet.java */
