/**
 * $Id: EventEmitConduit.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  java.util.StringTokenizer;

import  org.apache.tools.ant.BuildEvent;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.io.ErrOutHandle;
import  com.idaremedia.antx.starters.BuildListenerSkeleton;
import  com.idaremedia.apis.DiagnosticsEmitter;

/**
 * Captures logged Ant messages and redirects them to any listening Log4J-based
 * monitors.
 * <p>
 * Currently this class does some some work-arounds to cope with the situation
 * where a single conduit is installed across multiple calls to independent child
 * projects. This might occur, for example, if an &lt;emitlogs&gt; encloses one
 * or more calls to &lt;antcall&gt; or a similar task.
 * <p>
 * This class tries to organize Ant logged messages into a minimal hierarchy for
 * sanity's sake. More complex message-format based filtering can be done using
 * standard Log4J facilities or AntX grouping mappings. While a conduit can be
 * attached "as-is" to any Ant build-iteration using the standard Ant customization
 * build-listener facilities, it is most effective when associated with an AntX
 * build-iteration Iam task or a scoped taskset like {@linkplain EmitLogsTask}.
 * Note that this listener <em>cannot</em> be used as a command-line based Ant
 * listener (it requires project-based information that may not exist at Ant early
 * startup time).
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 * @see      EmitLogsTask
 **/

public class EventEmitConduit extends BuildListenerSkeleton
{
    /**
     * Initializes a new EventEmitConduit instance.
     **/
    public EventEmitConduit()
    {
    }


    /**
     * Sets the callback problem handler for this conduit.
     * This handler is used for errors this conduit isn't
     * designed to handle itself.
     * @param errH the problemm handler (use <i>null</i> to clear)
     **/
    public void setProblemHandler(ProblemHandler errH)
    {
        m_errHandler = errH;
    }


    /**
     * Returns this conduit's callback problem handler. If
     * an explicit handler has not been set for this conduit
     * this method returns the current thread context default
     * handler. Never returns <i>null</i>.
     **/
    public ProblemHandler getProblemHandler()
    {
        if (m_errHandler!=null) {
            return m_errHandler;
        }
        return Iteration.problemsHandler();
    }


    /**
     * Sets the default emit configuration for this conduit.
     * This configuration is used to obtain information like
     * grouping path separators, diagnostics emitters, etc.
     * @param ec the configuration (use <i>null</i> to clear)
     **/
    public void setConfiguration(EmitConfiguration ec)
    {
        m_emitConfiguration = ec;
    }


    /**
     * Returns this conduit's default emit configuration. If
     * never set explicitly this method will return the current
     * thread's default emit configuration. Never returns
     * <i>null</i>.
     **/
    public EmitConfiguration getConfiguration()
    {
        if (m_emitConfiguration!=null) {
            return m_emitConfiguration;
        }
        return EmitContext.getConfigurationNoNull();
    }


    /**
     * Sets the default grouping mapper for this conduit.
     * This mapper is used to transform each event into a
     * diagnostics grouping (which is then used to determine
     * the diagnostics emitter over which the Ant log message
     * is broadcast).
     * @param mapper the mapper (use <i>null</i> to clear)
     **/
    public void setGroupingMapper(GroupingMapper mapper)
    {
        if (mapper==null) {
            m_mapper = getDefaultGroupingMapper();
        } else {
            m_mapper = mapper;
        }
    }


    /**
     * Returns this conduit's grouping mapper. If never set
     * explicitly (or reset), this method will return a default
     * mapper (zone/project/target/task). Never returns
     * <i>null</i>.
     **/
    public final GroupingMapper getGroupingMapper()
    {
        return m_mapper;
    }


    /**
     * Transforms a symbolic field name to its matching filter
     * mask bit.
     **/
    private int includeFrom(String is)
    {
        if (is.startsWith("task"))   {
            return MainField.INCLUDE_TASK;
        }
        if (is.startsWith("target")) {
            return MainField.INCLUDE_TARGET;
        }
        if (is.startsWith("nested")) {
            return MainField.INCLUDE_NESTED;
        }
        if (is.startsWith("project")){
            return MainField.INCLUDE_PROJECT;
        }
        return 0;
    }


    /**
     * Sets a filter on the default groupings created by this conduit.
     * Without any filters, this conduit will organize messages like:
     * zone-name/project-name/target-name/task-name[/message]. Filters
     * can be used to change this default path organization by eliminating
     * one or more elements; for example: zone-name/project-name/target-name
     * (no task-name grouping will be included). This method is usually
     * called by the conduit's owning task's script-facing method.
     * @param filterstring symbolic filter string for MainField.INCLUDE_*
     **/
    public void setIncludes(String filterstring)
    {
        if (filterstring==null) {
            m_includesFilter = MainField.DEFAULT_FILTER;
            m_filtersByName = Strings.DEFAULT;
        }
        else {
            filterstring = Tk.lowercaseFrom(filterstring);
            int i;
            if (Strings.ALL.equals(filterstring)) {
                i= MainField.INCLUDE_ALL;
            } else if (Strings.DEFAULT.equals(filterstring)) {
                i= MainField.DEFAULT_FILTER;
            } else if (Strings.NONE.equals(filterstring)) {
                i= 0;
            } else {
                if (filterstring.startsWith(Strings.ALL)) {
                    i= MainField.INCLUDE_ALL;
                    String minuslist = filterstring.substring(Strings.ALL.length());
                    StringTokenizer st= new StringTokenizer(minuslist,"-");
                    while (st.hasMoreTokens()) {
                        i &= ~includeFrom(st.nextToken());
                    }
                } else {
                    i= 0;
                    StringTokenizer st= new StringTokenizer(filterstring,",");
                    while (st.hasMoreTokens()) {
                        i |= includeFrom(st.nextToken());
                    }
                }
            }
            m_includesFilter = i;
            m_filtersByName = filterstring;
        }
    }


    /**
     * Returns this conduit's groupings filter string. Never returns
     * <i>null</i>; returns "default" if never set.
     **/
    public final String getIncludes()
    {
        return m_filtersByName;
    }


    /**
     * Like {@linkplain #setIncludes setIncludes(String)} but
     * with direct filter and name information.
     * @param mask filter mask (combination of MainField.INCLUDE_*)
     * @param byName the string-version of mask (non-null)
     * @see #setIncludes
     **/
    public final void setIncludesFilter(int mask, String byName)
    {
        if (byName==null) {
            throw new IllegalArgumentException();
        }
        m_includesFilter = mask;
        m_filtersByName = byName;
    }


    /**
     * Returns this conduit's grouping filter as a mask.
     **/
    public final int getIncludesFilter()
    {
        return m_includesFilter;
    }



    /**
     * The default mapper implements the fallback (builtin)
     * grouping hierarchy (no user customization).
     **/
    private static final GroupingMapper sm_DefaultMapper =
        new GroupingMapper();


    /**
     * Returns the default conduit mapper. Never returns
     * <i>null</i>.
     **/
    public static final GroupingMapper getDefaultGroupingMapper()
    {
        return sm_DefaultMapper;
    }


// ---------------------------------------------------------------------------------------
// Build Listener Impl:
// ---------------------------------------------------------------------------------------

    /**
     * Converts the message as logged to the Ant log infrastructure
     * to a log4j event broadcast. Message emitted to any listening
     * log4j monitors.
     * @param e Ant log message event (non-null)
     * @.impl This method must ensure nested log-calls don't trigger an
     *        infinite loopback condition
     **/
    public void messageLogged(BuildEvent e)
    {
        NoiseLevel nl = NoiseLevel.from(e);
        if (m_suspendedLevel==0) {//NB:don't loop back onto self!
            m_suspendedLevel++;

            ErrOutHandle devnull= new ErrOutHandle();
            devnull.install();

            try {
                if (nl!=null) {
                    IndicatorZone iz = IndicatorZone.from(nl);
                    GroupingMapper mapper = getGroupingMapper();
                    EmitConfiguration ini = getConfiguration();

                    String evtpath = mapper.pathFrom
                        (e, iz, getIncludesFilter(), ini.getGroupingPathSeparator());

                    String msgpath = mapper.getLabel(MainField.MESSAGE,e,iz);
                    if (msgpath!=null) {
                        if (msgpath.startsWith(ini.getGroupingPathSeparator())) {
                            evtpath += msgpath;
                        } else {
                            evtpath += ini.getGroupingPathSeparator() + msgpath;
                        }
                    }
                    DiagnosticsEmitter route = ini.getCustomEmitter(evtpath);
                    Emit.broadcast(route, e.getMessage(), e.getException(), nl);

                } else {
                    handleUnknownNoiseLevel(e);
                }
            } finally {
                devnull.uninstall(false,true);
                m_suspendedLevel--;
            }
        }
    }


    /**
     * Called if this conduit cannot match the event's priority
     * to noise level.
     * @param e listener event (non-null)
     **/
    protected void handleUnknownNoiseLevel(BuildEvent e)
    {
        getProblemHandler().problem
            (AntX.uistrs().get("emit.unknown.noiselevel",String.valueOf(e.getPriority())),
             Project.MSG_WARN);//NB:careful of loopback here (ssmc)
    }


    private volatile int m_suspendedLevel;
    private ProblemHandler m_errHandler;
    private EmitConfiguration m_emitConfiguration;
    private int m_includesFilter= MainField.DEFAULT_FILTER;
    private String m_filtersByName= Strings.DEFAULT;
    private GroupingMapper m_mapper = getDefaultGroupingMapper();
}

/* end-of-EventEmitConduit.java */
