/**
 * $Id: TallyTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.parameters.TrueFalsePropertySetter;

/**
 * Supplement for standard ConditionTask that allows the AntX extra conditions to also be
 * nested. Doesn't really <em>do</em> anything explicitly. Effective action is the
 * side-effects of nested conditions (like property-setting, etc.). Usually defined as
 * &lt;tally&gt; or &lt;check&gt;.
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;<b>tally</b>&gt;
 *      &lt;isallset trueproperty="all.required.present"&gt;
 *         &lt;property  value="junit.present"/&gt;
 *         &lt;property  value="log4j.present"/&gt;
 *         &lt;reference value="tests.classpath"/&gt;
 *      &lt;/isallset&gt;
 *   &lt;/tally&gt;
 * OR
 *   &lt;<b>tally</b>&gt;
 *      &lt;tally ruleid="preferred.thirdparty.libs"/&gt;
 *      &lt;isnoneset trueproperty="no.test.engines"&gt;
 *         &lt;property value="junit.present"/&gt;
 *         &lt;property value="jware.pet.present"/&gt;
 *      &lt;/isnoneset&gt;
 *   &lt;/tally&gt;
 *
 * <i><b>For Ant 1.6 or later:</b></i>
 *   &lt;typedef name="moonisright" classname="..."/&gt; <i>[your condition impl]</i>
 *   &#46;&#46;&#46;
 *   &lt;<b>tally</b>&gt;
 *      &lt;<b>moonisright</b> date=${DSTAMP}.../&gt;
 *   &lt;/tally&gt;
 * </pre>
 *
 * <b>Implementation Notes</b>:<ul>
 * <li>The tally sets write-once properties (its 'true' and 'false' properties); however,
 *     if a tally executes a rule that triggers a false-preference it will update the current
 *     setting of the preference's update property to whatever the default value is. The
 *     scope of this update is the current project <em>and any child projects created
 *     when using 'ant,' 'antcall', 'do', 'callforeach' and 'foreach.'</em>
 * </ul>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 * @.expects Ant 1.6 or later for automatic freeform
 * @see      AssertTask
 * @see      PreferTask
 **/

public class TallyTask extends CompoundRuleTask
    implements FreeformRule, TrueFalsePropertySetter
{
    /**
     * Initialize a new TallyTask.
     **/
    public TallyTask()
    {
        super(AntX.rules+"tally");
    }


    /**
     * Initializes a new CV-labeled TallyTask.
     **/
    public TallyTask(String iam)
    {
        super(iam);
    }


    /**
     * Initializes a new optionally embedded CV-labeled TallyTask
     * subclass.
     **/
    protected TallyTask(String iam, boolean embedded)
    {
        super(iam,embedded);
    }


// ---------------------------------------------------------------------------------------
// Parameters, Nested Elements:
// ---------------------------------------------------------------------------------------

    /**
     * Sets the default condition combining logic operator for
     * this tally. Defaults to 'anding' multiple conditions. Example
     * usage: <pre>
     *   &lt;tally logic="or" trueproperty="tests.required"&gt;
     *       &lt;isset property="is.debug.build"/&gt;
     *       &lt;equals arg1="debug" arg2="${build.type}"/&gt;
     *  &lt;/tally&gt;
     * </pre>
     * @since JWare/AntX 0.3
     **/
    public final void setLogic(BooleanRule.Operator logic)
    {
        setDefaultCombineOperator(logic);
    }


    /**
     * Returns the combine logic of this task. Defaults to <i>AND</i>
     * if never set.
     * @since JWare/AntX 0.3
     **/
    public final BooleanRule.Operator getLogic()
    {
        return getDefaultCombineOperator();
    }


    /**
     * Sets whether or not this tally will evaluate all nested
     * conditions before returning.
     * @param evaluateAll <i>true</i> to force tally to evaluate
     *        all conditions
     * @since JWare/AntX 0.3
     **/
    public void setAll(boolean evaluateAll)
    {
        setStopQuickEvaluation(!evaluateAll);
    }


    /**
     * Returns <i>true</i> if this tally will evaluate all
     * nested conditions even if final result can be determined
     * before.
     * @since JWare/AntX 0.3
     **/
    public final boolean willEvaluateAll()
    {
        return !isStopQuickEvaluation();
    }


    /**
     * Sets the property to be created on a negative evaluation.
     * Property will be set to the string "<i>true</i>."
     * @param property the property to create (non-null)
     **/
    public void setFalseProperty(String property)
    {
        require_(property!=null,"setFalsP- nonzro nam");
        checkModify("setFalsProp");
        m_falseProperty = property;
        m_varMask &= ~F_VAR;
    }


    /**
     * Returns the property to be created/set on a negative
     * evaluation. Returns <i>null</i> if never set.
     **/
    public final String getFalseProperty()
    {
        return (m_varMask&F_VAR)==F_VAR ? null : m_falseProperty;
    }


    /**
     * Sets the variable to be created on a negative evaluation.
     * Variable will be set to the string "<i>true</i>."
     * @param variable the variable to create (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setFalseVariable(String variable)
    {
        require_(variable!=null,"setFalsV- nonzro nam");
        checkModify("setFalsVar");
        m_falseProperty = variable;
        m_varMask |= F_VAR;
    }


    /**
     * Returns the variable to be created/set on a negative
     * evaluation. Returns <i>null</i> if never set.
     * @since JWare/AntX 0.5
     **/
    public final String getFalseVariable()
    {
        return (m_varMask&F_VAR)==F_VAR ? m_falseProperty : null;
    }


    /**
     * Sets the property to be created on a positive evaluation.
     * Property will be set to the string "<i>true</i>."
     * @param property the property to create (non-null)
     **/
    public void setTrueProperty(String property)
    {
        require_(property!=null,"setTrueP- nonzro nam");
        checkModify("setTrueProp");
        m_trueProperty = property;
    }


    /**
     * Returns the property to be created/set on a positive
     * evaluation. Returns <i>null</i> if never set.
     **/
    public final String getTrueProperty()
    {
        return m_trueProperty;
    }


    /**
     * Sets the variable to be created on a positive evaluation.
     * Variable will be set to the string "<i>true</i>."
     * @param variable the variable to create (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setTrueVariable(String variable)
    {
        require_(variable!=null,"setTrueV- nonzro nam");
        checkModify("setTrueVar");
        m_trueProperty = variable;
        m_varMask |= T_VAR;
    }


    /**
     * Returns the variable to be created/set on a positive
     * evaluation. Returns <i>null</i> if never set.
     * @since JWare/AntX 0.5
     **/
    public final String getTrueVariable()
    {
        return (m_varMask&T_VAR)==T_VAR ? m_trueProperty : null;
    }


    /**
     * Sets this tally task to use an existing build rule as its
     * definition. The reference will be evaluated (and verified)
     * the first time this task is evaluated.
     * @param ruleId build rule reference id (non-null)
     * @throws BuildException if referral's effect is incompatible
     * @see #isIncompatibleReferral
     **/
    public void setRuleId(Reference ruleId)
    {
        setReferralReference(ruleId);
    }


    /**
     * Capture our identifier for feedback if ever used to refer
     * to self in circular manner.
     **/
    public void setId(String id)
    {
        m_Id= id;
    }


    /**
     * Tries to return an identifier for this rule.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        return super.getId();
    }


    /**
     * Adds a sub-tally to this tally task.
     **/
    public void addTally(TallyTask tt)
    {
        xaddCondition(tt);
    }


    /**
     * Adds a preference condition to this tally task.
     **/
    public final void addPrefer(PreferTask pt)
    {
        xaddCondition(pt);
    }


    /**
     * Adds an assertion to this tally task; diagnostic aid.
     **/
    public final void addAssert(AssertTask at)
    {
        xaddCondition(at);
    }

// ---------------------------------------------------------------------------------------
// Free-form (Application) Nested Elements:
// ---------------------------------------------------------------------------------------

    /**
     * Adds an arbitrary application-defined condition to
     * this tally.
     * @param c custom condition definition (non-null)
     * @since JWare/AntX 0.4
     **/
    public void addConfigured(Condition c)
    {
        require_(c!=null,"add- nonzro condition");
        xaddCondition(c);
    }

// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Returns evaluation result for nested conditions.
     **/
    public boolean eval()
    {
        boolean istrue;

        if (runReferral()) {
            istrue= getReferral().eval(new Referee());
        } else {
            verifyCanExecute_("eval");
            istrue= defaultMultipleConditionEval();
        }

        // Update regular write-once properties instead of the inherited
        // preference-like velcro effect.

        if (istrue && getTrueProperty()!=null) {
            String prop = getTrueProperty();
            log("Tally was true; setting true-property '"+prop+"' property",
                Project.MSG_DEBUG);
            if ((m_varMask&T_VAR)==T_VAR) {
                ExportedProperties.set(prop,Strings.TRUE);
            } else {
                checkIfProperty_(prop,true);
                getProject().setNewProperty(prop,Strings.TRUE);
            }
        }
        if (!istrue && getFalseProperty()!=null) {
            String prop = getFalseProperty();
            log("Tally was false; setting false-property '"+prop+"' property",
                Project.MSG_DEBUG);
            if ((m_varMask&F_VAR)==F_VAR) {
                ExportedProperties.set(prop,Strings.TRUE);
            } else {
                checkIfProperty_(prop,true);
                getProject().setNewProperty(prop,Strings.TRUE);
            }
        }

        return istrue;
    }


    /**
     * Returns <i>false</i> since tallys don't care much what kind-of
     * conditions are nested in 'em.
     **/
    protected boolean isIncompatibleReferral(ShareableCondition rule)
    {
        return false;
    }


    private static final int T_VAR=0x02;
    private static final int F_VAR=0x04;

    private String m_Id;
    private String m_falseProperty;
    private String m_trueProperty;
    private int m_varMask;
}

/* end-of-TallyTask.java */
