/**
 * $Id: RuleType.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-200 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.util.Iterator;
import  java.util.Stack;
import  java.util.Hashtable;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.NoiseLevel;
import com.idaremedia.antx.apis.BuildAssertionException;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.starters.Quiet;


/**
 * Base AntX shareable rule type. Rule types are always embedded and accessed from their
 * thread-safe evaluation APIs. This class implements support for the following:<ul>
 *    <li>Multi-thread support for evaluation and property updates.</li>
 *    <li>Enforcement of access through shareable APIs (not execute)</li>
 *    <li>Cloning (sort-of) for copying to child projects</li>
 * </ul>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 * @see      RuleTask
 **/

public abstract class RuleType extends BooleanRule
    implements FixtureComponent, ShareableCondition, Quiet, Cloneable
{

    /**
     * Initializes a new embedded RuleType instance.
     * @param iam CV-label (non-null)
     **/
    protected RuleType(String iam)
    {
        super(iam,true);
    }


    /**
     * Returns a clone of this rule type. Supports copying of
     * rules types between parent and child projects.
     **/
    public Object clone()
    {
        try {
            RuleType copy = (RuleType)super.clone();
            cloneInternals(copy);
            return copy;
        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }

// ---------------------------------------------------------------------------------------
// Parameters, Nested Elements:
// ---------------------------------------------------------------------------------------

    /**
     * Capture our identifier for feedback since Ant types aren't
     * updated with correct location information.
     **/
    public void setId(String id)
    {
        m_Id= id;
    }


    /**
     * Tries to return an unique identifier for this type. Looks
     * for a build-script identifier; if none found creates a
     * unique string based on type's class name and instance
     * reference.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        return super.getId();
    }


    /**
     * Returns the supposed effect if this rule evaluates <i>false</i>.
     * Should be called after this rule is fully configured. Returns
     * <i>null</i> by default (meaning unknown).
     **/
    public NoiseLevel getFailureEffect()
    {
        return null;
    }


// ---------------------------------------------------------------------------------------
// Multi-thread Support:
// ---------------------------------------------------------------------------------------

    /**
     * Our factories of thread-specific update-information handles. Each
     * factory is associated with a class <em>instance</em>.
     **/
    private static final Hashtable sm_updateJunkFactories= new Hashtable(11,0.9f);

    private static final RuleUserHandle.Factory getUpdateJunkFactory(Class c)
    {
        if (c==null) {
            throw new IllegalArgumentException("getUPJFactory- NULL class");
        }
        synchronized(c) {//!eyow!
            RuleUserHandle.Factory f= (RuleUserHandle.Factory)
                sm_updateJunkFactories.get(c);
            if (f==null) {
                f = new RuleUserHandle.Factory();
                sm_updateJunkFactories.put(c,f);
            }
            return f;
        }
    }


    /**
     * The getter method for class-specific, thread-specific
     * update-information handle used by a rule's eval() methods.
     * Currently only accessible to our types.
     * @.safety guarded (class)
     **/
    private static final RuleUserHandle getUpdatePropertyJunkHandle(Class c)
    {
        return (RuleUserHandle)getUpdateJunkFactory(c).get();
    }


    /**
     * The getter method for thread-specific update-information used
     * by a build rule's eval method. Should never return <i>null</i>.
     * @.safety guarded (class)
     **/
    protected static final ShareableConditionUser getUpdatePropertyJunk(Class c)
    {
        return ((RuleUserHandle)getUpdateJunkFactory(c).get()).getCaller();
    }


    /**
     * Returns the current thread's update msgId. Returns this rule's
     * msgid if update msgId not set by caller.
     **/
    public final String getMsgId()
    {
        ShareableConditionUser upj= getUpdatePropertyJunk(getClass());
        String msgid = upj!=null ? upj.getMsgId() : null;
        return (msgid!=null) ? msgid : super.getMsgId();
    }


    /**
     * Returns the current thread's update Location. Returns this
     * rule's location if not set by caller.
     **/
    public final Location getLocation()
    {
        ShareableConditionUser upj= getUpdatePropertyJunk(getClass());
        Location l = upj!=null ? upj.getLocation() : null;
        return (l!=null) ? l : super.getLocation();
    }


    /**
     * Returns the current thread's update property. Returns
     * <i>null</i> if update property not set by caller.
     **/
    protected final String getUpdateProperty()
    {
        ShareableConditionUser upj= getUpdatePropertyJunk(getClass());
        return upj!=null ? upj.getUpdateProperty() : super.getUpdateProperty();
    }


    /**
     * Returns the current thread's update value. Returns
     * "<i>true</i>" if update value not set by caller.
     **/
    protected final String getUpdatePropertyValue()
    {
        ShareableConditionUser upj= getUpdatePropertyJunk(getClass());
        String upv = upj!=null ? upj.getUpdateValue() : null;
        return (upv==null) ? super.getUpdatePropertyValue() : upv;
    }


    /**
     * Returns the current thread's update variable. Returns
     * <i>null</i> if update variable not set by caller.
     **/
    protected final String getUpdateVariable()
    {
        ShareableConditionUser upj= getUpdatePropertyJunk(getClass());
        return upj!=null ? upj.getUpdateVariable() : super.getUpdateVariable();
    }


// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Freezes this rule definition from further modification. First ensures
     * this rule was fully configured. Cannot be called from a constructor.
     * @.safety guarded
     **/
    public final synchronized void freezeDefinition()
    {
        if (!isFrozen()) {
            maybeConfigure();
            super.freezeDefinition();

            if (!m_circularityChecked) {
                Stack stk = new Stack();
                stk.push(this);
                verifyNoCircularDependency(stk, new Requester.ForComponent(this));
                m_circularityChecked=true;
            }
        }
    }


    /**
     * The evaluation bits that each type might need to customize. Seems
     * to work for both tallysets and buildrules. The thread-based fixture
     * has been configured by caller.
     **/
    protected boolean safeEvalImpl()
    {
        boolean stopQuik= isStopQuickEvaluation();
        boolean istrue= true;
        int N= 0;
        boolean stop= false;//NB:used to ensure N is incremented(ssmc)

        if (!isEmpty()) {
            Iterator itr= getConditions().iterator();
            while (!stop && itr.hasNext()) {
                try {
                    boolean condTrue = ((Condition)itr.next()).eval();
                    if (!condTrue) {
                        istrue = false;
                        if (stopQuik) {
                            stop = true;
                        }
                    }
                } catch(BuildAssertionException aX) {
                    istrue = false;
                    stop = true;
                }
                N++;
            }
        }

        setEvalResult(istrue,getConditionNames(N));
        return istrue;
    }


    /**
     * Evaluates this rule type. Warning: the first time this rule is
     * evaluated it locks itself from further modification.
     * @param calr rule user source (non-null)
     * @.safety threadlocal after guarded freeze-check
     * @throws BuildException if error occurs during evaluation
     **/
    public boolean eval(ShareableConditionUser calr)
        throws BuildException
    {
        require_(calr!=null, "eval- nonzro calr information");

        // Ensure we're configured and then frozen first time.
        freezeDefinition();

        // Enforce what gets updated on a failed evaluation.
        RuleUserHandle upj = getUpdatePropertyJunkHandle(getClass());
        final ShareableConditionUser previous = upj.setCaller(calr);

        // Evaluate the conditions until a failure or all done.
        try {
            return safeEvalImpl();
        }
        // HouseKeep the thread-local holder back to entry-state.
        finally {
            upj.setCaller(previous);
        }
    }


    /**
     * By default assume rule types can be nested within each other
     * so its possible a rule will be called by another. Subclasses
     * should no-op this eval() if this is not the case. Warning:
     * the first time this rule is evaluated it locks itself from
     * further modification.
     * @.safety multiple after guarded freeze-check
     * @.sideeffect Will freeze this rule from further modification.
     **/
    public boolean eval()
    {
        // Ensure we're configured and then frozen first time.
        freezeDefinition();

        //Evaluate the conditions until a failure or all done.
        return safeEvalImpl();
    }


// ---------------------------------------------------------------------------------------

    /**
     * No-op; rules types must be explicitly evaluated by other
     * tasks not executed as a task by the parsing process.
     **/
    public final void execute()
    {
        //burp...quietly
    }


// ---------------------------------------------------------------------------------------
    protected boolean m_circularityChecked;//NB:nope
    private String m_Id;//NB:needed for feedback
}

/* end-of-RuleType.java */
