/**
 * $Id: ExecuteCriteria.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.apis.Responses;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.ProjectPropertiesNet;
import  com.idaremedia.antx.starters.Quiet;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * A reusable condition that needs more than a sequence of simplier conditions
 * to determine whether its been met. Criteria can have two kinds of side-effects:
 * they can initialize or modify fixture variables (required to set the outgoing
 * result variable), and they can create or alter (indirectly) project references.
 * <p/>
 * A criteria is meant to be reusable and can be executed many times within a single
 * execution iteration. Therefore, the items you nest within a criteria <em>cannot</em>
 * rely on changing properties outside the scope of the criteria. All property
 * additions and/or modifications are limited to the criteria's scope (and are not
 * seen by the project once the criteria has exited). As you would expect, pre-existing
 * and overlaid properties can be read from within the criteria.
 * <p/>
 * For a criteria to ever pass, you <em>must</em> set the result variable (as 
 * defined by the <span class="src">resultvariable</span> parameter) before leaving
 * the criteria's body.
 * <p/>
 * Criteria are <em>not</em> tallies or build rules. Their purpose is to be a
 * general and reusable boolean function and their side-effects are unknown. This 
 * (lack of) definition makes them different from the other rule types and puts them
 * into their own category. However, their intended use as execution tests for the 
 * various AntX flow control task(set)s makes them part of the AntX build rules
 * framework.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;<b>criteria</b> id="alms-uptodate" resultvar="out:alms-uptodate"&gt;
 *        &lt;assign var="out:alms-uptodate" value="yes"/&gt;
 *        &lt;property name="_module.meta" value="${alms.module.dir}/.crdate"/&gt;
 *        &lt;domatch value="${$filenotempty:@(_module.meta)}"&gt;
 *            &lt;equals value="true"&gt;
 *               &lt;loadfile srcFile="${repository.arms}/${alms.module}/.crdate" 
 *                           property="_repo.crdate"/&gt;
 *               &lt;loadfile srcFile="${_module.meta}" 
 *                           property="_alms.crdate"/&gt;
 *               &lt;do true="${$newerdate:@(_repo.crdate),,@(_alms.crdate)}"&gt;
 *                  &lt;assign var="out:alms-uptodate" value="no"/&gt;
 *               &lt;/do&gt;
 *            &lt;/equals&gt;
 *            &lt;otherwise&gt;
 *               &lt;assign var="out:alms-uptodate" value="no"/&gt;
 *            &lt;/otherwise&gt;
 *        &lt;/domatch&gt;
 *    &lt;/criteria&gt;
 *
 *    &lt;<b>criteria</b> id="docs-uptodate" resultvar="out:docs-uptodate"&gt;
 *       &lt;require allset="src,html" msg="src and html directory defined"/&gt;
 *       &lt;require isnotset="docs-uptodate" msg="locals not in use"/&gt;
 *       &lt;tally trueproperty="docs-uptodate"&gt;
 *          &lt;issettrue property="disable.apidocs"/&gt;
 *          &lt;isnotset property="clean"/&gt;
 *          &lt;uptodate targetfile="${html}/index.html"&gt;
 *             &lt;srcfiles dir="${srcs}"&gt;
 *                &lt;include name="**&#47;doc-files&#47;"/&gt;
 *                &lt;include name="**&#47;*.java"/&gt;
 *                &lt;include name="**&#47;package.html"/&gt;
 *                &lt;include name="**&#47;overview.html"/&gt;
 *                &lt;exclude name="**&#47;tests*&#47;**" unless="allapis"/&gt;
 *                &lt;exclude name="**&#47;.*&#47;**"/&gt;
 *             &lt;/srcfiles&gt;
 *          &lt;/uptodate&gt;
 *       &lt;/tally&gt;
 *       &lt;assign var="out:docs-uptodate" fromproperty="docs-uptodate"/&gt;
 *    &lt;/criteria&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,infra
 **/

public final class ExecuteCriteria extends TaskSet
    implements ShareableCondition, Quiet, AntLibFriendly, FixtureComponent
{
    /**
     * Initializes a new criteria instance.
     **/
    public ExecuteCriteria()
    {
        super(AntX.rules+"ExecuteCriteria:");
    }


    /**
     * Returns {@linkplain NoiseLevel#INFO INFO} always.
     **/
    public NoiseLevel getFailureEffect()
    {
        return NoiseLevel.INFO;
    }


//  ---------------------------------------------------------------------------------------
//  Parameters:
//  ---------------------------------------------------------------------------------------

    /**
     * Captures this criteria's reference identifier. Will be used
     * in feedback and other internal needs.
     * @param id the reference handle (non-null)
     **/
    public void setId(String id)
    {
        require_(id!=null,"setId- nonzro id");
        m_id = id;
    }



    /**
     * Returns this criteria's reference id. Will return a default
     * name "<span class="src">criteria</span>" if not set explicitly.
     * Never returns <i>null</i>.
     **/
    public String getId()
    {
        return m_id;
    }



    /**
     * Tells this criteria the result variable it should check when
     * it is evaluated. The variable is checked <em>after</em> all
     * nested tasks have been performed. 
     * @param outVar the variable name (non-null)
     **/
    public void setResultVariable(String outVar)
    {
        require_(!Tk.isWhitespace(outVar),"setRsltVar- nonzro name");
        m_outParam = outVar;
    }



    /**
     * Synonym for {@linkplain #setResultVariable setResultVariable(&#8230;)}.
     * @param outVar the variable name (non-null)
     **/
    public final void setResultVar(String outVar)
    {
        setResultVariable(outVar);
    }



    /**
     * Returns this criteria's result variable. Will return 
     * <i>null</i> if never set explicitly.
     * @see #setResultVariable setResultVariable(&#8230;)
     **/
    public final String getResultVariable()
    {
        return m_outParam;
    }


//  ---------------------------------------------------------------------------------------
//  Evaluation:
//  ---------------------------------------------------------------------------------------

    /**
     * Ensure we have been assigned a (default) result variable. This is
     * required so we can determine the true/false result of the criteria
     * we represent.
     * @throws BuildException if have not been assigned a result variable. 
     */
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (getResultVariable()==null) {
            String e = getAntXMsg("type.needs.this.attr",getTaskName(),"resultvariable");
            log(e, Project.MSG_ERR);
            throw new BuildException(e, getLocation());
        }
    }



    /**
     * Calls this criterial nested tasks then checks the {@linkplain
     * #setResultVar result variable} for a positive boolean string.
     * Caller can override the prescribed result variable's name by
     * passing in an update variable.
     * @param calr caller (non-null)
     * @throws BuildException if any nested task does.
     **/
    public boolean eval(ShareableConditionUser calr)
    {
        verifyCanExecute_("eval");

        String resultVar = getResultVariable();
        if (calr.getUpdateVariable()!=null) {
            resultVar = calr.getUpdateVariable();
        }
        ExportedProperties.unset(resultVar);

        performNestedTasks();

        String evalResult = ExportedProperties.readstring(resultVar);
        return Tk.string2PosBool(evalResult)==Boolean.TRUE;
    }



    /**
     * Runs a new copy of nested tasks in order. The <span class="src">perform</span>
     * message is sent to each nested task. Subclasses can override this method to
     * decorate the central <span class="src">performIterationOfTasksList</span>
     * method which does the heavy lifting.
     * @throws BuildException if any nested task does.
     **/
    protected void performNestedTasks()
    {
        ProjectPropertiesNet bubble = new ProjectPropertiesNet(getId(),getProject());
        try {
            performIterationOfTheTasksList();
        } finally {
            bubble.uninstall(new Responses.LogUsing(this));
        }
    }



    /**
     * No-op; criteria must be explicitly evaluated by other tasks.
     **/
    public final void execute()
    {
        //burp...quietly
    }


    private String m_outParam;
    private String m_id = "criteria";
}

/* end-of-ExecuteCriteria.java */