/**
 * $Id: CompoundRuleTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  org.apache.tools.ant.taskdefs.Available;
import  org.apache.tools.ant.taskdefs.Checksum;
import  org.apache.tools.ant.taskdefs.UpToDate;
import  org.apache.tools.ant.taskdefs.condition.*;

import  com.idaremedia.antx.condition.AllSet;
import  com.idaremedia.antx.condition.AllSetTrue;
import  com.idaremedia.antx.condition.AnySet;
import  com.idaremedia.antx.condition.AnySetTrue;
import  com.idaremedia.antx.condition.FileNotEmpty;
import  com.idaremedia.antx.condition.IsAntVersion;
import  com.idaremedia.antx.condition.IsBoolean;
import  com.idaremedia.antx.condition.IsClass;
import  com.idaremedia.antx.condition.IsDirectory;
import  com.idaremedia.antx.condition.IsNotSet;
import  com.idaremedia.antx.condition.IsNotWhitespace;
import  com.idaremedia.antx.condition.IsNumeric;
import  com.idaremedia.antx.condition.IsReference;
import  com.idaremedia.antx.condition.IsResource;
import  com.idaremedia.antx.condition.IsSetTrue;
import  com.idaremedia.antx.condition.Matches;
import  com.idaremedia.antx.condition.NoneSet;
import  com.idaremedia.antx.condition.StringEquals;
import  com.idaremedia.antx.print.EchoItemsTask;

/**
 * Adds several Condition compatible nested elements to a generic RuleTask. Basically
 * a big ol' cut-n-paste from ConditionBase's source ;-) with the additional
 * AntX conditions thrown in. Although, as of Ant 1.6 with its introduction of
 * {@linkplain FreeformRule}, it is no longer necessary to update this task with all
 * nestable conditions, it's still a good idea to do. By explicitly including a nested
 * element handler we allow script writers to use our conditions without having to
 * declare them manually.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public abstract class CompoundRuleTask extends RuleTask
{
    /**
     * Initializes a new CompoundRuleTask instance.
     **/
    protected CompoundRuleTask()
    {
        super();
    }


    /**
     * Initializes a new CV-labeled CompoundRuleTask instance.
     * @param iam CV-label (non-null)
     **/
    protected CompoundRuleTask(String iam)
    {
        super(iam);
    }


    /**
     * Initializes a new CompoundRuleTask subclass with custom
     * embedded setting.
     * @param iam CV-label
     * @param embed <i>true</i> if embedded
     **/
    protected CompoundRuleTask(String iam, boolean embed)
    {
        super(iam,embed);
    }


// ---------------------------------------------------------------------------------------
// Conditions that match standard Ant 'condition':
// ---------------------------------------------------------------------------------------

    /**
     * Adds a &lt;not&gt; condition container to this rule.
     **/
    public void addNot(Not n)
    {
        xaddCondition(n);
    }


    /**
     * Adds an &lt;and&gt; condition container to this rule.
     **/
    public void addAnd(And a)
    {
        xaddCondition(a);
    }


    /**
     * Adds an &lt;or&gt; condition container to this rule.
     **/
    public void addOr(Or o)
    {
        xaddCondition(o);
    }


    /**
     * Adds an &lt;available&gt; condition to this rule.
     **/
    public void addAvailable(Available a)
    {
        xaddCondition(a);
    }


    /**
     * Adds an &lt;uptodate&gt; condition to this rule.
     **/
    public void addUpToDate(UpToDate u)
    {
        xaddCondition(u);
    }


    /**
     * Adds a &lt;contains&gt; condition to this rule.
     **/
    public void addContains(Contains match)
    {
        xaddCondition(match);
    }


    /**
     * Adds an &lt;istrue&gt; condition to this rule.
     **/
    public void addIsTrue(IsTrue isT)
    {
        xaddCondition(isT);
    }


    /**
     * Adds an &lt;isfalse&gt; condition to this rule.
     **/
    public void addIsFalse(IsFalse isF)
    {
        xaddCondition(isF);
    }


    /**
     * Adds an &lt;os&gt; condition to this rule.
     **/
    public void addOS(Os os)
    {
        xaddCondition(os);
    }


    /**
     * Adds an &lt;http&gt; condition to this rule.
     **/
    public void addHTTP(Http h)
    {
        xaddCondition(h);
    }


    /**
     * Synonym for {@linkplain #addHTTP addHTTP} that better
     * reflects its purpose.
     **/
    public final void addHTTPAlive(Http h)
    {
        addHTTP(h);
    }


    /**
     * Adds a &lt;socket&gt; condition to this rule.
     **/
    public void addSocket(Socket soc)
    {
        xaddCondition(soc);
    }


    /**
     * Adds an &lt;checksum&gt; condition to this rule.
     **/
    public void addChecksum(Checksum a)
    {
        xaddCondition(a);
    }


    /**
     * Adds an &lt;filesmatch&gt; condition to this rule.
     **/
    public void addFilesmatch(FilesMatch fm)
    {
        xaddCondition(fm);
    }


// ---------------------------------------------------------------------------------------
// AntX extended conditons:
// ---------------------------------------------------------------------------------------

    /**
     * Adds a dummy &lt;print&gt; condition; diagnostic aid.
     * @since JWare/AntX 0.2
     **/
    public final void addPrint(EchoItemsTask debugprint)
    {
        xaddCondition(debugprint);
    }


    /**
     * Adds an &lt;isnotwhitespace&gt; condition to this rule.
     **/
    public void addIsNotWhitespace(IsNotWhitespace ws)
    {
        xaddCondition(ws);
    }


    /**
     * Adds an &lt;isboolean&gt; condition to this rule.
     * @since JWare/AntX 0.2
     **/
    public void addIsBoolean(IsBoolean isb)
    {
        xaddCondition(isb);
    }


    /**
     * Adds an &lt;isnumeric&gt; condition to this rule.
     **/
    public void addIsNumeric(IsNumeric isN)
    {
        xaddCondition(isN);
    }


    /**
     * Adds an &lt;isclass&gt; condition to this rule.
     **/
    public void addIsClass(IsClass isClz)
    {
        xaddCondition(isClz);
    }


    /**
     * Adds an &lt;isresource&gt; condition to this rule.
     **/
    public void addIsResource(IsResource isR)
    {
        xaddCondition(isR);
    }


    /**
     * Adds an &lt;isdirectory&gt; condition to this rule.
     * @since JWare/AntX 0.5
     **/
    public void addIsDirectory(IsDirectory isD)
    {
        xaddCondition(isD);
    }


    /**
     * Adds an &lt;isreference&gt; condition to this rule.
     **/
    public void addIsReference(IsReference isR)
    {
        xaddCondition(isR);
    }


    /**
     * Adds a &lt;isset&gt; condition to this rule.
     **/
    public void addIsSet(com.idaremedia.antx.condition.IsSet iss)
    {
        xaddCondition(iss);
    }


    /**
     * Adds a &lt;issettrue&gt; condition to this rule.
     * @since JWare/AntX 0.2
     **/
    public void addIsSetTrue(IsSetTrue ist)
    {
        xaddCondition(ist);
    }


    /**
     * Adds a &lt;isnotset&gt; condition to this rule.
     **/
    public void addIsNotSet(IsNotSet ins)
    {
        xaddCondition(ins);
    }


    /**
     * Adds an &lt;allset&gt; condition to this rule.
     * @since JWare/AntX 0.2
     **/
    public void addIsAllSet(AllSet allset)
    {
        xaddCondition(allset);
    }


    /**
     * Synonym for {@linkplain #addIsAllSet addIsAllSet}.
     * @since JWare/AntX 0.2
     **/
    public final void addAllSet(AllSet allset)
    {
        addIsAllSet(allset);
    }


    /**
     * Adds an &lt;allsettrue&gt; condition to this rule.
     * @since JWare/AntX 0.5
     **/
    public void addIsAllSetTrue(AllSetTrue allset)
    {
        xaddCondition(allset);
    }


    /**
     * Synonym for {@linkplain #addIsAllSetTrue addIsAllSetTrue}.
     * @since JWare/AntX 0.5
     **/
    public final void addAllSetTrue(AllSetTrue allset)
    {
        addIsAllSetTrue(allset);
    }


    /**
     * Adds a &lt;noneset&gt; condition to this rule.
     * @since JWare/AntX 0.2
     **/
    public void addIsNoneSet(NoneSet noneset)
    {
        xaddCondition(noneset);
    }


    /**
     * Synonym for {@linkplain #addIsNoneSet addIsNoneSet}.
     * @since JWare/AntX 0.2
     **/
    public final void addNoneSet(NoneSet noneset)
    {
        addIsNoneSet(noneset);
    }


    /**
     * Adds an &lt;anyset&gt; condition to this rule.
     * @since JWare/AntX 0.2
     **/
    public void addIsAnySet(AnySet anyset)
    {
        xaddCondition(anyset);
    }


    /**
     * Synonym for {@linkplain #addIsAnySet addIsAnySet}.
     * @since JWare/AntX 0.2
     **/
    public final void addAnySet(AnySet anyset)
    {
        addIsAnySet(anyset);
    }


    /**
     * Adds an &lt;anysettrue&gt; condition to this rule.
     * @since JWare/AntX 0.5
     **/
    public void addIsAnySetTrue(AnySetTrue anyset)
    {
        xaddCondition(anyset);
    }


    /**
     * Synonym for {@linkplain #addIsAnySetTrue addIsAnySetTrue}.
     * @since JWare/AntX 0.5
     **/
    public final void addAnySetTrue(AnySetTrue anyset)
    {
        addIsAnySetTrue(anyset);
    }


    /**
     * Adds an &lt;filenotempty&gt; condition to this rule.
     * @since JWare/AntX 0.2
     **/
    public void addFileNotEmpty(FileNotEmpty fne)
    {
        xaddCondition(fne);
    }


    /**
     * Adds an &lt;equals&gt; condition to this rule.
     * @since JWare/AntX 0.3 (converted to AntX StringEquals)
     **/
    public void addEquals(StringEquals e)
    {
        xaddCondition(e);
    }


    /**
     * Adds an &lt;ismatch&gt; condition to this rule.
     * @since JWare/AntX 0.3
     **/
    public void addIsMatch(Matches mc)
    {
        xaddCondition(mc);
    }



    /**
     * Synonym for {@linkplain #addIsMatch addIsMatch}.
     * @since JWare/AntX 0.2
     **/
    public final void addMatches(Matches mc)
    {
        addIsMatch(mc);
    }



    /**
     * Adds an &lt;antversion&gt; condition to this rule.
     * @since JWare/AntX 0.4
     **/
    public void addAntVersion(IsAntVersion vc)
    {
        xaddCondition(vc);
    }
}

/* end-of-CompoundRuleTask.java */
