/**
 * $Id: AssertTask.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.io.File;
import  java.text.MessageFormat;

import  org.apache.tools.ant.AntTypeDefinition;
import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.ComponentHelper;
import  org.apache.tools.ant.DynamicAttribute;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;
import  org.apache.tools.ant.types.Path;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.StringEquality;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.apis.BuildAssertionException;
import  com.idaremedia.antx.condition.FileNotEmpty;
import  com.idaremedia.antx.condition.ShortHandConditions;
import  com.idaremedia.antx.condition.URIable;
import  com.idaremedia.antx.parameters.Handling;
import  com.idaremedia.antx.parameters.IsA;

/**
 * Base class for Eiffelisque Design-by-Contract assertion tasks. Usually mapped to the
 * &lt;assert&gt; Ant task.
 * <p>
 * Using any of the attribute-based inlined shortcuts to setup the assertion's condition,
 * forces the assertion into a single-condition mode where only that one (inlined)
 * condition will be permitted.
 * <p>
 * <b>Example usage:</b><pre>
 *   &lt;assert antIs="1.5.4"/&gt;
 *   &lt;assert antLike="^.*version 1\.[5-9].*"/&gt;
 *
 *   &lt;assert istrue="${j2se14.present}"/&gt;
 *   &lt;assert isnotset="build.number"/&gt;
 *   &lt;assert isfile="${buildnum.sql.file}" filepath="${sqlfiles.path}/&gt;
 *   &lt;assert isnotwhitespace="cvs.revision" isa="variable"/&gt;
 *   &lt;assert isnumeric="${build.number}"/&gt;
 *   &lt;assert matches="(internal)|(distribution)" value="${build.type}"/&gt;
 *   &lt;assert isref="build.copyfilters" class="org.apache.tools.ant.types.PatternSet"/&gt;
 *
 *   &lt;<b>assert</b> logic="or" msgid="require.j2se13"&gt;
 *     &lt;isset property="j2se13.present"/&gt;
 *     &lt;isset property="j2se14.present"/&gt;
 *   &lt;/assert&gt;
 *
 *   &lt;<b>assert</b> failproperty="dirty.build" msgid="require.dist.clean"&gt;
 *      &lt;isset property="build.type"&gt;
 *      &lt;equals match="${DISTRIBUTION}" property="dist.type"/&gt;
 *      &lt;noneset&gt;
 *        &lt;property value="build.number"&gt;
 *        &lt;property value="fetch.disable"&gt;
 *        &lt;property value="clean.disable"&gt;
 *        &lt;variable value="build.duration"&gt;
 *        &lt;reference value="buildnum"&gt;
 *      &lt;/noneset&gt;
 *   &lt;/assert&gt;
 *
 *   <i><b>For Ant 1.6 or later:</b></i>
 *   &lt;typedef name="moonisright" classname="..."/&gt; <i>[your condition impl]</i>
 *   &lt;<b>assert</b>&gt;
 *      &lt;<b>moonisright</b> date=${DSTAMP}.../&gt;
 *   &lt;/assert&gt;
 *   -OR-
 *   &lt;<b>assert</b> moonisright="${DSTAMP}"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1 (was AssertionRule)
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 * @.expects Ant 1.6 or later for automatic freeform
 **/

public class AssertTask extends CompoundRuleTask
    implements ShortHandConditions, FreeformRule, DynamicAttribute
{
    /**
     * Initializes a new AssertTask instance.
     **/
    public AssertTask()
    {
        super(AntX.rules+"assert");
    }


    /**
     * Initializes a new CV-labeled AssertTask instance.
     * @param iam CV-label (non-null)
     **/
    public AssertTask(String iam)
    {
        super(iam);
    }


    /**
     * Initializes a new optionally embedded AssertTask subclass.
     * @param iam CV-label (non-null)
     * @param embedded <i>true</i> if act like embedded rule
     * @param droppable <i>true</i> if can be disabled globally
     **/
    protected AssertTask(String iam, boolean embedded, boolean droppable)
    {
        super(iam,embedded);
        m_droppable = droppable;
    }


    /**
     * Initializes the enclosing project for this task. Also
     * updates internal project-component helpers.
     **/
    public void setProject(Project project)
    {
        super.setProject(project);
        m_shh.setProject(project);
    }


    /**
     * Returns <i>true</i> if this assertion pays attention to
     * the fixture's "disable assertions" option(s). Is <i>true</i>
     * by default.
     * @since JWare/AntX 0.4
     **/
    public final boolean isDroppable()
    {
        return m_droppable;
    }

// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets name of the property set if this assertion fails. Example usage:
     * <code>&lt;ensure isfile="${build.log}" failproperty="errs.no.logfile"/&gt;</code>
     **/
    public final void setFailProperty(String property)
    {
        require_(!Tk.isWhitespace(property),"setfailp- nonwspc");
        setUpdateProperty(property);
    }


    /**
     * Returns the property updated if this assertion fails. Returns
     * default property-name if this attribute never set.
     **/
    public final String getFailProperty()
    {
        return getUpdateProperty();
    }


    /**
     * Sets value to use for failure property if this assertion fails.
     * Default value is string "<i>true</i>."
     **/
    public final void setFailValue(String failPropertyValue)
    {
        setUpdatePropertyValue(failPropertyValue);
    }


    /**
     * Returns this assertion's fail property value. Returns "<i>true</i>"
     * if never explicitly set.
     **/
    public final String getFailValue()
    {
        return getUpdatePropertyValue();
    }


    /**
     * Sets the default condition combining logic operator for
     * this assertion. Defaults to 'anding' multiple conditions. Example
     * usage: <pre>
     *   &lt;require logic="or"&gt;
     *       &lt;isset property="j2se14.present"/&gt;
     *       &lt;isset property="j2ee15.present"/&gt;
     *  &lt;/require&gt;
     * </pre>
     **/
    public final void setLogic(BooleanRule.Operator logic)
    {
        setDefaultCombineOperator(logic);
    }


    /**
     * Returns the combine logic of this task. Defaults to <i>AND</i>
     * if never set.
     * @since JWare/AntX 0.3
     **/
    public final BooleanRule.Operator getLogic()
    {
        return getDefaultCombineOperator();
    }


    /**
     * Sets this assertion to use an existing build rule as its
     * definition.  The reference will be evaluated (and verified)
     * the first time this task is evaluated.
     * @param ruleId build rule reference id (non-null)
     * @since JWare/AntX 0.2
     **/
    public final void setRuleId(Reference ruleId)
    {
        setReferralReference(ruleId);
    }


    /**
     * Sets an inlined message to be displayed if assertion fails.
     * Diagnostics aid.
     **/
    public final void setMsg(String msg)
    {
        if (!Tk.isWhitespace(msg)) {
            m_defaultMsg = uistrs().get("brul.assert.failed.lead",msg);
        } else {
            m_defaultMsg = msg;
        }
    }


    /**
     * Returns this task's inlined default message. Returns
     * <i>null</i> if never set.
     */
    public final String getDefaultMsg()
    {
        return m_defaultMsg;
    }


// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Returns <i>true</i> if the given referral's effect is undefined or
     * less-drastic than an error.
     * @since JWare/AntX 0.2
     **/
    protected boolean isIncompatibleReferral(ShareableCondition rule)
    {
        NoiseLevel effect = rule.getFailureEffect();
        if (NoiseLevel.isAsBadAs(effect,NoiseLevel.ERROR)) {
            return false;
        }
        return true;
    }



    /**
     * Determines if this task should blindly return <i>true</i> because
     * all build script assertions have been disabled.
     * @since JWare/AntX 0.4
     **/
    protected final boolean isDropped()
    {
        return isDroppable() && 
            Iteration.defaultdefaults().isAssertionsDisabled();
    }



    /**
     * Called to record this conditon's evaluation results. For
     * assertions a <i>false</i> result is very bad. <i>true</i> results
     * are basically ignored.
     **/
    protected void setEvalResult(boolean istrue, final String listing)
        throws BuildException
    {
        String prop = getFailProperty();

        if (!istrue) {
            String what = getFailValue();

            if (prop!=null) {
                checkIfProperty_(prop,true);
                getProject().setNewProperty(prop, what);

                log("Assertion false; setting failure property "+prop+
                    " to "+what, Project.MSG_DEBUG);
            }

            String errmsg;
            if (getMsgId()==null) {
                String inlined = getDefaultMsg();
                //Default if no inlined or inlined contains unresolved properties
                if (inlined==null) {
                    errmsg = uistrs().get("brul.assert.failed",listing);
                } else if (inlined.indexOf("${")>=0) {
                    errmsg = inlined;
                } else {
                    errmsg = MessageFormat.format(inlined,new Object[]{listing});
                }
            } else {
                errmsg = getMsg(newMsgGetter(listing));
            }

            violationOccured(errmsg);

            throw new BuildAssertionException(errmsg, getLocation());

        } else if (prop!=null) {
            String state = "true";
            if (isDropped()) {
                state += " (disabled)";
            }
            log("Assertion "+state+"; not setting failure property "+prop,
                Project.MSG_DEBUG);
        }
    }


    /**
     * Evaluates this assertion condition. If assertion condition
     * not met, this evaluation throws a build exception immediately;
     * so never returns <i>false</i>.
     * @throws BuildAssertionException if assertion <i>false</i>
     **/
    public boolean eval() throws BuildException
    {
        if (runReferral()) {
            return getReferral().eval(new Referee());
        }

        if (isDropped()) {
            return true;
        }

        verifyCanExecute_("eval");

        if (m_shh.isActivated()) {
            return defaultSingleConditionEval();

        }
        return defaultMultipleConditionEval();
    }



    /**
     * Called whenever this assertion's condition has not been met.
     * Subclasses can extend the default response which simply logs
     * and error message.
     * @param ermsg error message
     **/
    protected void violationOccured(String ermsg)
    {
        if (ermsg==null) {
            ermsg= uistrs().get("brul.assert.failed.default");
        }
        log(ermsg, Project.MSG_ERR);
    }


// ---------------------------------------------------------------------------------------
// Free-form (Application) Nested Elements:
// ---------------------------------------------------------------------------------------

    /**
     * Adds an arbitrary application-defined condition to
     * this assertion.
     * @param c custom condition definition (non-null)
     * @since JWare/AntX 0.3
     **/
    public void addConfigured(Condition c)
    {
        require_(c!=null,"add- nonzro condition");
        xaddCondition(c);
    }
    
    
    
    /**
     * Adds an arbitrary appication-defined condition shorthand to this
     * assertion. The name of the parameter must be the typedef'd name of
     * the condition (which must be all lowercased to work with Ant's dynamic
     * attribute facility).
     * @param name condition name (no namespace)
     * @param value condition evaluation URI fragment (non-null)
     * @since JWare/AntX 0.5
     * @throws BuildException if attribute's name is not a valid URIable condition.
     **/
    public void setDynamicAttribute(String name, String value)
    {
        require_(name!=null, "setFreeform- nonzro condition name");
        require_(value!=null,"setFreeform- nonzro uri fragment");

        boolean ok = false;
        Project P = getProject();
        
        AntTypeDefinition td = ComponentHelper.getComponentHelper(P).getDefinition(name);
        if (td!=null) {
            Class cls = td.getExposedClass(P);
            if (cls!=null && URIable.class.isAssignableFrom(cls)) {
                URIable c = (URIable)td.create(P);
                c.xsetFromURI(value);
                m_shh.activateCondition(c,"set"+name);
                ok = true;
            }
        }
        if (!ok) {
            String err = getAntXMsg("task.unsupported.attr",getTaskName(),name);
            log(err,Project.MSG_ERR);
            throw new BuildException(err,getLocation());
        }
    }

// ---------------------------------------------------------------------------------------
// ShortHand-Condition Definitions:
// ---------------------------------------------------------------------------------------

    /**
     * Attribute-shortcut for a single 'istrue' condition; for
     * example: <code>&lt;ensure istrue="${jdk14.present}"/&gt;</code>
     **/
    public void setIsTrue(String value)
    {
        m_shh.setIsTrue(value);
    }


    /**
     * Attribute-shortcut for a single 'isfalse' condition; for
     * example: <code>&lt;require isfalse="${jdk14.present}"/&gt;</code>
     **/
    public void setIsFalse(String value)
    {
        m_shh.setIsFalse(value);
    }


    /**
     * Attribute-shortcut for a single 'isset' condition; for
     * example: <code>&lt;require isset="stem"/&gt;</code>
     **/
    public void setIsSet(String property)
    {
        m_shh.setIsSet(property);
    }


    /**
     * Attribute-shortcut for a single 'issettrue' condition; for
     * example: <code>&lt;assert issettrue="test.engine.present"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setIsSetTrue(String property)
    {
        m_shh.setIsSetTrue(property);
    }


    /**
     * Attribute-shortcut for a single 'isnotset' condition; for
     * example: <code>&lt;require isnotset="build.number"/&gt;</code>
     **/
    public void setIsNotSet(String property)
    {
        m_shh.setIsNotSet(property);
    }


    /**
     * Set the type-cast modifier for this task's short-hand flex
     * condition.
     * @param isa the value source for short-hand condition
     * @since JWare/AntX 0.2
     **/
    public void setIsA(IsA isa)
    {
        m_shh.setIsA(isa);
    }


    /**
     * Attribute-shortcut for a single 'varset' condition; for
     * example: <code>&lt;require varset="is.distrib"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setVarSet(String variable)
    {
        m_shh.setVarSet(variable);
    }


    /**
     * Attribute-shortcut for a single 'varnotset' condition; for
     * example: <code>&lt;require varnotset="__loopcount"/&gt;</code>
     **/
    public void setVarNotSet(String variable)
    {
        m_shh.setVarNotSet(variable);
    }


    /**
     * Attribute-shortcut for a single 'varsettrue' condition; for
     * example: <code>&lt;assert varsettrue="loop.isdebug"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setVarSetTrue(String variable)
    {
        m_shh.setVarSetTrue(variable);
    }


    /**
     * Attribute-shortcut for checking existence of a reference;
     * for example: <code>&lt;assert isref="test.sources"/&gt;</code>
     **/
    public void setIsRef(String refId)
    {
        m_shh.setIsRef(refId);
    }


    /**
     * Parameter short-hand for a &lt;not&gt;'isreference' condition.
     * @since JWare/AntX 0.5
     **/
    public void setIsNotRef(String refId)
    {
        m_shh.setIsNotRef(refId);
    }


    /**
     * Specifies a required class for an is-reference condition.
     * @see #setIsRef
     **/
    public void setClass(String clzId)
    {
        m_shh.setClassName(clzId);
    }


    /**
     * Specifies the leniency of the kind-of-class check in an
     * is-reference condition.
     * @see #setIsRef
     **/
    public void setIsKindOf(boolean isKindOf)
    {
        m_shh.setIsKindOf(isKindOf);
    }


    /**
     * Attribute-shortcut for a single 'anyset' condition; for
     * example: <code>&lt;assert anyset="ant14.present,ant15.present"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setAnySet(String properties)
    {
        m_shh.setAnySet(properties);
    }


    /**
     * Attribute-shortcut for a single 'allset' condition; for
     * example: <code>&lt;assert allset="junit.present,log4j.present"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setAllSet(String properties)
    {
        m_shh.setAllSet(properties);
    }


    /**
     * Attribute-shortcut for a single 'allset' condition; for
     * example:
     * <span class="src">&lt;assert allsetlike="^build\..*"
     * whitespace="reject"/&gt;</span>.
     * @param pattern the filtering pattern for all script
     *        properties
     * @since JWare/AntX 0.4
     **/
    public void setAllSetLike(String pattern)
    {
        m_shh.setAllSetLike(pattern);
    }


    /**
     * Attribute-shortcut for a single 'noneset' condition; for
     * example: <code>&lt;assert noneset="build.number,build.time"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setNoneSet(String properties)
    {
        m_shh.setNoneSet(properties);
    }

// ---------------------------------------------------------------------------------------

    /**
     * Attribute-shortcut for a single 'isboolean' condition; for
     * example: <code>&lt;assert isboolean="${jprobe.isdebug}"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setIsBoolean(String value)
    {
        m_shh.setIsBoolean(value);
    }


    /**
     * Attribute-shortcut for a single 'isnumeric' condition; for
     * example: <code>&lt;assert isnumeric="${build.number}"/&gt;</code>
     **/
    public void setIsNumeric(String value)
    {
        m_shh.setIsNumeric(value);
    }


    /**
     * Set the 'greather-than-or-equal' modifier for a short-hand
     * 'isnumeric' condition.
     **/
    public void setGT(String gt)
    {
        m_shh.setGT(gt);
    }


    /**
     * Sets the 'greather-than' modifier for a short-hand
     * 'isnumeric' condition.
     **/
    public void setGTE(String gte)
    {
        m_shh.setGTE(gte);
    }


    /**
     * Sets the 'less-than' modifier for a short-hand 'isnumeric
     * condition.
     **/
    public void setLT(String lt)
    {
        m_shh.setLT(lt);
    }


    /**
     * Sets the 'less-than-or-equal' modifier for a short-hand
     * 'isnumeric' condition.
     **/
    public void setLTE(String lte)
    {
        m_shh.setLTE(lte);
    }


// ---------------------------------------------------------------------------------------

    /**
     * Attribute-shortcut for checking existence of a file;
     * for example: <code>&lt;ensure isfile="${build.logfile}"/&gt;</code>
     **/
    public void setIsFile(File file)
    {
        m_shh.setIsFile(file);
    }


    /**
     * Attribute-shortcut for checking existence of a directory;
     * for example: <code>&lt;ensure isdirectory="${module.classes}"/&gt;</code>
     **/
    public void setIsDirectory(File directory)
    {
        m_shh.setIsDirectory(directory);
    }


    /**
     * Attribute-shortcut for checking that a file exists and is not empty;
     * for example: <code>&lt;assert filenotempty="${sql.results}"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setFileNotEmpty(File file)
    {
        FileNotEmpty t= new FileNotEmpty(file,getProject());
        m_shh.activateCondition(t,"setFileNotEmpty");
    }


    /**
     * Attribute-shortcut for a single 'isresource' condition; for
     * example: <code>&lt;assert isresource="${build.uistrs}"/&gt;</code>
     **/
    public void setIsResource(String resource)
    {
        m_shh.setIsResource(resource);
    }


    /**
     * Attribute-shortcut for a single 'isclass' condition; for
     * example: <code>&lt;assert isclass="junit.framework.Assert"/&gt;</code>
     * @param classname name of class to verify (can be loaded)
     **/
    public void setIsClass(String classname)
    {
        m_shh.setIsClass(classname);
    }


    /**
     * Specifies a lookup classpath for a shorthand 'available'
     * condition.
     **/
    public void setClasspath(Path clsp)
    {
        m_shh.setClasspath(clsp);
    }


    /**
     * Specifies a lookup filepath for a shorthand 'available'
     * condition.
     * @since JWare/AntX 0.2
     **/
    public void setFilepath(Path filp)
    {
        m_shh.setFilepath(filp);
    }


    /**
     * Specifies a lookup classpath-ref for an inlined 'available'
     * condition.
     **/
    public void setClasspathRef(Reference cpr)
    {
        m_shh.setClasspathRef(cpr);
    }


    /**
     * Specifies Ant's runtime classes should be included for
     * an inlined 'isclass' condition; like <code>&lt;assert isclass="foo"
     * systemclasses="false"/&gt;</code>.
     * @param included <i>false</i> if system classes excluded
     * @since JWare/AntX 0.2
     **/
    public void setSystemClasses(boolean included)
    {
        m_shh.setSystemClasses(included);
    }


// ---------------------------------------------------------------------------------------
    /**
     * Shortcut for checking the OS's family name; for example:
     * <code>&lt;require osfamily="unix"/&gt;</code>
     **/
    public void setOSFamily(String family)
    {
        m_shh.setOSFamily(family);
    }


    /**
     * Shortcut for checking the OS's information; for example:
     * <code>&lt;require os="unix,Linux,i686"/&gt;</code>
     * @since JWare/AntX 0.4
     **/
    public void setOS(String selector)
    {
        m_shh.setOS(selector);
    }


    /**
     * Shortcut for checking whether a web server is alive or
     * not; for example: <code>&lt;assert httpalive="http://pet.mycompany.com"/&gt;</code>
     * @since JWare/AntX 0.3
     **/
    public void setHttpAlive(String url)
    {
        m_shh.setHttpAlive(url);
    }


// ---------------------------------------------------------------------------------------

    /**
     * Attribute-shortcut for checking whether a property's value is
     * all whitespace; for example:
     * <code>&lt;require isnotwhitespace="${module.title}"/&gt;</code>
     **/
    public void setIsNotWhitespace(String value)
    {
        m_shh.setIsNotWhitespace(value);
    }


    /**
     * Attribute-shorthand for checking an '<i>equals</i>' condition. Most
     * useful during task build-file verification; for example,
     * <code>&lt;assert equals="com.idaremedia.antx" value="${stem}"/&gt;</code>
     * @param arg1 value against which other arg ('value') compared
     **/
    public void setEquals(String arg1)
    {
        StringEquality t= m_shh.getEqualityCondition();
        t.setKnownArg(arg1);
        m_shh.activateCondition(t,"setEquals");
    }


    /**
     * Attribute-shortcut for checking a '<i>notequals</i>' condition;
     * for example, <code>&lt;assert notequals="debug" ignorecase="true"
     * value="${build.type}"/&gt;</code>
     * @param arg1 value against which other arg ('value') compared
     **/
    public void setNotEquals(String arg1)
    {
        StringEquality t= m_shh.getEqualityCondition();
        t.setKnownArg(arg1);
        t.setNegate(true);
        m_shh.activateCondition(t,"setNotEquals");
    }


    /**
     * Attribute-shortcut for checking a '<i>startswith</i>' condition; for
     * example, <code>&lt;assert startswith="com.idaremedia." value="${stem}"/&gt;</code>
     * @param arg1 value which other arg ('value') must start-with
     **/
    public void setStartsWith(String arg1)
    {
        StringEquality t= m_shh.getEqualityCondition();
        t.setKnownArg(arg1);
        t.setOperator(StringEquality.OP_STARTSWITH);
        m_shh.activateCondition(t,"setStartsWith");
    }


    /**
     * Attribute-shortcut for checking a '<i>endswith</i>' condition; for
     * example, <code>&lt;assert endswith=".xml" ignorecase="true" value="${filename}"/&gt;</code>
     * @param arg1 value which other arg ('value') must end-with
     **/
    public void setEndsWith(String arg1)
    {
        StringEquality t= m_shh.getEqualityCondition();
        t.setKnownArg(arg1);
        t.setOperator(StringEquality.OP_ENDSWITH);
        m_shh.activateCondition(t,"setEndsWith");
    }


    /**
     * Attribute-shortcut for checking a '<i>matches</i>' condition; for
     * example, <code>&lt;assert matches="(internal)|(dist)" value="${build.type}"/&gt;</code>
     * @param pattern pattern against which other arg ('value') matched
     **/
    public void setMatches(String pattern)
    {
        m_shh.setMatches(pattern);
    }


    /**
     * Attribute-shortcut for checking a '<i>contains</i>' condition; for example,
     * <code>&lt;assert contains="idaremedia.antx" value="${stem}"/&gt;</code>
     * @param substring substring which must exist in other arg ('value')
     **/
    public void setContains(String substring)
    {
        StringEquality t= m_shh.getEqualityCondition();
        t.setKnownArg(substring);
        t.setOperator(StringEquality.OP_CONTAINS);
        m_shh.activateCondition(t,"setContains");
    }


    /**
     * Sets the unknown value being compared in a inlined equality condition.
     * @param value the second (unknown) value to match
     **/
    public void setValue(String value)
    {
        m_shh.setValue(value);
    }


    /**
     * Sets the unknown value in an inlined equality condition as a variable
     * property's name. The exported property's value will be read at comparison
     * time.
     * @param variable the variable whose contents will be matched
     * @since JWare/AntX 0.2
     **/
    public void setVariable(String variable)
    {
        m_shh.setVariable(variable);
    }


    /**
     * Synonym for {@linkplain #setVariable setVariable}.
     **/
    public final void setVar(String variable)
    {
        setVariable(variable);
    }


    /**
     * Sets the unknown value in an inlined equality condition as a property's
     * name. The property's value will be read at comparison time.
     * @param property the property whose contents will be matched
     * @since JWare/AntX 0.2
     **/
    public void setProperty(String property)
    {
        m_shh.setProperty(property);
    }


    /**
     * Sets the unknown value in an inlined equality condition as a reference's
     * name. The reference's (stringified) value will be read at comparison time.
     * @param refid the reference whose contents will be matched
     * @since JWare/AntX 0.2
     **/
    public void setReference(String refid)
    {
        m_shh.setReference(refid);
    }

// ---------------------------------------------------------------------------------------

    /**
     * Defines short hand &lt;antversion is="&#8230;"/&gt; condition.
     * @since JWare/AntX 0.4
     **/
    public void setAntIs(String version)
    {
        m_shh.setAntIs(version);
    }


    /**
     * Defines short hand &lt;antversion pattern="&#8230;"/&gt;
     * condition.
     * @since JWare/AntX 0.4
     **/
    public void setAntLike(String pattern)
    {
        m_shh.setAntLike(pattern);
    }

// ---------------------------------------------------------------------------------------

    /**
     * Set the case-sensitivity of this task's short-hand condition.
     * @param ignore <i>true</i> if case insensitive
     **/
    public void setIgnoreCase(boolean ignore)
    {
        m_shh.setIgnoreCase(ignore);
    }


    /**
     * Returns <i>true</i> if task's short-hand condition's ignore case
     * option is activated.
     * @since JWare/AntX 0.2
     **/
    public boolean isIgnoreCase()
    {
        return m_shh.isIgnoreCase();
    }


    /**
     * Set the will-trim option of this task's short-hand condition
     * @param trim <i>true</i> if condition' unknown strings will be trimmed
     **/
    public void setTrim(boolean trim)
    {
        m_shh.setTrim(trim);
    }


    /**
     * Returns <i>true</i> if task's short-hand condition's trim strings
     * option is activated.
     * @since JWare/AntX 0.2
     **/
    public boolean willTrim()
    {
        return m_shh.willTrim();
    }


    /**
     * Set the ignore-whitespace of a inlined string comparison condition.
     * @param response <i>ignore</i> if all-whitespace values should be ignored
     * @since JWare/AntX 0.2
     **/
    public void setWhitespace(Handling response)
    {
        m_shh.setWhitespace(response);
    }


    /**
     * Returns <i>true</i> if task's short-hand condition's ignore-
     * whitespace option is activated.
     * @since JWare/AntX 0.2
     **/
    public boolean ignoreWhitespace()
    {
        return m_shh.ignoreWhitespace();
    }


    /**
     * Set the allow-synonyms modifier for this task's short-hand condition.
     * @param allowAll <i>true</i> if boolean synonyms are allowed
     * @since JWare/AntX 0.2
     **/
    public void setSynonyms(boolean allowAll)
    {
        m_shh.setSynonyms(allowAll);
    }


    /**
     * Specifies how a property-dependent condition should treat
     * unresolved property values.
     * @since JWare/AntX 0.4
     **/
    public void setMalformed(Handling response)
    {
        m_shh.setMalformed(response);
    }


    /**
     * Returns how this assertion will handle unresolved
     * properties.
     * @since JWare/AntX 0.4
     **/
    public Handling getMalformedHandling()
    {
        return m_shh.getMalformedHandling();
    }


// ---------------------------------------------------------------------------------------

    /**
     * Diagnostic assertion for checking that a copied reference's project has not
     * been stomped on. Generic name but currently only implemented for 'project'
     * field.
     * @since JWare/AntX 0.3
     * @see #setItem1
     **/
    public void setSameField(String fieldname)
    {
        require_("project".equals(fieldname),"setfield- only 'project'");
        EqualFields t = (EqualFields)m_shh.getHalfBakedObject(EqualFields.class);
        m_shh.activateCondition(t,"setSameField");
    }


    /**
     * Sets the first reference for a &lt;samefield&gt; shorthand assertion.
     * @param refid reference identifier (non-null)
     * @since JWare/AntX 0.3
     **/
    public void setItem1(String refid)
    {
        EqualFields t = (EqualFields)m_shh.getHalfBakedObject(EqualFields.class);
        t.setItem1(refid);
    }


    /**
     * Sets the second reference for a &lt;samefield&gt; shorthand assertion.
     * @param refid reference identifier or "null" string for null comparison
     * @since JWare/AntX 0.3
     **/
    public void setItem2(String refid)
    {
        EqualFields t = (EqualFields)m_shh.getHalfBakedObject(EqualFields.class);
        t.setItem2(refid);
    }

// ---------------------------------------------------------------------------------------

    private String m_defaultMsg;
    private final ShortHandHelper m_shh = new ShortHandHelper(this);
    private boolean m_droppable=true;
}

/* end-of-AssertTask.java */
