/**
 * $Id: StringEquals.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.StringEquality;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.TrimEnabled;
import  com.idaremedia.antx.parameters.ValueMatchEnabled;

/**
 * Flexible String-Equals condition that evaluates <i>true</i> if an unknown value
 * equals a predefined value. This condition allows script writers to include simple
 * &lt;equals&gt; checks in rules where properties or variables may or may not
 * exist before the rule is parsed; for instance:
 * <pre>
 *  &lt;rule id="is.public.dist"&gt;
 *    &lt;require&gt;
 *      &lt;equals match="public" property="dist.type"/&gt;
 *      &lt;noneset&gt;
 *          &lt;property name ="build.number"&gt;
 *          &lt;property name ="disable.cvs"&gt;
 *          &lt;property name ="disable.clean"&gt;
 *      &lt;/noneset&gt;
 *    &lt;/require&gt;
 *  &lt;/rule&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple (after fully configured)
 * @.group   api,infra
 * @see      StringEquality
 **/

public final class StringEquals extends FlexCondition
    implements IgnoreCaseEnabled, TrimEnabled, ValueMatchEnabled
{
    /**
     * Initializes a new StringEquals condition. The new
     * condition's match value and flex value must be defined
     * before it is evaluated.
     **/
    public StringEquals()
    {
        super();
        m_impl.setOperator(StringEquality.OP_EQUALS);
        getValueHelper().setLenient(false);
    }


    /**
     * Initializes the enclosing project of this condition.
     * Also updates this condition's helper bits.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_impl.setProject(P);
    }


    /**
     * Returns this equality condition's underlying unknown
     * argument's flex string. Never returns <i>null</i>.
     **/
    protected final FlexString getValueHelper()
    {
        return m_impl.getUnknownValueGetter();
    }


// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets this condition's known to-be-equaled value.
     * @param value value against which determined value is checked (non-null)
     **/
    public void setMatch(String value)
    {
        require_(value!=null,"setMatch- nonzro mtch valu");
        m_impl.setKnownArg(value);
    }


    /**
     * Returns value against which the determined value is checked.
     * Returns <i>null</i> if never set.
     **/
    public final String getMatch()
    {
        return m_impl.getKnownArg();
    }



    /**
     * Sets a to-be-checked literal value.
     * @param value value to be matched (non-null)
     **/
    public final void setValue(String value)
    {
        require_(value!=null,"setValue- nonzro");
        setLiteral(value);
    }



    /**
     * Set whether the values should be trimmed of whitespace
     * before they are compared.
     **/
    public void setTrim(boolean trim)
    {
        m_impl.setTrim(trim);
    }


    /**
     * Returns <i>true</i> if the values will be trimmed before
     * they are compared.
     **/
    public final boolean willTrim()
    {
        return m_impl.willTrim();
    }



    /**
     * Set whether the values should be lower-cased before
     * they are compared.
     **/
    public void setIgnoreCase(boolean ignore)
    {
        m_impl.setIgnoreCase(ignore);
    }


    /**
     * Returns <i>true</i> if the values will be lower-cased
     * before they are compared.
     **/
    public final boolean isIgnoreCase()
    {
        return m_impl.isIgnoreCase();
    }


// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Returns <i>true</i> if the calculated value equals the expected
     * value.
     **/
    public boolean eval()
    {
        verifyCanEvaluate_("eval");

        return m_impl.eval();
    }


    /**
     * Verifies that this condition is in valid project and has both its
     * known and unknown values defined.
     * @param calr caller's identifier
     * @throws BuildException if not in project or all bits not defined
     **/
    protected void verifyCanEvaluate_(String calr)
    {
        super.verifyCanEvaluate_(calr);

        if (getMatch()==null) {
            String ermsg = uistrs().get("task.needs.this.attr",
                                        getTypicalName(),"match");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
    }


    private final StringEquality m_impl = new StringEquality();
}

/* end-of-StringEquals.java */
