/**
 * $Id: FlexIncludesIterator.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;
import  java.util.NoSuchElementException;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ownhelpers.InnerProperties;
import  com.idaremedia.antx.ownhelpers.ProjectDependentSkeleton;
import  com.idaremedia.antx.ownhelpers.PropertiesIterator;
import  com.idaremedia.antx.ownhelpers.UnresolvedProperty;

/**
 * Special iterator of the set conditions (like &lt;allset&gt;) nested elements. This
 * iterator returns a {@linkplain com.idaremedia.antx.FlexString FlexString} for each
 * iteration including the collections of properties returned by nested
 * <span class="src">&lt;properties&gt</span> elements. For unresolved property values,
 * this iterator will substitute the marker {@linkplain UnresolvedProperty#VALUE unresolved}
 * value as the returned item's value.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public final class FlexIncludesIterator extends ProjectDependentSkeleton
    implements Iterator
{
    private static final String IAM_ = AntX.conditions+"FlexIncludesIterator:";


    /**
     * Initializes a new value iterator for an AntX set-based
     * condition.
     * @param owner the set condition (non-null)
     **/
    public FlexIncludesIterator(FlexOuterCondition owner)
    {
        AntX.require_(owner!=null,IAM_,"ctor- nonzro owning set");
        m_owner = owner;
        m_scratchFV = new ExternalPropertyDef();
    }


    /**
     * Triggered on first call to {@linkplain #hasNext}. Will
     * setup iteration parameters. Returns <i>false</i> if nothing
     * to iterate.
     **/
    private boolean prepareToIterate()
    {
        List l= m_owner.getIncludes();
        if (l.isEmpty()) {
            return false;//NB:don't waste time+memory
        }

        setProject(m_owner.getProject());
        m_scratchFV.setProject(getProjectNoNull());
        m_scratchFV.setTrim(m_owner.willTrim());
        m_includesItr = l.iterator();
        return true;
    }


    /**
     * Returns <i>true</i> if there is at least one item left
     * to iterate.
     **/
    public boolean hasNext()
    {
        if (m_includesItr==null) {
            if (!prepareToIterate()) {
                return false;
            }
        }

        boolean gotOne = (m_nextItem!=null);

        if (!gotOne) {
            if (m_innerItr!=null) {
                if (m_innerItr.hasNext()) {
                    Map.Entry entry= (Map.Entry)m_innerItr.next();
                    m_scratchFV.set(entry);
                    m_nextItem = m_scratchFV;
                    return true;
                }
                m_innerItr.dispose();
                m_innerItr = null;
                if (m_privateCall) {
                    return false;
                }
            }
            while (m_includesItr.hasNext()) {
                Object candidate = m_includesItr.next();

                if (candidate instanceof InnerProperties) {
                    m_innerItr = new PropertiesIterator((InnerProperties)candidate);
                    m_innerItr.setCheckBrokenSubstitution();
                    m_innerItr.setBrokenSubstitutionProxy(UnresolvedProperty.VALUE);
                    m_privateCall=true;
                    gotOne = hasNext();//get 1st one setup!
                    m_privateCall=false;
                    if (gotOne) {
                        return true;
                    }
                    candidate = null;
                }

                if (candidate!=null) {
                    m_nextItem = candidate;
                    gotOne = true;
                    break;
                }
            }
        }

        return gotOne;
    }


    /**
     * Returns the next <span class="src">FlexString</span> to
     * be handled. Note that the returned value may be part of the
     * property collection of a nested property set.
     * @throws NoSuchElementException if nothing left to iterate
     **/
    public Object next()
    {
        if (m_nextItem==null) {
            throw new NoSuchElementException();
        }
        Object result = m_nextItem;
        m_nextItem = null;
        return result;
    }


    /**
     * Always generates an
     * <span class="src">UnsupportedOperationException</span>.
     * @throws UnsupportedOperationException always
     **/
    public void remove()
    {
        throw new UnsupportedOperationException();
    }



    private final FlexOuterCondition m_owner;
    private Iterator m_includesItr;
    private PropertiesIterator m_innerItr;
    private Object m_nextItem;
    private final ExternalPropertyDef m_scratchFV;
    private boolean m_privateCall;
}

/* end-of-FlexIncludesIterator.java */
