/**
 * $Id: ResultsHelper.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  java.util.Map;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.parameters.IsA;

/**
 * Utility to assign the result values to the appropriate type of fixture element.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    impl,helper
 **/

public final class ResultsHelper
{
    /**
     * Assign the result values to the appropriate type of fixture element. Note
     * that the incoming conclusion must be one of the fixed interpreter
     * constant values! If the result type is anything but write-once properties
     * this helper will <em>clear</em> pre-existing values if results are clean.
     * @param config call config (non-null)
     * @param errs number of final errors as a string (non-null)
     * @param warnings number of final warnings as a string (non-null)
     * @param result the <em>normalized</em> conclusion (non-null)
     **/
    public static void set(InterpretParameters config, String errs, String warnings, String result)
    {
        final Project P = config.getProject();
        if (config.updateProperties()) {
            boolean unclean = !LogInterpreter.CLEAN.equals(result);
            switch (config.getResultType().getIndex()) {
                case IsA.VARIABLE_INDEX: {
                    if (unclean) {
                        ExportedProperties.set(config.getErrorCountProperty(),errs);
                        ExportedProperties.set(config.getWarningCountProperty(),warnings);
                    } else {
                        ExportedProperties.unset(config.getErrorCountProperty());
                        ExportedProperties.unset(config.getWarningCountProperty());
                    }
                    ExportedProperties.set(config.getUpdateProperty(),result);
                    break;
                }
                case IsA.REFERENCE_INDEX: {
                    Map refstable = P.getReferences();
                    synchronized(refstable) {
                        if (unclean) {
                            refstable.put(config.getErrorCountProperty(),errs);
                            refstable.put(config.getWarningCountProperty(),warnings);
                        } else {
                            refstable.remove(config.getErrorCountProperty());
                            refstable.remove(config.getWarningCountProperty());
                        }
                        refstable.put(config.getUpdateProperty(),result);
                    }
                    break;
                }
                default: {
                    if (unclean) {
                        P.setNewProperty(config.getErrorCountProperty(),errs);
                        P.setNewProperty(config.getWarningCountProperty(),warnings);
                    }
                    P.setNewProperty(config.getUpdateProperty(),result);
                }
            }//switch
        }//do-updates
    }



    /**
     * Like 
     * {@linkplain #set(InterpretParameters config, String errs, String warnings, String result)
     *  set(InterpretParameters,String,String,String)} but results are passed as scalar values.
     **/
    public static void set(InterpretParameters config, int nErrs, int nWarnings, String result)
    {
        set(config,String.valueOf(nErrs),String.valueOf(nWarnings),result);
    }
    

    private ResultsHelper()
    {
        /*disallow*/
    }
}

/* end-of-ResultsHelper.java */