/**
 * $Id: LogsUsingTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.ownhelpers.FeedbackSink;

/**
 * Starter implementation for any task that reads and/or manipulates information
 * captured by the various {@linkplain LogsRecorder LogsRecorders}.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public abstract class LogsUsingTask extends AssertableTask
{
    /**
     * Initializes a CV-labeled task instance.
     * @param iam CV-label (non-null)
     **/
    protected LogsUsingTask(String iam)
    {
        super(iam);
    }


// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------


    /**
     * Sets whether only the captured important logs will be used by
     * this task. Is <i>true</i> by default.
     * @param is <i>false</i> to use all captured logs
     **/
    public void setImportant(boolean is)
    {
        m_isImportant = is;
    }


    /**
     * Returns <i>true</i> if only the important logs are used.
     * Is <i>true</i> by default. If <i>false</i> all captured
     * log information is used.
     **/
    public final boolean shouldBeImportant()
    {
        return m_isImportant;
    }


    /**
     * Tells this task to clear its enclosing logs recorder after
     * copy.
     **/
    public void setReset(boolean reset)
    {
        m_resetAfter = reset;
    }


    /**
     * Returns <i>true</i> if this task will clear its enclosing
     * logs recorder's buffers after using it. Recorder is cleared
     * only if this task succesfully completes processing.
     **/
    public final boolean willReset()
    {
        return m_resetAfter;
    }


    /**
     * Sets the log information source for this task. Output is used
     * from the nearest logs recorder of matching type.
     * @param source one of [stdio,antlog]
     * @throws BuildException if bad source defined
     **/
    public void setFrom(FeedbackSink source)
    {
        require_(source!=null,"setFrom- nonzro src");
        verifyLogsSource(source);
        m_from = source;
    }


    /**
     * Returns this task's log information source. If never defined
     * will return {@linkplain FeedbackSink#ANTLOG ANTLOG}. Never
     * returns <i>null</i>.
     **/
    public FeedbackSink getFrom()
    {
        return m_from;
    }


    /**
     * Verifies the log information source for this task. By default
     * ensures the logs source is a grouping identifier (like "antlog").
     * @param source one of [stdio,antlog]
     * @throws BuildException if bad source defined
     **/
    protected void verifyLogsSource(FeedbackSink source)
    {
        if (!FeedbackSink.isGrouping(source)) {
            String ermsg = getAntXMsg("brul.logs.bad.capture.sink",
                                      getTaskName(), source.getValue());
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }

// ---------------------------------------------------------------------------------------
// Execution:
// ---------------------------------------------------------------------------------------

   /**
    * Return <i>true</i> if this task can accept it's inputs from a
    * source other than a logs recorder. Important to support sources other
    * than logs recorders for some evaluation tasks.
    * @since JWare/AntX 0.5 
    **/
    protected boolean isSourceFlexible()
    {
        return false;
    }



    /**
     * Determine the logs recorder to use. The current value of
     * the "{@linkplain #setFrom from}" option determines the recorder.
     * @param allowNull <i>true</i> if can return <i>null</i> otherwise
     *        the {@linkplain EmptyLogs#INSTANCE null-proxy} is returned
     **/
    protected final LogsRecorder getRecorder(boolean allowNull)
    {
        LogsRecorder r=null;

        switch (getFrom().getIndex()) {
            case FeedbackSink.ANTLOG_INDEX: {
                r = CapturedLogs.getRecorder();
                break;
            }
            case FeedbackSink.STDIO_INDEX:
            case FeedbackSink.STDOUT_INDEX:
            case FeedbackSink.STDERR_INDEX: {
                r = CapturedLogs.getStdIORecorder();
                break;
            }
        }
        if (r==null && !allowNull) {
            r = EmptyLogs.INSTANCE;
        }
        return r;
    }


    /**
     * Determine the logs to be copied. Never returns <i>null</i>,
     * but can return the empty string.
     * @see CapturedLogs#getRecorder
     **/
    protected final String getVUTLog()
    {
        LogsRecorder r = getRecorder(false);
        if (shouldBeImportant()) {
            return r.copyOfImportantLogs();
        }
        return r.copyOfAllLogs();
    }


    /**
     * Verifies that this task is enclosed by at least one
     * {@linkplain LogsRecorder} enabled taskset.
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (getRecorder(true)==null) {
            String warning = getAntXMsg("brul.logs.no.recorder");
            int level = isSourceFlexible() ? Project.MSG_VERBOSE : Project.MSG_WARN;
            log(warning,level);
        }
    }


    private boolean m_isImportant=true;//NB:by default only important checked
    private boolean m_resetAfter;//NB: leaves logs alone
    private FeedbackSink m_from = FeedbackSink.ANTLOG;//NB:compatible with 0.2
}

/* end-of-LogsUsingTask.java */
