/**
 * $Id: CaptureStreamsTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  java.io.PrintStream;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.BuildError;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.apis.Responses;
import  com.idaremedia.antx.ownhelpers.FeedbackSink;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * Execution "net" task that captures all output to either
 * <span class="src">System</span> (or stdio) stream. The  enclosing application
 * context must have the property permission to replace the System stream
 * handlers.  Like CaptureLogsTask, CaptureStreamsTasks can be nested (but are not
 * usually).
 * <p>
 * The CaptureStreamsTask has a (immutable) predefined notion of "important" events.
 * If either system stream is being captured (singlely) then all of the information
 * written to that stream is considered important. If both streams are being captured,
 * anything written to the <span class="src">System.err</span> stream is considered
 * important.
 * <p>
 * You can easily extend the CaptureStreamsTask to capture additional kinds of
 * execution context; refer to the {@linkplain #performTheEnclosedTaskList} method.
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;captureoutput&gt;
 *      &lt;...<i>Your Tasks Here</i>&gt;
 *      &lt;copylogged from="stdio" important="no" tofile="..."/&gt;
 *   &lt;/captureoutput&gt;
 *
 *   &lt;captureoutput fromstream="stderr"&gt;
 *      &lt;...<i>Your Tasks Here</i>&gt;
 *      &lt;copylogged from="stdio" toproperty="my.errs"/&gt;
 *   &lt;/captureoutput&gt;
 *   &lt;assert equals="" property="my.errs"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @see      CaptureLogsTask
 **/

public class CaptureStreamsTask extends TaskSet implements LogsRecorder
{
    /**
     * Initializes a new CaptureStreamsTask instance.
     **/
    public CaptureStreamsTask()
    {
        super(AntX.capture+"CaptureStreamsTask:");
    }


    /**
     * Initializes a new CV-labeled CaptureStreamsTask instance.
     * @param iam CV-label (non-null)
     **/
    public CaptureStreamsTask(String iam)
    {
        super(iam);
    }



    /**
     * Sets the stream from which this task will capture output.
     * Must be one of [<span class="src">{@linkplain FeedbackSink#STDIO}</span>,
     * <span class="src">{@linkplain FeedbackSink#STDERR}</span>,
     * <span class="src">{@linkplain FeedbackSink#STDOUT}</span>].
     * @param ss system stream (non-null)
     * @throws BuildException if bad source defined
     **/
    public void setFromStream(FeedbackSink ss)
    {
        require_(ss!=null, "setstrm- err|out|both");

        if (!FeedbackSink.isSystem(ss)) {
            String ermsg = getAntXMsg("brul.logs.bad.capture.sink",
                                      getTaskName(),ss.getValue());
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        m_streamChoice = ss;
    }



    /**
     * Returns the capture stream choice for this task. If never set,
     * defaults to <span class="src">{@linkplain FeedbackSink#STDIO}</span>
     * to capture both System err and out streams.
     **/
    public FeedbackSink getFromStream()
    {
        return m_streamChoice;
    }



    /**
     * Returns <i>true</i> if this task would consider the given
     * noise level important.
     * @param nl noise level (non-null)
     **/
    public final boolean isImportant(NoiseLevel nl)
    {
        require_(nl!=null,"isImportnt- nonzro NL");
        if (NoiseLevel.isAsBadAs(nl,NoiseLevel.WARNING)) {
            return true;
        }
        return false;
    }



    /**
     * Returns a <em>copy</em> of current main stream buffer's contents.
     * If both streams are being captured, this method returns the contents
     * of the <span class="src">System.err</span> stream. Returns an empty
     * string if nothing being captured.
     **/
    public String copyOfImportantLogs()
    {
        if (m_mainFilterStream!=null) {
            return m_mainFilterStream.copyBuffer();
        }
        return "";
    }



    /**
     * Returns a <em>copy</em> of the current secondary stream buffer's
     * contents or the empty string if not such stream. When tracking
     * both streams, this method returns the messages written to the
     * <span class="src">System.out</span> stream. Never returns <i>null</i>.
     **/
    public String copyOfSecondaryLogs()
    {
        if (m_secondFilterStream!=null) {
            return m_secondFilterStream.copyBuffer();
        }
        return "";
    }



    /**
     * Returns a <em>copy</em> of all content captured by this task
     * so far. This can include all the messages written to the standard
     * <span class="src">System.err</span> stream and the standard
     * <span class="src">System.out</span> stream if both are being
     * captured. The ordering of the contents is determined by the
     * locking policies of the underlying streams.
     * @see #copyOfImportantLogs
     **/
    public String copyOfAllLogs()
    {
        if (m_allBuffer!=null) {
            return m_allBuffer.substring(0);
        }
        return m_mainFilterStream.copyBuffer();
    }



    /**
     * Resets this task's buffers as iff no messages ever recorded.
     **/
    public void clearLogs()
    {
        if (m_allBuffer!=null) {
            m_allBuffer.delete(0,m_allBuffer.length());
        }
        if (m_secondFilterStream!=null) {
            m_secondFilterStream.clearBuffer();
        }
        if (m_mainFilterStream!=null) {
            m_mainFilterStream.clearBuffer();
        }
    }



    /**
     * Clears contents of all of this task's capture buffers and
     * resets references to <i>null</i>. Called by the
     * <span class="src">perform</span> method to cleanup after
     * execution (successful or not). If you override this method
     * you <em>must</em> call this method from your subclassed
     * method.
     **/
    protected void uninstallBuffers()
    {
        sysOut = null;
        sysErr = null;

        if (m_allBuffer!=null) {
            m_allBuffer.delete(0,m_allBuffer.length());
            m_allBuffer = null;
        }
        if (m_secondFilterStream!=null) {
            m_secondFilterStream.clearBuffer();
            m_secondFilterStream = null;
        }
        if (m_mainFilterStream!=null) {
            m_mainFilterStream.clearBuffer();
            m_mainFilterStream = null;
        }
    }



    /**
     * Template method to start capturing messages written to both
     * system streams to this task's backup buffers and calls nested
     * tasks. Your subclass is more likely to need to customize one
     * of the methods called by this template instead of the template
     * itself.
     * @throws SecurityException if enclosing context does not have
     *         permission to swap in/out the required streams.
     * @throws BuildException if the underlying (un)capture mechanism
     *         does.
     * @see #startCapturing
     * @see #stopCapturing stopCapturing()
     * @see #performTheEnclosedTaskList
     * @see #uninstallBuffers
     **/
    protected void performNestedTasks()
    {
        Throwable uncaught=null;
        try {
            startCapturing();
            performTheEnclosedTaskList();
        } catch(Throwable anyX) {
            uncaught = anyX;
            if (anyX instanceof RuntimeException) {
                throw (RuntimeException)anyX;
            }
            throw (Error)anyX;
        } finally {
            stopCapturing(uncaught);
            uninstallBuffers();
        }
    }



    /**
     * Called to execute this taskset's nested task from the wrapped
     * {@linkplain #performNestedTasks} method. This method is a convenient
     * extension point for subclasses and other tasks that want to capture
     * stdio information without overriding the standard way to do this.
     * @since JWare/AntX 0.4
     **/
    protected void performTheEnclosedTaskList()
    {
        performTheTasksList();
    }



    /**
     * Template method called to start capturing the various
     * <span class="src">System</span> streams. This method is called
     * just before the nested tasks are performed. The default
     * implementation replaces the current System streams with our own
     * decorators; the current system streams are still written to after
     * we've saved the information. Subclasses <em>must</em> call this
     * method (first) if they want to capture System output.
     * @throws SecurityException if enclosing context does not have
     *         permission to swap in/out System streams.
     * @throws BuildException if the stream-to-be-captured is not one
     *         of the standard System streams
     * @since JWare/AntX 0.4
     **/
    protected void startCapturing()
    {
        verify_(!m_netInstalled,"perform- not installed");
        verify_(!m_strmSwitched,"perform- not partially installed");

        if (shouldCapture(FeedbackSink.STDIO)) {

            System.out.flush();
            sysOut = System.out;

            System.err.flush();
            sysErr = System.err;

            PrintStream filtered;
            FeedbackSink ss = getFromStream();

            switch (ss.getIndex()) {
                case FeedbackSink.STDIO_INDEX: {
                    m_allBuffer = new StringBuffer(320);
                    m_secondFilterStream = new MimOutputStream(sysOut,m_allBuffer);
                    filtered = new PrintStream(m_secondFilterStream);
                    System.setOut(filtered);
                    //fallthru 4 err (ssmc)
                }
                case FeedbackSink.STDERR_INDEX: {
                    m_mainFilterStream = new MimOutputStream(sysErr,m_allBuffer);
                    filtered = new PrintStream(m_mainFilterStream);
                    System.setErr(filtered);
                    break;
                }
                case FeedbackSink.STDOUT_INDEX: {
                    m_mainFilterStream = new MimOutputStream(sysOut,m_allBuffer);
                    filtered = new PrintStream(m_mainFilterStream);
                    System.setOut(filtered);
                    break;
                }
                default: {
                    throw new BuildError(ss.getValue());//never!
                }
            }
            m_strmSwitched = true;

            CapturedLogs.installStdIORecorder(this, m_errHandler);

            m_netInstalled = true;
        }
    }



    /**
     * Template method called after the nestes tasks have been
     * performed but while the <span class="src">System</span>
     * streams are still being captured. No-op by default.
     * @param cause [optional] error that stopped execution
     * @since JWare/AntX 0.4
     **/
    protected void stopCapturing(Throwable cause)
    {
        if (m_netInstalled) {
            m_netInstalled = false;
            CapturedLogs.unwindStdIORecorder(m_errHandler);
        }
        if (m_strmSwitched) {//@.bug Fix for if only 1 swapped!
            System.setOut(sysOut);
            System.setErr(sysErr);
        }
    }


    /**
     * Filter that determines what information should captured
     * by this task. By default returns <i>true</i> for everything.
     * @param sel [unused] thing to be captured (non-null)
     * @since JWare/AntX 0.4
     **/
    protected boolean shouldCapture(FeedbackSink sel)
    {
        return true;
    }


    private volatile boolean m_netInstalled;
    private boolean m_strmSwitched;
    private ProblemHandler m_errHandler = new Responses.LogUsing(this);
    private StringBuffer m_allBuffer;
    private MimOutputStream m_secondFilterStream, m_mainFilterStream;
    private FeedbackSink m_streamChoice= FeedbackSink.STDIO;//err+out
    private PrintStream sysOut;
    private PrintStream sysErr;
}

/* end-of-CaptureStreamsTask.java */
