/**
 * $Id: CaptureLogsTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  org.apache.tools.ant.BuildEvent;
import  org.apache.tools.ant.BuildListener;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.starters.BuildListenerSkeleton;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * Diagnostics helper task that records all the Ant messages logged by its nested tasks.
 * Particularly useful to test scripts when combined with the &lt;assertlogged&gt;
 * evaluation task.
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;capturelogs&gt;
 *      &lt;echo level="warning" message="((A warning))"/&gt;
 *      &lt;assertlogged value="((A warning))"/&gt;
 *      &lt;echo level="verbose" message="((Verbose message))"/&gt;
 *      &lt;assertlogged value="((Verbose message))" occurances="0"/&gt;
 *      &lt;assertlogged important="no" value="((Verbose message))" occurances="1"/&gt;
 *      &lt;assertlogged important="no"&gt;
 *          &lt;string value="((A warning))"/&gt;
 *          &lt;string value="((Verbose message))"/&gt;
 *      &lt;/assertlogged&gt;
 *    &lt;/capturelogs&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded (for log writes/reads after fully configured)
 * @.group   impl,helper
 * @see      AssertLoggedTask
 * @see      CopyLoggedTask
 **/

public class CaptureLogsTask extends TaskSet implements LogsRecorder
{
    /**
     * Initializes a new CaptureLogsTask instance.
     **/
    public CaptureLogsTask()
    {
        super(AntX.capture+"CaptureLogsTask:");
    }


    /**
     * Initializes a new CV-labeled CaptureLogsTask instance.
     * @param iam CV-label (non-null)
     **/
    public CaptureLogsTask(String iam)
    {
        super(iam);
    }



    /**
     * Sets the context's important noise level threshold. Messages
     * logged at or lower than the important threshold are captured
     * in this task's <i>{@linkplain #getImportantBuffer high priority}</i>
     * buffer as well as the regular <i>{@linkplain #getAllBuffer all}</i>
     * buffer. Usually the important threshold is set at INFO or lower
     * (WARNING, ERROR, FATAL). Defaults to capturing messages at the
     * the AntX default noise level (INFO) or lower to the high priority
     * buffer. Setting to <i>null</i> will reset this task to use the
     * default.
     * @param threshold the important noise level threshold (inclusive)
     * @.safety single
     **/
    public void setImportantFrom(NoiseLevel threshold)
    {
        if (threshold==NoiseLevel.FATAL) {
            threshold = NoiseLevel.ERROR;//Pin since never logged here!
        }
        m_importantThreshold= threshold;
    }



    /**
     * Returns this recorder's important noise level threshold.
     * Returns the {@linkplain NoiseLevel#getDefault AntX default} if
     * never explicitly defined or reset.
     **/
    public NoiseLevel getImportantThreshold()
    {
        NoiseLevel nl= m_importantThreshold;
        return (nl!=null) ? nl : NoiseLevel.getDefault(getProject());
    }



    /**
     * Returns <i>true</i> if this task would consider the given
     * noise level important.
     * @param nl noise level (non-null)
     **/
    public final boolean isImportant(NoiseLevel nl)
    {
        require_(nl!=null,"isImportnt- nonzro NL");
        if (NoiseLevel.isAsBadAs(nl,getImportantThreshold())) {
            return true;
        }
        return false;
    }



    /**
     * Tells this task whether captured messages should be separated
     * with a platform-specific newline sequence.
     * @param splitEm <i>true</i> to separate messages entries
     * @since JWare/AntX 0.3
     **/
    public void setSplitEntries(boolean splitEm)
    {
        m_splitMsgs = splitEm;
    }


    /**
     * Returns <i>true</i> if this task will insert a platform-specific
     * newline sequence between each captured message. Is off by default.
     * @since JWare/AntX 0.3
     **/
    public final boolean willSplitEntries()
    {
        return m_splitMsgs;
    }



    /**
     * Tells this task to include a message's source if possible.
     * A source label is not included by default.
     * @since JWare/AntX 0.4
     **/
    public void setIncludeSource(boolean includeEm)
    {
        m_addMsgSource = includeEm;
    }


    /**
     * Returns <i>true</i> if this task will include source
     * information (like task or target name) for each captured
     * message. Is off by default.
     * @since JWare/AntX 0.4
     **/
    public final boolean willIncludeSource()
    {
        return m_addMsgSource;
    }



    /**
     * Returns a <em>copy</em> of current high priority buffer's
     * contents. The messages captured to this buffer are determined
     * by this task's importance threshold.
     * @see #setImportantFrom
     **/
    public String copyOfImportantLogs()
    {
        return getImportantBuffer().substring(0);
    }



    /**
     * Returns a <em>copy</em> of all contents captured by this
     * task so far. The returned string contains all levels of messages
     * from DEBUG to FATAL since this task was executed or last reset.
     * @see #copyOfImportantLogs
     **/
    public String copyOfAllLogs()
    {
        return getAllBuffer().substring(0);
    }



    /**
     * Resets this task's logs as iff no event ever recorded.
     **/
    public void clearLogs()
    {
        resetBuffers();
    }



    /**
     * Returns this task's high priority messages buffer. Never
     * returns <i>null</i>.
     * @see #setImportantFrom
     **/
    protected final StringBuffer getImportantBuffer()
    {
        return m_boobooBuffer;
    }



    /**
     * Returns this task's all messages buffer. Never returns
     * <i>null</i>.
     **/
    protected final StringBuffer getAllBuffer()
    {
        return m_allBuffer;
    }



    /**
     * Clears contents of all of this task's capture buffer.
     **/
    protected final void resetBuffers()
    {
        getAllBuffer().delete(0,getAllBuffer().length());
        getImportantBuffer().delete(0,getImportantBuffer().length());
    }



    /**
     * Starts capturing logged messages to this task's buffers.
     **/
    protected void performNestedTasks()
    {
        Project P = getProject();
        try {
            verify_(!m_isInstalled,"perform- not installed");
            verify_(!P.getBuildListeners().contains(getBuildListener()),"perform- not installed");
            P.addBuildListener(getBuildListener());
            m_isInstalled = true;
            CapturedLogs.installRecorder(this,m_errHandler);
            performTheTasksList();
        } finally {
            if (m_isInstalled) {
                m_isInstalled = false;
                P.removeBuildListener(getBuildListener());
                CapturedLogs.unwindRecorder(m_errHandler);
            }
            resetBuffers();
        }
    }



    /**
     * Returns the Ant BuildListener implementation for this task.
     **/
    protected final BuildListener getBuildListener()
    {
        return m_buildListener;
    }



    private volatile boolean m_isInstalled;
    private NoiseLevel m_importantThreshold;//NB:determined live
    private boolean m_splitMsgs;//NB:nope
    private boolean m_addMsgSource;//NB:nope
    private final Recorder m_buildListener = new Recorder();
    private final StringBuffer m_allBuffer = new StringBuffer(320);
    private final StringBuffer m_boobooBuffer = new StringBuffer(150);
    private final ProblemHandler m_errHandler= new ProblemHandler() {
            public void problem(Object nugget, int nl) {
                CaptureLogsTask.this.log(String.valueOf(nugget),nl);
            }
        };


    /**
     * Helper that captures all logged messages to enclosing
     * CaptureLogTask's buffers.
     * @since   JWare/AntX 0.2
     * @author  ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version 0.5
     **/
    private class Recorder extends BuildListenerSkeleton {
        /**
         * Captures logged information to enclosing task's buffers. Implementation
         * Note: We depend on (current) fact that more important noise level indices
         * are lower that less important ones. If this ever changes we must update
         * this code (ssmc).
         **/
        public void messageLogged(BuildEvent e) {
            String msg = e.getMessage();
            if (willIncludeSource()) {
                msg = LocalTk.purtyEventMsg(e);
            }
            addMsg(getAllBuffer(),msg);
            if (NoiseLevel.isAsBadAs(e.getPriority(),getImportantThreshold())) {
                addMsg(getImportantBuffer(),msg);
            }
        }
        /** Ensures messages written with trailing newline if needed. **/
        private void addMsg(StringBuffer sb, String msg)
        {
            sb.append(msg);
            if (willSplitEntries()) {
                sb.append(Strings.NL);
            }
        }
    }
}

/* end-of-CaptureLogsTask.java */
