/**
 * $Id: Requester.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.apis;

import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.ProjectComponent;
import  org.apache.tools.ant.Target;
import  org.apache.tools.ant.Task;

/**
 * Minimal interaction interface between a generic service and a user of that service.
 * In this context a "service" is any object that can handle requests from objects
 * other than itself. This interface allows generic utilities to easily link feedback
 * and diagnostics trails back to individual users.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   n/a
 * @.group    impl,helper
 **/

public interface Requester extends ProblemHandler, LogEnabled
{
    /**
     * Returns a publishable name for the user. Often used in
     * feedback and/or error messages.
     * @return name (non-null)
     */
    String getName();



    /**
     * Returns a usable location for the user. Used solely in
     * signal generation to link the service user to problems.
     **/
    Location getLocation();



    /**
     * Returns the project associated with caller. Can return
     * <i>null</i> if caller is independent of any particular
     * project (like with utilities).
     * @return associated project or <i>null</i>
     **/
    Project getProject();



    /**
     * Returns the (original) script target or <i>null</i> if
     * this information unknown.
     **/
    Target getSourceTarget();



    /**
     * An anonymous <em>totally silent</em> service requester proxy.
     * Unlike an {@linkplain Anonymous} adapter, this requester will
     * <em>not</em> send messages or indicate problems to the system
     * consoles.
     **/
    public static final Requester DEVNULL= new Requester()
    { 
        /** 
         * Always returns <span class="src">undefined</span>. 
         **/
        public String getName() {
            return "undefined";
        }
        /** 
         * Always returns <span class="src">UNKNOWN_LOCATION</span>.
         **/
        public Location getLocation() {
            return Location.UNKNOWN_LOCATION;
        }
        /**
         * Always returns <i>null</i>.
         **/
        public Project getProject() {
            return null;
        }
        /**
         * Always returns <i>null</i> for no source target.
         **/
        public Target getSourceTarget() {
            return null;
        }
        /** 
         * Does nothing.
         **/
        public void log(String message) {
        }
        /** 
         * Does nothing.
         **/
        public void log(String message, int msgLevel) {
        }
        /** 
         * Does nothing.
         **/
        public void problem(Object nugget, int nl) {
        }
    };




    /**
     * An anonymous service requester proxy. Unlike the standard 
     * {@linkplain #DEVNULL null proxy} adapter, anonymous requesters
     * can specify both a location and an associated project. This is
     * often the available state from standalone utility methods.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
    public static final class Anonymous extends LogEnabled.ForProject
        implements Requester
    {
        private final Location m_loc;

        /**
         * Initializes a new anonymous requester.
         **/
        public Anonymous() {
            super(null);
            m_loc = Location.UNKNOWN_LOCATION;
        }
        /**
         * Initializes a new location-bound anonymous requester.
         * @param loc caller's location (non-null)
         **/
        public Anonymous(Location loc) {
            super(null);
            m_loc = loc==null ? Location.UNKNOWN_LOCATION : loc;
        }
        /**
         * Initializes a new location-bound anonymous requester.
         * @param project caller's associated project (non-null)
         * @param loc caller's location (non-null)
         **/
        public Anonymous(Project project, Location loc) {
            super(project);
            m_loc = loc==null ? Location.UNKNOWN_LOCATION : loc;
        }
        /** 
         * Always returns <span class="src">undefined</span>. 
         **/
        public String getName() {
            return "undefined";
        }
        /** 
         * Always returns either <span class="src">UNKNOWN_LOCATION</span>
         * or a proxy location bound at creation time.
         **/
        public Location getLocation() {
            return m_loc;
        }
        /**
         * Always returns <i>null<i> or the project bound at
         * creation time.
         **/
        public Project getProject() {
            return getImpl();
        }
        /**
         * Always returns <i>null</i> for no owning target.
         **/
        public Target getSourceTarget() {
            return null;
        }
        /** 
         * Records problem using our own log API.
         **/
        public void problem(Object nugget, int nl) {
            log(Tk.stringFrom(nugget,null),nl);
        }
    }



    /**
     * An anonymous service requester that logs to system consoles.
     **/
    public static final Requester ANONYMOUS= new Anonymous();



    /**
     * Default {@linkplain Requester} adapter for a standard Ant component.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
    public static class ForComponent extends LogEnabled.ForComponent
        implements Requester
    {
        /**
         * Initializes a new requester adapter for given component.
         * @param pc the component being wrapped (non-null)
         **/
        public ForComponent(ProjectComponent pc) {
            super(pc);
        }
        /**
         * Returns component's script-supplied name if possible or 
         * the item's class (leaf) name.
         **/
        public String getName() {
            if (m_PC instanceof Task) {
                return ((Task)m_PC).getTaskName();
            } 
            if (m_PC instanceof Named) {
                String n = ((Named)m_PC).getName();
                if (n!=null) {
                    return n;
                }
            }
            return Tk.leafNameFrom(m_PC.getClass());

        }
        /**
         * Returns components assigned parse location or the 
         * <span class="src">UNKNOWN</span> marker if no location
         * unknown or not defined.
         * @see ScriptLocatable
         **/
        public Location getLocation() {
            if (m_PC instanceof Task) {
                return ((Task)m_PC).getLocation();
            }
            if (m_PC instanceof ScriptLocatable) {
                return ((ScriptLocatable)m_PC).getLocation();
            }
            return Location.UNKNOWN_LOCATION;
        }
        /**
         * Returns compoent's owning target or <i>null</i> if
         * no such target.
         **/
        public Target getSourceTarget() {
            if (m_PC instanceof Task) {
                return ((Task)m_PC).getOwningTarget();
            }
            return null;
        }
        /**
         * Always returns task's associated project.
         **/
        public Project getProject() {
            return m_PC.getProject();
        }
        /** 
         * Records problem using our own log API.
         **/
        public void problem(Object nugget, int nl) {
            log(Tk.stringFrom(nugget,null),nl);
        }
    }
    



    /**
     * Default {@linkplain Requester} adapter for a utility object.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
    public static final class ForUtility extends LogEnabled.ForUtility 
        implements Requester
    {
        /**
         * Initializes a new adapter for given utility.
         * @param utility the utility being wrapped (non-null)
         */
        public ForUtility(ProjectDependent utility) {
            super(utility);
        }
        /** 
         * Always returns leafname of utility's class.
         **/
        public String getName() {
            return Tk.leafNameFrom(getImpl().getClass());
        }
        /** 
         * Always returns <span class="src">UNKNOWN_LOCATION</span>
         * unless the utility implements {@linkplain ScriptLocatable}.
         **/
        public Location getLocation() {
            if (getImpl() instanceof ScriptLocatable) {
                return ((ScriptLocatable)getImpl()).getLocation();
            }
            return Location.UNKNOWN_LOCATION;
        }
        /**
         * Always returns utility's associated project.
         **/
        public Project getProject() {
            return getImpl().getProject();
        }
        /**
         * Always returns <i>null</i> for no owning target.
         **/
        public Target getSourceTarget() {
            return null;
        }
        /** 
         * Records problem using our own log API.
         **/
        public void problem(Object nugget, int nl) {
            log(Tk.stringFrom(nugget,null),nl);
        }
    }



    /**
     * Default {@linkplain Requester} adapter for a utility object.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
    public static final class ForProject extends LogEnabled.ForProject
        implements Requester
    {
        /**
         * Initializes a new adapter for a utility method.
         * @param project the project being wrapped (non-null)
         */
        public ForProject(Project project) {
            super(project);
        }
        /** 
         * Always returns the project's name or the empty
         * string "project".
         **/
        public String getName() {
            Project P = getProject();
            if (P!=null) {
                return P.getName();
            }
            return "project";
        }
        /** 
         * Always returns <span class="src">UNKNOWN_LOCATION</span>.
         **/
        public Location getLocation() {
            return Location.UNKNOWN_LOCATION;
        }
        /**
         * Always returns associated project.
         **/
        public Project getProject() {
            return getImpl();
        }
        /**
         * Always returns <i>null</i> for no owning target.
         **/
        public Target getSourceTarget() {
            return null;
        }
        /** 
         * Records problem using our own log API.
         **/
        public void problem(Object nugget, int nl) {
            log(Tk.stringFrom(nugget,null),nl);
        }
    }




    /**
     * Default {@linkplain Requester} adapter for a compound requester.
     * A composite requester uses the local helper task's name and
     * location, but the original source's logging, problem handling, and
     * project reference.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
//    public static class ForComposite extends LogEnabled.ForComposite 
//        implements Requester
//    {
//        private final Task m_task;
//
//        /**
//         * Initializes a new requester adapter for local helper
//         * task that is part of a request processing chain.
//         * @param task the task being wrapped (non-null)
//         **/
//        public ForComposite(Task task, Requester source) {
//            super(source);
//            m_task = task;
//        }
//        /**
//         * Always returns local task's script-supplied name.
//         **/
//        public String getName() {
//            return getTaskImpl0().getTaskName();
//        }
//        /**
//         * Always returns local task's parse location.
//         **/
//        public Location getLocation() {
//            return getTaskImpl0().getLocation();
//        }
//        /**
//         * Returns the source requester. Never <i>null</i>.
//         **/
//        public Requester getRequester0() {
//            return ((Requester)getLogImpl());
//        }
//        /**
//         * Returns the source's project (not local tasks).
//         **/
//        public Project getProject() {
//            return getRequester0().getProject();
//        }
//        /** 
//         * Delegates problem notification to source.
//         **/
//        public void problem(Object nugget, int nl) {
//            getRequester0().problem(nugget,nl);
//        }
//        /**
//         * Returns the local task helper's instance.
//         **/
//        public final Task getTaskImpl0() {
//            return m_task;
//        }
//    }
}

/* end-of-Requester.java */