/**
 * $Id: UISMContext.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  com.idaremedia.apis.EmptyStringManager;
import  com.idaremedia.apis.UIStringManager;

import  com.idaremedia.antx.apis.ProblemHandler;

/**
 * Fixture administrator of iteration-based UIStringManagers for the active thread.
 * The {@linkplain AssertableTask} base task uses a UISMContext to implements its
 * resource bundle-based message support instead of directly accessing the
 * FixtureOverlays stacks.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,infra
 **/

public final class UISMContext implements FixtureCore, FixtureAdministrator
{
    /**
     * The fixture component id for all strings manager information.
     **/
    public static final String FXID= FixtureIds.MSGS_STRINGMANAGER;



    /**
     * Returns the current thread's nearest UIStringManager instance if 
     * one exists. Returns <i>null</i> if no string manager installed for
     * executing thread.
     **/
    public static UIStringManager getStringManager()
    {
        return (UIStringManager)FixtureOverlays.getContextInstance().nearest(FXID);
    }



    /**
     * Helper to {@linkplain #getStringManager getStringManager} that returns
     * the {@linkplain #getDefaultStringManager default string manager} if the
     * current thread has no explicit configuration. Never returns <i>null</i>.
     **/
    public static UIStringManager getStringManagerNoNull()
    {
        UIStringManager sm = getStringManager();
        return (sm!=null) ? sm : getDefaultStringManager();
    }



    /**
     * Installs a new UIStringManager instance for the current thread. This
     * string manager becomes the active (and only visible) string manager
     * until it is unwound or another string manager is installed.
     * @param sm the new string manager (non-null)
     * @param noInstallHandler [optional] used to notify if cannot install
     * @return previous string manager if any (can be <i>null</i>)
     * @throws BuildException if incoming manager already on iteration stack
     **/
    public static UIStringManager installStringManager(UIStringManager sm,
                                                  ProblemHandler noInstallHandler)
    {
        if (sm==null) {
            throw new IllegalArgumentException
                (AntX.uistrs().get("task.uism.null.sm"));
        }

        String whoami = AntX.uistrs().dget("task.uism.whoami", "UIStringManager");
        return (UIStringManager)FixtureOverlays.installIfNot
                        (FXID, sm, noInstallHandler, whoami);
    }



    /**
     * Removes the most recently installed UIStringManager for the current
     * thread. The previous installation is reactivated, or if this was
     * the only string manager, the current thread's string manager is
     * undefined (should use default if necessary).
     * @param noUninstallHandler [optional] used if unable to uninstall
     **/
    public static void unwindStringManager(ProblemHandler noUninstallHandler)
    {
        String whoami = AntX.uistrs().dget("task.uism.whoami","UIStringManager");
        FixtureOverlays.uninstallIfIs(FXID, noUninstallHandler, whoami);
    }



    /**
     * Returns the default UIStringManager used by this context. Never 
     * returns <i>null</i>.
     * @see #setDefaultStringManager
     **/
    public static final UIStringManager getDefaultStringManager()
    {
        UIStringManager sm = (UIStringManager)Iteration.getProperty(FXID);
        return (sm==null) ? EmptyStringManager.INSTANCE : sm;
    }



    /**
     * Initializes the default UIStringManager returned by this context. Not
     * necessary since by default, is set to an EmptyStringManager. Should not
     * be set to a string manager that exists on a thread's UISM stack. This
     * method <em>replaces</em> any pre-existing default string manager-- the
     * current setting is lost unless the caller maintains it.
     * @param sm the new default string manager (non-null)
     * @.safety guarded
     **/
    public static final UIStringManager setDefaultStringManager(UIStringManager sm)
    {
        if (sm==null) {
            throw new IllegalArgumentException
                (AntX.uistrs().get("task.uism.null.sm"));
        }
        return (UIStringManager)Iteration.setProperty(FXID,sm);
    }



    /**
     * Returns <i>true</i> if the default UIStringManager is defined as
     * the empty placeholder.
     * @see #setDefaultStringManager setDefaultStringManager()
     * @since JWare/AntX 0.4
     **/
    public static final boolean isDefaultUndefined()
    {
        return getDefaultStringManager()==EmptyStringManager.INSTANCE;
    }



    /**
     * Resets the default UIStringManager to the empty placeholder. Your
     * scripts and application should never need to do this. This method is
     * provided for test scripts and harnesses.
     * @.safety guarded
     * @since JWare/AntX 0.4
     **/
    static final void unsetDefaultStringManager()
    {
        Iteration.removeProperty(FXID);
    }




    /**
     * Installs a test-harness helper that clears up the various UISMContext
     * fixture components. Your application should never use this helper
     * directly; it is provided so that test harnesses can reset the environment
     * to a known state.
     * @since JWare/AntX 0.4
     **/
    static {
        AntXFixture.setKillMethod
            (FXID,
             new String[]{"messagebundles","stringmanagers"},
             new KillMethodSkeleton() {
                     protected boolean killDefault(ProblemHandler from) {
                         UISMContext.unsetDefaultStringManager();
                         return true;
                     }
                     protected String getComponentId() {
                         return UISMContext.FXID;
                     }
                     protected String getComponentName() {
                         return AntX.uistrs().dget
                             ("task.uism.whoami", "UIStringManager");
                     }
                 }
             );
    }



    /** Disallow; only public static utility methods. **/
    private UISMContext()
    { }
}

/* end-of-UISMContext.java */
