/**
 * $Id: Stringifier.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.io.File;
import  java.net.URL;
import  java.util.Date;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.EnumeratedAttribute;
import  org.apache.tools.ant.types.Environment;
import  org.apache.tools.ant.types.Path;
import  org.apache.tools.ant.types.RegularExpression;
import  org.apache.tools.ant.util.regexp.RegexpMatcher;

import  com.idaremedia.antx.apis.FlexStringFriendly;
import  com.idaremedia.antx.helpers.DateTimeFormat;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Base utility for converting fixture components to their string form. This default
 * stringifiers understand a subset of standard Ant elements (listed below) and any AntX
 * component that implements the {@linkplain FlexStringFriendly} interface. AntX 
 * feedback tasks like &lt;print&gt; can fall back to Stringifiers if needed.
 * <p/>
 * A <i>lenient</i> stringifier is one that tries to make an intelligent decision about
 * how to stringify an object but if all fails will just default to whatever 
 * <span class="src">Object.toString</span> returns. A strict stringifier will return
 * <i>null</i> if it does not understand how to convert an object to a
 * {@linkplain FlexString} friendly value.
 * <p>
 * <b>Recognized Types:</b><ul>
 *   <li>{@linkplain FlexStringFriendly}</li>
 *   <li>InnerStrings</li>
 *   <li>String</li>
 *   <li>Boolean (converted to "true" or "false")</li>
 *   <li>Dates (converted to GMT string)</li>
 *   <li>URLs</li>
 *   <li>Files</li>
 *   <li>RegularExpressions (converted to underlying pattern)</li>
 *   <li>Paths</li>
 *   <li>Numbers</li>
 *   <li>Throwable (converted to throwable's message)</li>
 *   <li>Class (converted to class's fully qualified name)</li>
 *   <li>EnumeratedAttribute</li>
 *   <li>Environment.Variable</li>
 * </ul>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   api,helper
 * @see      FlexStringFriendly
 **/

public class Stringifier implements FixtureCore
{
    /**
     * Pseudo factory method that returns the one of the current
     * iteration's two class stringifier singletons. Never returns
     * <i>null</i>.
     * @param lenient <i>true</i> to return the lenient stringifier
     * @see Iteration#lenientStringifer
     * @see Iteration#strictStringifier
     **/
    public static final Stringifier get(boolean lenient)
    {
        return lenient ? Iteration.lenientStringifer() 
            : Iteration.strictStringifier();
    }


    /**
     * Initializes a new lenient stringifier.
     **/
    public Stringifier()
    {
        m_isLenient = true;
    }


    /**
     * Initializes a new stringifier.
     * @param lenient <i>true</i> if this stringifier should
     *                do lenient stringification(tm)
     **/
    public Stringifier(boolean lenient)
    {
        m_isLenient = lenient;
    }


    /**
     * Returns <i>true</i> if this stringifier will tolerate any
     * type of object, defaulting to the standard 'toString()'
     * method to stringify it. Defined when this stringifier is
     * created.
     **/
    public final boolean isLenient()
    {
        return m_isLenient;
    }


    /**
     * Tries to convert a generic reference object to a flex value
     * friendly string. If the project is unspecified and this stringifier
     * needs a project reference, this method will throw an
     * IllegalArgumentException.
     * @param o the thing to be stringified
     * @param project [optional] project from which information read
     *                if necessary (unused by default)
     * @throws IllegalArgumentException if project <i>null</i> and
     *               is required
     **/
    public String stringFrom(Object o, Project project)
    {
        if (o==null) {
            return null;
        }
        if (o instanceof FlexStringFriendly) {
            return ((FlexStringFriendly)o).stringFrom(project);
        }

        //Implementation Notes:
        //   => ugly (and slow) but works...
        //   => to extend this if ever necessary need to customize
        //      the stringifier used by the Iteration object...
        //
        if (o instanceof String || o instanceof Boolean) {
            return o.toString();
        }
        if (o instanceof InnerString) {
            return ((InnerString)o).toString(project);
        }
        if (o instanceof Throwable) {
            return ((Throwable)o).getMessage();
        }
        if (o instanceof RegularExpression) {
            return ((RegularExpression)o).getPattern(project);
        }
        if (o instanceof RegexpMatcher) {
            return ((RegexpMatcher)o).getPattern();
        }
        if (o instanceof Path) {
            return ((Path)o).toString();
        }
        if (o instanceof File || o instanceof URL || o instanceof Number) {
            return o.toString();
        }
        if (o instanceof Date) {
            return DateTimeFormat.GMTformat(((Date)o).getTime());
        }
        if (o instanceof EnumeratedAttribute) {
            return ((EnumeratedAttribute)o).getValue();
        }
        if (o instanceof Environment.Variable) {
            return ((Environment.Variable)o).getValue();
        }
        if (o instanceof Class) {
            return ((Class)o).getName();
        }

        if (!isLenient()) {
            return null;//abandon-ship!
        }

        //NB: Ant's default types aren't very 'toString'-friendly
        //    but let's assume that caller knows what they're doing
        return Tk.stringFrom(o,project);
    }


    private final boolean m_isLenient;
}

/* end-of-Stringifier.java */
