/**
 * $Id: Iteration.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.lang.reflect.Field;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.apis.UIStringManager;

import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.InstanceFactory;
import  com.idaremedia.antx.helpers.SIDs;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Embodies the configured fixture of an execution iteration. In other words, a
 * glorified "holder of stuff".
 * <p/>
 * It is important that an Iteration be "self-creating" so we can function in
 * environments where AntX is not part of a single controlling application or tool or
 * when AntX is not started explicitly by some initialization task. To support this
 * automagical creation for Iteration subclasses, we rely on the standard Java service
 * lookup pattern: we first look for a class name stored under the system property
 * <span class="src">com.idaremedia.antx.Iteration</span>. If no such property exists,
 * we look for a class definition in a file resource stored at
 * <span class="src">META-INF/services/com.idaremedia.antx.Iteration</span>.
 * <p/>
 * If used from an always-on or iteration assembly line service (like an Ant console
 * or remote Ant server), it is that launcher's responsibility to synchronize the
 * re-creation of the active AntX iteration instance. The idea is that calls to the
 * mutative admin interface (<span class="src">{@linkplain #set set}</span> and
 * <span class="src">{@linkplain #reset}</span>) will be restricted (and synchronized)
 * by the launch mechanism. Typically the "new iteration" sequence would first reset
 * the existing global fixture components using <span class="src">AntXFixture.reset</span>
 * and then reset the Iteration using <span class="src">Iteration.reset</span>.
 * <p/>
 * AntX-based tools can extend the standard Iteration to both add new fixture elements
 * and replace the default standard ones. For instance, a tool could make the default
 * {@linkplain Stringifier} something a client could specify instead of the default
 * instance defined by AntX.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   guarded
 * @.group    api,infra
 * @see       com.idaremedia.antx.starters.IterationConfigurator IterationConfigurator
 **/

public class Iteration
{
//  ---------------------------------------------------------------------------------------
//  Administation API for AntX-based tool or application
//  ---------------------------------------------------------------------------------------

    private static final Map IDs = AntXFixture.newMap();//*all* IDs ever used


    /**
     * Factory method for creating a new Iteration instance. We try to use
     * a service-provider-isque way of looking for an application specific
     * Iteration class. AntX-based applications can create a
     * <span class="src">META-INF/services/com.idaremedia.antx.Iteration</span>
     * entry in their distribution jars. Script writers can define a system
     * property "<span class="src">com.idaremedia.antx.Iteration</span>" to
     * locally override whatever's in their classpaths.
     * @return a new iteration instance (never <i>null</i>)
     * @throws BuildException if a custom class was specified but it could
     *   not be loaded as an Iteration.
     **/
    private static Iteration newi()
    {
        String classname = Tk.loadSpiImpl(Iteration.class,Requester.ANONYMOUS);
        if (classname!=null) {
            try {
                Class c = Class.forName(classname);
                return (Iteration)c.newInstance();
            } catch(Exception anyX) {
                String error = AntX.uistrs().get("task.bad.custimpl.class1",
                    classname, Iteration.class.getName());
                Requester.ANONYMOUS.log(error, Project.MSG_ERR);
                throw new BuildException(error,anyX);
            }
        }
        return new Iteration();
    }



    /**
     * Global Iteration instance (only one active at a time).
     * <b>!!! Location of this static variable is important !!!</b>
     */
    private static Iteration INSTANCE = newi();//NB:be valid always!



    /**
     * Returns the current iteration object. Never returns <i>null</i>.
     **/
    public static final Iteration get()
    {
        return INSTANCE;
    }



    /**
     * Initializes the (next) iteration to a specific subclass instance.
     * This method should be used by harnesses and "always-on" tools to
     * create a fresh (custom) iteration. <em>Information from the current
     * iteration is discarded.</em>
     * @param iteration the next iteration (non-null)
     * @throws IllegalArgumentException if iteration is <i>null</i>
     */
    public static final void set(Iteration iteration)
    {
        if (iteration==null) {
            throw new IllegalArgumentException("iteration reference");
        }
        INSTANCE = iteration;
    }



    /**
     * Initializes the (next) iteration to a new standard iteration
     * instance. This method should be used by harnesses and "always-on"
     * tools to create a fresh iteration. <em>Information from the current
     * iteration is discarded.</em>
     */
    public static final void reset()
    {
        INSTANCE = newi();
    }

//  ---------------------------------------------------------------------------------------
//  Diagnostics API for Iterations (see valueURIs and Conditions)
//  ---------------------------------------------------------------------------------------

    /**
     * Returns a diagnostic string that represents the information
     * indicated by incoming URI. Will return <i>null</i> if unable
     * to understand incoming URI. Currently understands: "id",
     * "classname", "valueuri?<i>scheme-name</i>", and 
     * "property?<i>property-name</i>".
     * @param uri information to be described (non-null)
     * @return description or <i>null</i> if URI not understood.
     * @throws IllegalArgumentException if URI is <i>null</i>.
     * @since JWare/AntX 0.5
     * @.impl Helps debug Iteration-related issues from scripts.
     **/
    public String toString(String uri)
    {
        if (uri==null) {
            throw new IllegalArgumentException();
        }
        if ("classname".equals(uri)) {
            return getClass().getName(); 
        }
        if ("id".equals(uri)) {
           return getITID();
        }
        if (uri.startsWith("property?")) {
            String property = uri.substring(9/*"property?".length*/);
            return FixtureExaminer.getIterationProperty(property,null);
        }
        if (uri.startsWith("valueuri?")) {
            String urischeme = uri.substring(9/*"valueuri?".length*/);
            if (FixtureExaminer.isBuiltinValueURIScheme(urischeme)) {
                return "builtin";
            }
            Class c = valueURIHandler(urischeme).getClass();
            return (c==ValueURIHandler.None.class ? "none" : c.getName());
        }
        return null;
    }


//  ---------------------------------------------------------------------------------------
//  "Subclassable" instance API and matching singleton API
//  ---------------------------------------------------------------------------------------

    /**
     * Initialize a new Iteration instance. Default implementation
     * has few member fields (relies mostly on existing, common
     * utilities and/or singletons).
     **/
    protected Iteration()
    {
        String sId = null;
        synchronized(IDs) {
            sId = SIDs.next(new SIDs.Finder() {
                public boolean exists(String id) {
                    if (!IDs.containsKey(id)) {
                        IDs.put(id,Boolean.TRUE);
                        return false;
                    }
                    return true;
                }
            });
        }
        m_itid = "Iteration"+sId;
        m_refTable.put("Iteration.ITID",m_itid);
        m_valueUriFactory.put("null",ValueURIHandler.None.class);
        m_testField=sId; // @.impl Used for testing injection(do not remove)
    }



    /**
     * Returns this iteration's unique identifier. This id is generated
     * when each iteration is created and it immutable thereafter. Each
     * id is unique to a particular VM instance.
     * @since JWare/AntX 0.5
     **/
    public final String getITID()
    {
        return m_itid;
    }



    /**
     * Returns this iteration's configuration lock. Must never
     * return <i>null</i> although can return different locks for
     * different configuration aspects.
     * @param aspect [optional] purpose of lock
     * @return lock (non-null)
     **/
    public Object getConfigLock(String aspect)
    {
        return m_configLock;
    }



    /**
     * Returns the <em>current</em> iteration's configuration lock. Use
     * this lock to coordinate iteration configuraton via reflection.
     * @param aspect [optional] purpose of lock
     * @return lock (non-null)
     **/
    public static final Object lock(String aspect)
    {
        return Iteration.get().getConfigLock(aspect);
    }



    /**
     * Returns this iteration's application's builtin strings.
     * Never returns <i>null</i>; returns {@linkplain AntX#uistrs}
     * by default.
     **/
    protected UIStringManager getUIStrs()
    {
        return AntX.uistrs();
    }



    /**
     * Returns the <em>current</em> iteration's builtin strings.
     * Never returns <i>null</i>.
     * @see AntX#uistrs
     **/
    public static final UIStringManager uistrs()
    {
        return Iteration.get().getUIStrs();
    }



    /**
     * Returns this iteration's application's configuration
     * properties prefix. Never returns <i>null</i>; returns
     * {@linkplain AntX#ANTX_CONFIG_ID} by default.
     **/
    protected String getConfigId()
    {
        return AntX.ANTX_CONFIG_ID;
    }



    /**
     * Returns the current application's configuration
     * properties prefix. Never returns <i>null</i>.
     **/
    public static final String configId()
    {
        return Iteration.get().getConfigId();
    }



    /**
     * Returns this iteration's property value defaults. Never
     * returns <i>null</i>.
     **/
    protected Defaults getDefaults()
    {
        return m_defaultsINSTANCE;
    }



    /**
     * Returns the current iteration's property value defaults.
     * Never returns <i>null</i>.
     **/
    public static final Defaults defaultdefaults()
    {
        return Iteration.get().getDefaults();
    }



    /**
     * Returns this iteration's default set of fixture exclusions. 
     * Never returns <i>null</i>.
     **/
    protected ExcludedFixture getFixtureExcludes()
    {
        return m_xcludedFixtureINSTANCE;
    }



    /**
     * Returns the current iteration's default fixture exclusions.
     * Never returns <i>null</i>.
     **/
    public static final ExcludedFixture defaultFixtureExcludes()
    {
        return Iteration.get().getFixtureExcludes();
    }



    /**
     * Returns the lenient Stringifier for this iteration. Never
     * returns <i>null</i>.
     **/
    protected Stringifier getLenientStringifier()
    {
        return m_stringifierINSTANCE;
    }



    /**
     * Returns the lenient Stringifier for the <em>current</em>
     * iteration. Never returns <i>null</i>.
     **/
    public static final Stringifier lenientStringifer()
    {
        return Iteration.get().getLenientStringifier();
    }



    /**
     * Returns the strict Stringifier for this iteration. Never
     * returns <i>null</i>.
     **/
    protected Stringifier getStrictStringifier()
    {
        return m_strictStringifierINSTANCE;
    }



    /**
     * Returns the strict Stringifier for the <em>current</em>
     * iteration. Never returns <i>null</i>.
     **/
    public static final Stringifier strictStringifier()
    {
        return Iteration.get().getStrictStringifier();
    }



    /**
     * Returns the collection of exportable, modifiable properties
     * for this iteration. Never returns <i>null</i>.
     **/
    protected ExportedProperties getExportableProperties()
    {
        return m_exportedPropertiesINSTANCE;
    }



    /**
     * Returns the collection of exportable, modifiable properties
     * for the <em>current</em> iteration. Never returns <i>null</i>.
     **/
    public static final ExportedProperties exportableProperties()
    {
        return Iteration.get().getExportableProperties();
    }



    /**
     * Returns the collection of <em>all</em> fixture overlays 
     * linked to the curent iteration. The returned value contains
     * all thread-local overlay stacks.
     * @.impl Used by the FixtureOverlays helper only. We should
     *        use that class's APIs in our own getters.
     **/
    static FixtureOverlays.Handle fixtureOverlays()
    {
        return Iteration.get().m_fixtureOverlays; 
    }



    /**
     * Returns the last installed overlay in this iteration
     * for the named category. Will return <i>null</i> if no such
     * item exists.
     * @param FXID the fixture category identifier (non-null)
     * @return last installed overlay or <i>null</i> if none
     **/
    protected Object getNearestOverlayFor(String FXID)
    {
        return FixtureOverlays.getContextInstance().nearest(FXID);
    }



    /**
     * Returns the last installed overlay in the <em>current</em>
     * iteration for the named category. Will return <i>null</i> if
     * no such component exists.
     * @param FXID the fixture category identifier (non-null)
     * @return last installed overlay or <i>null</i> if none
     **/
    public static final Object nearestOverlayFor(String FXID)
    {
        return Iteration.get().getNearestOverlayFor(FXID);
    }



    /**
     * Returns the string manager most recently installed as the
     * default for the iteration. Will return the default string
     * manager only if no other string manager is installed. Never
     * returns <i>null</i>.
     **/
    protected UIStringManager getStringManager()
    {
        return UISMContext.getStringManagerNoNull();
    }



    /**
     * Returns the <em>current</em> iteration's nearest string
     * manager. Never returns <i>null</i>.
     **/
    public static final UIStringManager stringManager()
    {
        return Iteration.get().getStringManager();
    }



    /**
     * Returns the current default problem handler in this thread.
     * Never returns <i>null</i>.
     **/
    protected ProblemHandler getProblemsHandler()
    {
        return Problems.getDefaultHandlerNoNull();
    }



    /**
     * Returns the <em>current</em> iteration's default problem
     * handler. Never returns <i>null</i>.
     **/
    public static final ProblemHandler problemsHandler()
    {
        return Iteration.get().getProblemsHandler();
    }

//  ---------------------------------------------------------------------------------------
//  Value URI Management:
//  ---------------------------------------------------------------------------------------

    /**
     * Returns a value URI handler for given prefix. Never returns
     * <i>null</i> but can return a no-op proxy handler. Note that the 
     * incoming prefix name should <em>not</em> include the leading '$'
     * or the trailing ':'.
     * @param prefix value uri's marker prefix (like "alias" or "baseurl");
     * @return a handler (never <i>null</i>)
     * @throws IllegalArgumentException if parameter is null.
     **/
    public static ValueURIHandler valueURIHandler(String prefix)
    {
        if (prefix==null) {
            throw new IllegalArgumentException();
        }
        return (ValueURIHandler)
            Iteration.get().m_valueUriFactory.newInstance(prefix);
    }



    /**
     * Installs a new value URI handler into the current iteration iff
     * there isn't a handler already installed. Note that the incoming prefix
     * name should <em>not</em> include the leading '$' or the trailing ':'.
     * @param prefix value uri's marker prefix (like "alias" or "baseurl");
     * @param handlerClass the handler's class (non-null)
     * @throws IllegalArgumentException if either parameter is null.
     * @return <i>true</i> if installation went ok.
     **/
    public static boolean initValueURIHandler(String prefix, Class handlerClass)
    {
        if (prefix==null || handlerClass==null) {
            throw new IllegalArgumentException();
        }
        InstanceFactory factory = Iteration.get().m_valueUriFactory;
        synchronized(factory) {
            if (!factory.containsKey(prefix)) {
                factory.put(prefix,handlerClass);
                return true;
            }
        }
        return false;
    }



    /**
     * Installs a value URI handler into the current iteration replacing
     * any existing handler unconditionally. Note that the incoming prefix
     * name should <em>not</em> include the leading '$' or the trailing ':'.
     * @param prefix value uri's marker prefix (like "alias" or "baseurl");
     * @param handlerClass the handler's class (non-null)
     * @throws IllegalArgumentException if either parameter is null.
     * @return the previous handler (useful for temporary replacement)
     **/
    public static Class replaceValueURIHandler(String prefix, Class handlerClass)
    {
        if (prefix==null || handlerClass==null) {
            throw new IllegalArgumentException();
        }
        InstanceFactory factory = Iteration.get().m_valueUriFactory;
        return (Class)factory.put(prefix,handlerClass);
    }



    /**
     * Uninstalls a temporarily assigned AntX value URI handler
     * for the current iteration. Note that the incoming prefix
     * name should <em>not</em> include the leading '$' or the trailing ':'.
     * @param prefix the marker uri prefix (like "alias" or "artifact");
     * @throws IllegalArgumentException if parameter is null.
     **/
    public static void removeValueURIHandler(String prefix)
    {
        if (prefix==null) {
            throw new IllegalArgumentException();
        }
        InstanceFactory factory = Iteration.get().m_valueUriFactory;
        factory.put(prefix,null);
    }

//  ---------------------------------------------------------------------------------------
//  Generic Reference/Property Management:
//  ---------------------------------------------------------------------------------------

    private static final String PROPERTYARG= "property name";
    private static final String CATEGORYARG= "category";

    /**
     * Returns the named iteration property or <i>null</i> if no
     * match found.
     * @param name property's name (non-null)
     * @throws IllegalArgumentException if name is <i>null</i>.
     **/
    public static Object getProperty(String name)
    {
        if (name==null) {
            throw new IllegalArgumentException(PROPERTYARG);
        }
        return Iteration.get().m_refTable.get(name);
    }



    /**
     * Returns the named iteration category property or <i>null</i> 
     * if no match found.
     * @param category the iteration category (non-null)
     * @param name property's name (non-null)
     * @throws IllegalArgumentException if name or category is <i>null</i>.
     **/
    public static Object getProperty(String category, String name)
    {
        if (category==null) {
            throw new IllegalArgumentException(CATEGORYARG);
        }
        if (name==null) {
            throw new IllegalArgumentException(PROPERTYARG);
        }
        Object item = Iteration.get().m_refTable.get(category);
        if (item instanceof Map) {
            return ((Map)item).get(name);
        }
        return null;
    }



    /**
     * Updates an iteration property's value.
     * @param name property's name (non-null)
     * @param value the property's value
     * @throws IllegalArgumentException if name is <i>null</i>.
     * @return the previously installed value (can be <i>null</i>)
     **/
    public static Object setProperty(String name, Object value)
    {
        if (name==null) {
            throw new IllegalArgumentException(PROPERTYARG);
        }
        return Iteration.get().m_refTable.put(name,value);
    }



    /**
     * Updates a named iteration category property. If the category's map
     * does not exist, a thread-safe map in created and inserted automatically.
     * @param category the iteration category (non-null)
     * @param name property's name (non-null)
     * @param value the property's value
     * @throws IllegalArgumentException if name or category is <i>null</i>.
     * @return the previously installed value (can be <i>null</i>)
     **/
    public static Object setProperty(String category, String name, Object value)
    {
        if (category==null) {
            throw new IllegalArgumentException(CATEGORYARG);
        }
        if (name==null) {
            throw new IllegalArgumentException(PROPERTYARG);
        }
        Map reftable = Iteration.get().m_refTable;
        Object prev = null;
        synchronized(reftable) {
            Object item = reftable.get(category);
            if (item==null) {
                item = AntXFixture.newSynchronizedMap();
                reftable.put(category,item);
                prev = ((Map)item).put(name,value);
            } else if (item instanceof Map) {
                prev = ((Map)item).put(name,value);
            }
        }
        return prev;
    }



    /**
     * Removes an existing iteration property and any associated values.
     * @param name property's name (non-null)
     * @throws IllegalArgumentException if name is <i>null</i>.
     * @return the previously installed value (can be <i>null</i>)
     **/
    public static Object removeProperty(String name)
    {
        if (name==null) {
            throw new IllegalArgumentException(PROPERTYARG);
        }
        return Iteration.get().m_refTable.remove(name);
    }



    /**
     * Removes an existing iteration category property. Does not remove the
     * category's map if this is the last property; caller must do that explicitly
     * with a call to {@linkplain #removeProperty(String) removeProperty(category)}.
     * @param category the iteration category (non-null)
     * @param name property's name (non-null)
     * @throws IllegalArgumentException if name or category is <i>null</i>.
     * @return the previously installed value (can be <i>null</i>)
     **/
    public static Object removeProperty(String category, String name)
    {
        if (category==null) {
            throw new IllegalArgumentException(CATEGORYARG);
        }
        if (name==null) {
            throw new IllegalArgumentException(PROPERTYARG);
        }
        Object item = Iteration.get().m_refTable.get(category);
        if (item instanceof Map) {
            return ((Map)item).remove(name);
        }
        return null;
    }



//  ---------------------------------------------------------------------------------------
//  Sub-class visible member fields:
//  ---------------------------------------------------------------------------------------

    /**
     * This iteration's default property values manager. Initialized
     * to a standard AntX {@linkplain Defaults defaults} object.
     * @since JWare/AntX 0.5
     */
    protected Defaults m_defaultsINSTANCE= new Defaults();


    /**
     * This iteration's default set of fixture exclusion (for isolates and
     * other configuration tasks). Initialized to a standard AntX
     * {@linkplain ExcludedFixture excluded fixture} object.
     * @since JWare/AntX 0.5
     */
    protected ExcludedFixture m_xcludedFixtureINSTANCE= new ExcludedFixture();
    

    /**
     * This iteration's lenient stringifier. Initialized to a standard
     * {@linkplain Stringifier}.
     */
    protected Stringifier m_stringifierINSTANCE= new Stringifier(true);


    /**
     * This iteration's strict stringifier. Initialized to a standard
     * {@linkplain Stringifier}.
     */
    protected Stringifier m_strictStringifierINSTANCE= new Stringifier(false);


    /**
     * This iteration's manager for variable properties. Initialized to
     * a standard {@linkplain ExportedProperties}.
     */
    protected ExportedProperties m_exportedPropertiesINSTANCE= new ExportedProperties();


    /**
     * This iteration's value URI handler factory. Will produce
     * no-op proxy handlers for any prefix. Subclasses can change this
     * item's contents and configuration; they cannot remove or
     * delete this object.
     * @since JWare/AntX 0.5
     */
    protected final InstanceFactory m_valueUriFactory =
        new InstanceFactory(ValueURIHandler.None.class);



    /**
     * Returns the current value of an iteration member field. Will
     * also return <i>null</i> if no such field exists within this
     * iteration.
     * @param fieldname field's name (non-null)
     * @return field value (can be <i>null</i>)
     * @throws NoSuchFieldException if unable to find field in iteration.
     * @throws IllegalAccessException if unable to retrieve field value.
     * @since JWare/AntX 0.5
     **/
    protected final Object getFieldValue(String fieldname)
        throws NoSuchFieldException, IllegalAccessException
    {
        if (fieldname==null) {
            throw new IllegalArgumentException();
        }
        Field f = Tk.findField(getClass(),fieldname);
        boolean was = f.isAccessible();
        if (!was) {
            f.setAccessible(true);
        }
        try {
            return f.get(this);
        } finally {
            if (!was) {
                f.setAccessible(false);
            }
        }
    }



    private Object m_configLock = new int[0];
    private final String m_itid;
    private Map m_refTable = AntXFixture.newSynchronizedMap();

    // @.impl 
    // This is a VM-shared <em>class</em> variable. OK slowly... with overlays classes
    // are the expected clients. Each client class uses the same overlay handle
    // singleton, but each class scopes its iteration variables based on class-specific 
    // categories. Of course, the whole mess is scoped per thread so one threads overlay 
    // changes do not impact anothers.
    private final FixtureOverlays.Handle m_fixtureOverlays
        = new FixtureOverlays.Handle();
    
    // @.impl Used for testing injection-- do not remove or alter!
    String m_testField;
}


/* end-of-Iteration.java */