/**
 * $Id: AssertableDataType.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.util.Iterator;
import  java.util.Map;
import  java.util.Stack;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.DataType;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.apis.UIStringManager;

import  com.idaremedia.antx.apis.ProjectDependent;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Extension of basic Ant <i>DataType</i> that adds builtin assertions and helpers for
 * working with the 'refid' attribute.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public abstract class AssertableDataType extends DataType
    implements ProjectDependent
{
    /**
     * Initializes a new unlabeled datatype.
     **/
    protected AssertableDataType()
    {
        super();
        Iam_="";
    }


    /**
     * Initializes new CV-labeled datatype.
     * @param iam CV-label (non-null)
     **/
    protected AssertableDataType(String iam)
    {
        super();
        Iam_= Tk.cvlabelFrom(iam);
    }


    /**
     * Returns this datatype's CV-label. Never <i>null</i>.
     **/
    protected final String cvlabel_()
    {
        return Iam_;
    }


    /**
     * Shortcut that returns this datatypes's internal AntX UI
     * strings manager. Never returns <i>null</i>.
     * @see AntX#uistrs
     **/
    public final UIStringManager uistrs()
    {
        return AntX.uistrs();
    }


    /**
     * Throws assertion error if pre-condtion is not met.
     * @param c pre-condition
     * @param msg [optional] failure message (if not met)
     * @throws IllegalArgumentException if condition not met
     **/
    protected final void require_(boolean c, String msg)
    {
        if (!c) {
            String ermsg = uistrs().get("cv.require",Iam_,msg);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalArgumentException(ermsg);
        }
    }


    /**
     * Throws assertion error if post-condition is not met. Used
     * for post-condition verification.
     * @param c post-condition
     * @param msg [optional] failure message (if not met)
     * @throws IllegalStateException if condition not met
     **/
    protected final void ensure_(boolean c, String msg)
    {
        if (!c) {
            String ermsg = uistrs().get("cv.ensure",Iam_,msg);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalStateException(ermsg);
        }
    }


    /**
     * Throws assertion error if condition is not met. Used for
     * block and invariant verification.
     * @param c condition
     * @param msg [optional] failure message (if not met)
     * @throws IllegalStateException if condition not met
     **/
    protected final void verify_(boolean c, String msg)
    {
        if (!c) {
            String ermsg = uistrs().get("cv.verify",Iam_,msg);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalStateException(ermsg);
        }
    }


    /**
     * Notes an unexpected but manageable problem has occured.
     * Just logs a warning by default.
     * @param t [optional] causing throwable
     * @param msg caller's additional (context) message
     **/
    protected final void unexpected_(Throwable t, String msg)
    {
        String ermsg = uistrs().get("cv.unexpected",Iam_,msg,t);
        log(ermsg, Project.MSG_WARN);
    }



    /**
     * Verifies we're in a live project (created from build
     * process).
     **/
    protected final void verifyInProject_(String calr)
    {
        if (getProject()==null) {
            String ermsg = uistrs().get("cv.verifyInP",Iam_,calr);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalStateException(ermsg);
        }
    }



    /**
     * Tries to return an interesting identifier for this type.
     **/
    public String getId()
    {
        return getClass().getName()+"@"+
            String.valueOf(System.identityHashCode(this));
    }




    /**
     * Set this instance as a reference to an existing item.
     * @throws BuildException if any other attribute or nested element
     *         has been defined
     **/
    public void setRefid(Reference r)
        throws BuildException
    {
        require_(r!=null,"setRef- nonzro ref");
        if (isIndependentlyEdited()) {
            throw tooManyAttributes();
        }
        super.setRefid(r);
    }



    /**
     * Increments the non-refid attribute modified count; if user tries
     * to subsequently set this item as a reference, it veill die
     * very big.
     **/
    protected void edited(String what)
    {
        m_attrsEdited++;
    }

    /**
     * Shortcut of {@linkplain #edited(String) edited(&#46;&#46;&#46;)}
     * when the caller's identity isn't passed.
     **/
    protected final void edited()
    {
        edited("");
    }


    /**
     * Returns <i>true</i> if attributes other than the refid have
     * been defined.
     **/
    protected final boolean isIndependentlyEdited()
    {
        return m_attrsEdited>0;
    }



    /**
     * Returns a type-checked referenced object. Common utility that
     * generates resource bundle based messages if reference broken.
     * @param theProject the source project (null=> this task's project)
     * @param refid the referred-to thing's identifier (non-null)
     * @param requiredClass the required class of referred-to thing (non-null)
     * @throws BuildException if no such reference or object is not compatible
     * @since JWare/AntX 0.3
     **/
    public final Object getReferencedObject(Project theProject, String refid,
                                            Class requiredClass)
    {
        require_(refid!=null,"getRefObj- nonzro id");
        if (theProject==null) {
            theProject= getProject();
            verify_(theProject!=null,"getRefObj- hav project");
        }

        Object o = theProject.getReference(refid);
        String error = null;

        if (o==null) {
            error = uistrs().get("task.missing.refid", refid);
        }
        else if (!requiredClass.isInstance(o)) {
            error = uistrs().get("task.bad.refid", refid,
                                 requiredClass.getName(),
                                 o.getClass().getName());
        }
        if (error!=null) {
            log(error,Project.MSG_ERR);
            throw new BuildException(error);
        }
        return o;
    }



    /**
     * Returns a circular-dependency/type-checked reference. Like inherited
     * method except uses resource bundle-based error messages.
     **/
    protected Object getCheckedRef(Class requiredClass, String typeName)
    {
        if (!isChecked()) {
            Stack stk = new Stack();
            stk.push(this);
            dieOnCircularReference(stk,getProject());
        }
        return getReferencedObject(getProject(),getRefid().getRefId(),requiredClass);
    }



    /**
     * Tries to determine the (fully qualified) script name of
     * this object. Will return "<span class="src">none</span>" if
     * not able to determine proper name.
     * @since JWare/AntX 0.4
     * @.bug Does not work for two declarations of same type in
     *       different namespaces!
     **/
    public synchronized String getTypeName()
    {
        if (m_typeName==null) {
            verifyInProject_("typeName");
            Class myClass = getClass();
            Iterator itr= getProject().getDataTypeDefinitions().
                entrySet().iterator();//lock|clone?
            while (itr.hasNext()) {
                Map.Entry mE= (Map.Entry)itr.next();
                if (myClass==mE.getValue()) {
                    m_typeName = (String)mE.getKey();
                    break;
                }
            }
        }
        return m_typeName==null ? Strings.NONE : m_typeName;
    }


    private int m_attrsEdited;//NB: to zap refid+others
    private final String Iam_;
    private String m_typeName;//NB: lazy-inited
}

/* end-of-AssertableDataType.java */
