<!--
  ~ Copyright (c) 1999-2018 Allette Systems Pty Ltd
-->
<!--
  A set templates and functions common to all groups of services on the PageSeeder Main site.

  @author Christophe Lauret
-->
<xsl:stylesheet version="2.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:f="http://www.pageseeder.com/function"
                xmlns:json="http://www.w3.org/2005/xpath-functions"
                exclude-result-prefixes="#all">

<!-- The user currently logged in -->
<xsl:variable name="app"    select="/root/header/app"/>

<!-- Nonce for CSP -->
<xsl:variable name="nonce" select="/root/header/security/@nonce"/>

<!-- Site prefix -->
<xsl:variable name="site-prefix" select="/root/header/path/@context"/>

<!-- The current group -->
<xsl:variable name="current-group" select="/root/header/uri-parameters/parameter[@name='group'][1]"/>

<!-- The current project -->
<xsl:variable name="current-project" select="if ($current-group)
 then string-join(tokenize($current-group,'-')[position() lt last()], '-')
 else (//uri-parameters/parameter[@name='project'])[1]"/>

<!-- PageSeeder version -->
<xsl:variable name="version" select="/root/content/constants/@build-number"/>

<!-- PageSeeder UI version -->
<xsl:variable name="ui-version" select="'6.2.10'"/>

<!-- PDF JS version -->
<xsl:variable name="pdfjs-version" select="'5.4.394'"/>

<!-- Axios JS version -->
<xsl:variable name="axios-version" select="'1.13.2'"/>

<!-- Chart JS version -->
<xsl:variable name="chartjs-version" select="'4.5.1'"/>

<!-- Reference to the session object -->
<xsl:variable name="session" select="/root/content/session"/>

<!-- Reference to the context object -->
<xsl:variable name="context" select="/root/content/context"/>

<!-- Flag for internal development to load the development version of VueJS -->
<xsl:variable name="dev-mode" select="/root/header/berlioz/@mode = 'dev' or /root/header//parameter[@name='dev-mode'] = 'true'"/>

<!-- STRUCTURAL TEMPLATES                                                                       -->
<!-- ========================================================================================== -->

<!--
  HTML Head
-->
<xsl:template match="root" mode="html-head">
  <xsl:variable name="js-path"     select="concat($site-prefix, '/weborganic/ui/js')"/>
  <head>
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <meta name="application-name" content="PageSeeder v{$version}"/>
    <xsl:for-each select="//session/@csrf-token">
      <meta name="csrf-token" content="{.}"/>
    </xsl:for-each>
    <title>
      <xsl:variable name="page-title" as="xs:string*"><xsl:apply-templates select="." mode="html-title"/></xsl:variable>
      <xsl:choose>
        <xsl:when test="$current-group">
          <xsl:value-of select="string-join(($page-title, $current-group), ' &#x2022; ')"/>
        </xsl:when>
        <xsl:when test="$current-project">
          <xsl:value-of select="string-join(($page-title, $current-project), ' &#x2022; ')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="string-join(($page-title, header/location/@host), ' &#x2022; ')"/>
        </xsl:otherwise>
      </xsl:choose>
    </title>

    <!-- Preloading the core scripts -->
    <link rel="preload" href="{$js-path}/lib/axios-{$axios-version}.min.js" as="script" nonce="{$nonce}"/>
    <link rel="preload" href="{$site-prefix}/weborganic/ui/js/icons-{$ui-version}.min.js" as="script" nonce="{$nonce}"/>
    <link rel="preload" href="{$js-path}/vendors-{$ui-version}.min.js" as="script" nonce="{$nonce}"/>
    <link rel="preload" href="{$js-path}/commons-{$ui-version}.min.js" as="script" nonce="{$nonce}"/>
    <link rel="preload" href="{$js-path}/{@group}-{$ui-version}.min.js" as="script" nonce="{$nonce}"/>

    <!-- No need to load the styles in dev mode as they are inlined by the style loader -->
    <xsl:if test="not(header/location/@port = '9000' and header/berlioz/@mode = 'dev')">
      <link rel="stylesheet" href="{$site-prefix}/weborganic/ui/css/commons-{$ui-version}.min.css" />
      <link rel="stylesheet" href="{$site-prefix}/weborganic/ui/css/{@group}-{$ui-version}.min.css" />
    </xsl:if>

    <!-- For pages that contain a PDF, we use prefetch -->
    <xsl:if test="content/context/uri/@mediatype='application/pdf'">
      <link rel="prefetch" as="script" src="{$site-prefix}/weborganic/ui/js/lib/pdfjs-{$pdfjs-version}/pdf.min.mjs" nonce="{$nonce}" />
      <link rel="prefetch" as="script" src="{$site-prefix}/weborganic/ui/js/lib/pdfjs-{$pdfjs-version}/pdf.worker.min.mjs" nonce="{$nonce}"/>
    </xsl:if>

    <xsl:apply-templates select="." mode="favicons"/>

    <!-- Analytics (body) -->
    <xsl:for-each select="content/analytics/include[@location='head' or not(@location)]">
      <xsl:copy-of select="*|text()"/>
    </xsl:for-each>
  </head>
</xsl:template>

<!--
  No specific page title by default
-->
<xsl:template match="root" mode="html-title" as="xs:string*"/>

<!--
  HTML Body
-->
<xsl:template match="root" mode="html-body">
<xsl:variable name="fixed-theme" select="(content/ui)[1]/style[@theme != 'any']/@theme"/>
<xsl:variable name="js-path"     select="concat($site-prefix, '/weborganic/ui/js')"/>
<xsl:variable name="emaildomain" select="if (content/constants/@email-domain = 'pshosts')
                                         then content/context/defaulthost
                                         else content/constants/@email-domain"/>
<xsl:variable name="documentBaseUrl" select="concat(content/constants/@document-scheme, '://',
                                             if (content/context/defaulthost) then content/context/defaulthost else content/resolved-host,
                                             if ((content/constants/@document-scheme = 'http'  and content/constants/@document-port = '80') or
                                                 (content/constants/@document-scheme = 'https' and content/constants/@document-port = '443')) then '' else concat(':', content/constants/@document-port))"/>
<body class="pages-{@group} page-{@service} theme-{if ($fixed-theme) then $fixed-theme else 'default'}"
      id="page-{@service}"
      data-email-domain="{$emaildomain}"
      data-document-url="{$documentBaseUrl}"
      data-error-reporting="{content/context/@error-reporting}"
      data-global-template="{content/constants/@global-template}"
      data-global-theme="{if ($fixed-theme) then $fixed-theme else 'default'}"
      data-administration-enabled="{content/context/@administration-enabled}"
      data-ai-enabled="{content/context/@ai-enabled}"
      data-siteprefix="{$site-prefix}"
      data-version="{$version}"
      data-ui-version="{$ui-version}"
      data-default-minimal="{content/constants/@default-minimal}">
  <xsl:if test="content/server-message">
    <xsl:attribute name="data-server-message" select="f:to-json(content/server-message)"/>
  </xsl:if>
  <xsl:if test="content/context/@server-timezone">
    <xsl:attribute name="data-server-timezone" select="content/context/@server-timezone"/>
  </xsl:if>
  <xsl:if test="content/ui/portal">
    <xsl:attribute name="data-portal" select="f:to-json((content/ui)[1]/portal)"/>
  </xsl:if>
  <xsl:if test="content/ui/legal">
    <xsl:attribute name="data-legal" select="f:to-json((content/ui)[1]/legal)"/>
  </xsl:if>
  <xsl:if test="@service='account-search' or @service='group-search' or @service='project-search'">
    <xsl:attribute name="data-search" select="f:to-json-search-parameters(header//parameter)"/>
  </xsl:if>
  <xsl:if test="$session/member">
    <xsl:attribute name="data-member-id" select="$session/member/@id"/>
  </xsl:if>
  <xsl:if test="$context/project">
    <xsl:attribute name="data-project-name" select="$context/project/@name"/>
  </xsl:if>
  <xsl:if test="$context/group">
    <xsl:attribute name="data-group-name" select="$context/group/@name"/>
  </xsl:if>

  <!-- Include session data -->
  <xsl:if test="$session">
    <xsl:variable name="session-json">
      <json:map>
        <xsl:apply-templates select="$session/member" mode="json"><xsl:with-param name="key">member</xsl:with-param></xsl:apply-templates>
        <xsl:apply-templates select="$session/permissions" mode="json"><xsl:with-param name="key">permissions</xsl:with-param></xsl:apply-templates>
        <xsl:apply-templates select="$session/group-permissions" mode="json"><xsl:with-param name="key">groupPermissions</xsl:with-param></xsl:apply-templates>
        <json:array key="managerDisable">
          <xsl:for-each select="tokenize(content/constants/@manager-disable, ',')">
            <json:string><xsl:value-of select="." /></json:string>
          </xsl:for-each>
        </json:array>
        <json:boolean key="external"><xsl:value-of select="$session/@external = 'true'"/></json:boolean>
        <json:boolean key="externalOnly"><xsl:value-of select="$session/@external-only = 'true'"/></json:boolean>
      </json:map>
    </xsl:variable>
    <script id="session" type="application/json" nonce="{$nonce}"><xsl:value-of select="json:xml-to-json($session-json)"/></script>
  </xsl:if>

  <!-- Include context data -->
  <xsl:if test="$context/project">
    <script id="context-project" type="application/json" nonce="{$nonce}"><xsl:value-of select="f:to-json($context/project)"/></script>
  </xsl:if>
  <xsl:if test="$context/group">
    <script id="context-group" type="application/json" nonce="{$nonce}"><xsl:value-of select="f:to-json($context/group)"/></script>
  </xsl:if>
  <xsl:if test="$context/properties">
    <script id="context-properties" type="application/json" nonce="{$nonce}"><xsl:value-of select="f:to-json($context/properties)"/></script>
  </xsl:if>
  <xsl:if test="$context/label-config">
    <script id="context-labels" type="application/json" nonce="{$nonce}"><xsl:value-of select="f:to-json-labels-array($context/label-config)"/></script>
  </xsl:if>
  <xsl:if test="$context/uri">
    <script id="context-uri" type="application/json" nonce="{$nonce}"><xsl:value-of select="f:to-json($context/uri)"/></script>
  </xsl:if>

  <!-- Body content -->
  <div id="app" class="app">
    <xsl:apply-templates select="." mode="app-content"/>
  </div>

  <!-- Container for the dialogs -->
  <div id="dialogs" />

  <!-- Common utility scripts to load at bottom of page -->
  <script src="{$js-path}/lib/axios-{$axios-version}.min.js" nonce="{$nonce}"/>

  <!-- PageSeeder icons -->
  <script src="{$site-prefix}/weborganic/ui/js/icons-{$ui-version}.min.js" nonce="{$nonce}"/>

  <!-- Graphs -->
  <xsl:if test="contains(@flags, 'chartjs')">
    <script src="{$js-path}/lib/chart-{$chartjs-version}.min.js" nonce="{$nonce}"/>
  </xsl:if>

  <!-- Any special script for some pages -->
  <xsl:apply-templates select="." mode="html-script"/>

  <!-- Webpack bundles -->
  <script src="{$js-path}/vendors-{$ui-version}.min.js" nonce="{$nonce}"/>
  <script src="{$js-path}/commons-{$ui-version}.min.js" nonce="{$nonce}"/>
  <script src="{$js-path}/{@group}-{$ui-version}.min.js" nonce="{$nonce}"/>

  <!-- Analytics (body) -->
  <xsl:for-each select="content/analytics/include[@location='body']">
    <xsl:copy-of select="*|text()"/>
  </xsl:for-each>

</body>
</xsl:template>

<!--
  HTML Body for Panels (no script just content)
-->
<xsl:template match="root[@group='panel']" mode="html-body">
  <body>
    <div id="panel-wrapper" class="panel-wrapper">
      <xsl:apply-templates select="." mode="panel-content" />
    </div>
  </body>
</xsl:template>

<!--
  No specific script title by default
-->
<xsl:template match="root" mode="html-script" />

<!-- BODY TEMPLATES                                                                             -->
<!-- ========================================================================================== -->

<!-- Fallback templates -->
<xsl:template match="header|content" mode="#all"/>

<!--
  Footer
-->
<xsl:template match="root" mode="footer">
<footer class="app-footer" role="contentinfo">
  <p>Copyright &#169; 1999-<xsl:value-of select="format-date(current-date(), '[Y0001]')"/> Allette (Australia) &#x2014; PageSeeder v<xsl:value-of select="/root/header/app/@version"/></p>
</footer>
</xsl:template>

<!--
  Favicons
-->
<xsl:template match="root" mode="favicons">
  <xsl:variable name="available-themes" select="('blue','contrast','green','purple','red','steel','teal','yellow')"/>
  <xsl:variable name="organization-theme" select="(content/ui)[1]/style/@theme"/>
  <xsl:variable name="theme" select="if ($organization-theme and index-of($available-themes, $organization-theme)) then $organization-theme else 'blue'"/>
  <link rel="icon" href="{$site-prefix}/weborganic/ui/favicons/{$theme}/favicon.ico" sizes="any"/>
  <link rel="icon" href="{$site-prefix}/favicon.svg" type="image/svg+xml"/>
  <link rel="apple-touch-icon" href="{$site-prefix}/weborganic/ui/favicons/{$theme}/apple-touch-icon.png"/>
  <link rel="manifest" href="{$site-prefix}/weborganic/ui/favicons/{$theme}/site.webmanifest" />
  <meta name="theme-color" content="#ffffff"/>
</xsl:template>

</xsl:stylesheet>