<!--
  ~ Copyright (c) 1999-2019 Allette Systems Pty Ltd

  Module containing all customizations for the editing.

  Templates must return plain HTML that can be used in edit mode:
   - no components
   - classes and styles as expected by editor
-->
<xsl:stylesheet version="3.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:psml="http://pageseeder.com/PSML"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                exclude-result-prefixes="#all">

<!--
  Element `blockxref` display the block xref
-->
<xsl:template match="blockxref" mode="psml-edit">
  <div>
    <xsl:attribute name="id" select="concat('xref-', @id)"/>
    <xsl:attribute name="data-psml" select="'blockxref'"/>
    <xsl:attribute name="contenteditable" select="'false'"/>
    <xsl:attribute name="href">
      <xsl:apply-templates select="." mode="psml-href"/>
    </xsl:attribute>
    <xsl:apply-templates select="@*" mode="psml-data"/>
    <xsl:attribute name="class">
      <!-- For now xrefs are not editable in Tiny MCE -->
      <xsl:text>xref blockxref mceNonEditable</xsl:text>
      <xsl:for-each select="tokenize(@labels, ',')">
        <xsl:sequence select="concat(' ', psml:label-class(.))"/>
      </xsl:for-each>
      <xsl:if test="@unresolved='true'"> is-unresolved</xsl:if>
      <xsl:if test="@archived='true'"> is-archived</xsl:if>
    </xsl:attribute>
    <xsl:choose>
      <xsl:when test="@type = 'transclude'">
        <xsl:value-of select="if (@urititle) then @urititle else text()"/>
        <xsl:if test="@frag != 'default' and not(@unresolved='true')"> (fragment <xsl:value-of select="@frag"/>)</xsl:if>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates mode="#current"/>
      </xsl:otherwise>
    </xsl:choose>
  </div>
</xsl:template>

<!-- Break mapped to a `<br>` element. -->
<xsl:template match="br" mode="psml-edit">
  <br/>
</xsl:template>

<!--
  An unresolved image mapped to an <img> element.

  The 'width', 'height' and 'alt' attributes are simply copied.
-->
<xsl:template match="image" mode="psml-edit">
  <xsl:variable name="src-regex" select="'/ps/uri/(\d+)\.(jpg|JPG|jpeg|JPEG|png|PNG|gif|GIF)'"/>
  <xsl:choose>
    <xsl:when test="@uriid">
      <img src="{$site-prefix}/uri/{@uriid}" data-uriid="{@uriid}">
        <xsl:copy-of select="@width | @height | @alt"/>
        <xsl:if test="@labels">
          <xsl:attribute name="data-labels" select="@labels"/>
        </xsl:if>
      </img>
    </xsl:when>
    <xsl:when test="matches(@src, $src-regex)">
      <img src="{$site-prefix}/uri/{replace(@src, $src-regex, '$1')}" data-src="{@src}">
        <xsl:copy-of select="@width | @height | @alt"/>
        <xsl:if test="@labels">
          <xsl:attribute name="data-labels" select="@labels"/>
        </xsl:if>
      </img>
    </xsl:when>
    <xsl:otherwise>
      <img class="unresolved-image" style="{if (@width) then concat('width:',@width,'px;') else ''}{if (@height) then concat('height:',@height,'px') else ''}">
        <xsl:copy-of select="@width | @height | @alt"/>
        <xsl:if test="@labels">
          <xsl:attribute name="data-labels" select="@labels"/>
        </xsl:if>
      </img>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!--
  Element `<link>` mapped to simple one way HTML link `<a>`
-->
<xsl:template match="link" mode="psml-edit">
  <a target="_new" id="link-{count(preceding::link)}" href="{@href}">
    <xsl:apply-templates select="@role | @uriid | @labels | @unresolved" mode="psml-data"/>
    <xsl:apply-templates mode="#current"/>
  </a>
</xsl:template>

<!--
  Element `<list>` (Unordered list) mapped to `<ul>` element.

  We need to add the list styles for the editor.
-->
<xsl:template match="list" mode="psml-edit">
  <ul>
    <!-- In edit mode we need to use the style attribute for lists, as this is what the Tiny list plugin uses -->
    <xsl:apply-templates select="@role" mode="psml-data"/>
    <xsl:variable name="classNames" as="xs:string*">
      <xsl:if test="@role"><xsl:value-of select="concat('list-', @role)"/></xsl:if>
      <xsl:if test="@type"><xsl:value-of select="@type"/></xsl:if>
    </xsl:variable>
    <xsl:if test="count($classNames) gt 0">
      <xsl:attribute name="class"><xsl:value-of select="$classNames" separator=" "/></xsl:attribute>
    </xsl:if>
    <xsl:if test="@type">
      <xsl:attribute name="style">
        <xsl:choose>
          <xsl:when test="@type = 'disc'">list-style-type: default</xsl:when>
          <xsl:when test="@type = 'circle'">list-style-type: circle</xsl:when>
          <xsl:when test="@type = 'square'">list-style-type: square</xsl:when>
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates mode="#current"/>
  </ul>
</xsl:template>

<!--
  Element `<nlist>` ordered list mapped to `<ol>` element.

  We need to add the list styles for the editor.
-->
<xsl:template match="nlist" mode="psml-edit">
  <ol>
    <!-- In edit mode we need to use the style attribute for lists, as this is what the Tiny list plugin uses -->
    <xsl:apply-templates select="@role" mode="psml-data"/>
    <xsl:copy-of select="@start"/>
    <xsl:variable name="classNames" as="xs:string*">
      <xsl:if test="@role"><xsl:value-of select="concat('list-', @role)"/></xsl:if>
      <xsl:if test="@type"><xsl:value-of select="@type"/></xsl:if>
    </xsl:variable>
    <xsl:if test="count($classNames) gt 0">
      <xsl:attribute name="class"><xsl:value-of select="$classNames" separator=" "/></xsl:attribute>
    </xsl:if>
    <xsl:if test="@type">
      <xsl:attribute name="style">
        <xsl:choose>
          <xsl:when test="@type = 'arabic'">list-style-type: default</xsl:when>
          <xsl:when test="@type = 'upperalpha'">list-style-type: upper-alpha</xsl:when>
          <xsl:when test="@type = 'loweralpha'">list-style-type: lower-alpha</xsl:when>
          <xsl:when test="@type = 'upperroman'">list-style-type: upper-roman</xsl:when>
          <xsl:when test="@type = 'lowerroman'">list-style-type: lower-roman</xsl:when>
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates mode="#current"/>
  </ol>
</xsl:template>

<!-- Element `<para>` paragraph mapped to a `<p>` element. -->
<xsl:template match="para" mode="psml-edit">
  <p>
    <xsl:apply-templates select="@indent|@prefix|@numbered" mode="psml-data"/>
    <xsl:if test="@indent"><xsl:attribute name="class">indent-<xsl:value-of select="@indent"/></xsl:attribute></xsl:if>
    <xsl:apply-templates mode="#current"/>
    <!-- Required to make empty paragraph visible -->
    <xsl:if test="not(text()|*)"><br data-mce-bogus="1" /></xsl:if>
  </p>
</xsl:template>

<!-- Element `<placeholder>` paragraph mapped to a `<var>` element. -->
<xsl:template match="placeholder" mode="psml-edit">
  <var class="psml-placeholder" title="Property {@name}">
    <xsl:apply-templates select="@name" mode="psml-data"/>
    <xsl:apply-templates mode="#current"/>
  </var>
</xsl:template>

<!-- Element `<preformat>` (preformatted text) mapped to a `<pre>` element. -->
<xsl:template match="preformat" mode="psml-edit">
  <pre>
    <xsl:apply-templates select="@role" mode="psml-data"/>
    <xsl:if test="@role">
      <xsl:attribute name="class" select="if (starts-with(@role, 'lang-') or starts-with(@role, 'language-')) then @role else concat('pre-', @role)" />
    </xsl:if>
    <xsl:apply-templates mode="#current"/>
  </pre>
</xsl:template>

<!--
  Element `<xref>` (PageSeeder cross-references) map to `<a>` tag.

  The href is computed using the `psml:resolve-xref(@href)` function.
-->
<xsl:template match="xref" mode="psml-edit">
  <!-- Always a link for editing -->
  <a id="xref-{@id}" data-psml="xref">
    <xsl:attribute name="href">
      <xsl:apply-templates select="." mode="psml-href"/>
    </xsl:attribute>
    <xsl:apply-templates select="@*" mode="psml-data"/>
    <xsl:attribute name="class">
      <xsl:text>xref</xsl:text>
      <xsl:if test="@config"> config-<xsl:value-of select="@config"/></xsl:if>
      <xsl:for-each select="tokenize(@labels, ',')">
        <xsl:sequence select="concat(' ', psml:label-class(.))"/>
      </xsl:for-each>
      <xsl:if test="@unresolved='true'"> is-unresolved</xsl:if>
      <xsl:if test="@archived='true'"> is-archived</xsl:if>
    </xsl:attribute>
    <xsl:choose>
      <xsl:when test="@type = 'math'">
        <xsl:value-of select="@urititle"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates mode="#current"/>
      </xsl:otherwise>
    </xsl:choose>
  </a>
</xsl:template>

<!-- Path resolution =============================================================================================== -->

<!--
  Returns the value of the href attribute for the specified xref.
-->
<xsl:template match="xref" mode="psml-href" as="xs:string">
  <!-- FIXME Bug with xrefs to documents that are not shared, we need to find a matching group -->
  <xsl:value-of select="if (@unresolved = 'true' and @href) then @href else
                      if (@unresolved = 'true') then '#' else
                      concat($site-prefix, '/ui/g/', $current-group,'/d/', @uriid, '.html')"/>
</xsl:template>

<!--
  Returns the value of the href attribute for the specified xref.
-->
<xsl:template match="blockxref" mode="psml-href" as="xs:string">
  <xsl:variable name="publication" select="(//parameter[@name='publication'])[1]"/>
  <xsl:choose>
    <xsl:when test="@unresolved = 'true' and @href"><xsl:value-of select="@href" /></xsl:when>
    <xsl:when test="@unresolved = 'true'">#</xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="concat($site-prefix, '/ui/g/', $current-group,'/d/', @uriid, '.html')"/>
      <xsl:if test="$publication">?publication=<xsl:value-of select="$publication"/></xsl:if>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

</xsl:stylesheet>
