<?xml version="1.0"?>
<!DOCTYPE xsl:stylesheet [
  <!ENTITY styles SYSTEM "styles.css">
]>
<!--
  This XSLT module defines a function and templates for doing some basic
  CSS inlining.

  Note: There is no reason to modify this module.

  To use the inliner use the function:

    f:inliner( $content )
-->
<xsl:stylesheet version="2.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:f="http://www.pageseeder.com/function"
                exclude-result-prefixes="#all">

<!-- Styles defined externally
<xsl:variable name="external-styles" select="unparsed-text('styles.css')"/>
-->
<xsl:variable name="external-styles">&styles;</xsl:variable>

<!--
  List of CSS rules to inline (processed from external styles)
-->
<xsl:variable name="rules" as="element(rule)*">
  <!-- (1:selector(2:element)|(3:class)|(4(5:ancestor-class)(6:element)))(7:declaration) -->
  <xsl:analyze-string regex="{'(([\w\-_]+)|(\.[\w\-_]+)|((\.[\w\-_]+)\s+([\w\-_]+)))\s*\{\s*([^}]+)\}'}" select="$external-styles">
    <xsl:matching-substring>
      <rule selector="{regex-group(1)}">
        <xsl:if test="regex-group(2) != '' or regex-group(6)  != ''"><xsl:attribute name="element" select="regex-group(2),regex-group(6)" separator=""/></xsl:if>
        <xsl:if test="regex-group(3) != ''"><xsl:attribute name="class" select="substring(regex-group(3), 2)"/></xsl:if>
        <xsl:if test="regex-group(5) != ''"><xsl:attribute name="ancestor-class" select="substring(regex-group(5), 2)"/></xsl:if>
        <xsl:value-of select="regex-group(7)"/>
      </rule>
    </xsl:matching-substring>
  </xsl:analyze-string>
</xsl:variable>

<!--
  Basic CSS inlining function which will insert the CSS styles inline in the style
  attribute for every element it finds in the specified content.

  This inliner will only apply content based on class or element name.

  If the element already has some styles, no style is added.

  @param content The HTML content

  @return the same content with CSS inline styles inserted.
-->
<xsl:function name="f:inliner">
  <xsl:param name="content"/>
  <xsl:apply-templates select="$content" mode="inliner"/>
</xsl:function>

<!--
  Returns all the CSS rules applicable to an element by name or class.
-->
<xsl:function name="f:rules" as="element(rule)*">
  <xsl:param name="element"/>
  <xsl:param name="class"/>
  <xsl:sequence select="$rules[@element = $element]"/>
  <xsl:if test="$class">
    <xsl:for-each select="tokenize($class, '\s')">
      <xsl:sequence select="$rules[@class=current()]"/>
    </xsl:for-each>
  </xsl:if>
</xsl:function>

  <!--
    Process elements to insert inline style
  -->
<xsl:template match="*" mode="inliner">
  <xsl:copy>
    <!-- Copy all attributes except 'style' -->
    <xsl:copy-of select="@*[not(name() = 'style')]"/>
    <!-- Construct style in the following order: element, class, class + element, inline -->
    <xsl:variable name="local-rules" select="f:rules(name(), @class)" as="element(rule)*"/>
    <xsl:variable name="style">
      <xsl:value-of select="f:semi-colon($local-rules[@element][not(@ancestor-class)])"/>
      <xsl:value-of select="f:semi-colon($local-rules[@class][not(@ancestor-class)])"/>
      <xsl:value-of select="f:semi-colon($local-rules[@element][@ancestor-class = current()/ancestor::*/@class])"/>
      <xsl:value-of select="f:semi-colon(@style)"/>
      <xsl:apply-templates select="." mode="custom-styles" />
    </xsl:variable>
    <!-- Add style attribute if there is some inline style -->
    <xsl:if test="$style != ''">
      <!-- get only last value of each style, organize them in a variable -->
      <xsl:variable name="all-styles">
        <styles xmlns="">
          <xsl:for-each select="tokenize($style, ';')[normalize-space(tokenize(., ':')[1]) != ''][normalize-space(tokenize(., ':')[2]) != '']">
            <style name="{normalize-space(tokenize(., ':')[1])}" value="{normalize-space(tokenize(., ':')[2])}" />
          </xsl:for-each>
        </styles>
      </xsl:variable>
      <xsl:attribute name="style">
        <!-- loop through and get only last value -->
        <xsl:for-each select="distinct-values($all-styles//style/@name)[. != '']">
          <xsl:value-of select="." />
          <xsl:text>:</xsl:text>
          <xsl:value-of select="($all-styles//style[@name = current()])[last()]/@value" />
          <xsl:text>; </xsl:text>
        </xsl:for-each>
      </xsl:attribute>
    </xsl:if>
    <!-- bgcolor for 'body', 'table', 'td' and 'th' elements -->
    <xsl:if test="self::body or self::table or self::td or self::th">
      <xsl:if test="not(@bgcolor)">
        <xsl:analyze-string regex="{'background(-color)?:\s*(#[a-fA-F0-9]{6})'}" select="$style">
          <xsl:matching-substring>
            <xsl:attribute name="bgcolor" select="regex-group(2)"/>
          </xsl:matching-substring>
        </xsl:analyze-string>
      </xsl:if>
    </xsl:if>
    <xsl:apply-templates select="*|text()" mode="inliner"/>
  </xsl:copy>
</xsl:template>

<xsl:template match="*" mode="custom-styles" />

<!--
  Load custom inline styles
-->
<xsl:variable name="custom-styles" select="/notification/organization/style" />

<!--
  Font family
-->
<xsl:template match="body | td | th | div | p | ul | ol | li |
                      h1 | h2 | h3 | h4 | h5 | h6 |
                      *[@class = 'type-above' or
                        @class = 'ps-table_th' or
                        @class = 'ps-table_td' or
                        @class = 'ps-message_th' or
                        @class = 'ps-message_td']" mode="custom-styles">
  <xsl:if test="$custom-styles/text/@font-family">
    <xsl:text> font-family:</xsl:text>
    <xsl:value-of select="$custom-styles/text/@font-family" />
    <xsl:text>;</xsl:text>
  </xsl:if>
</xsl:template>

<!--
  Header
-->
<xsl:template match="*[@class = 'header-wrap'][$custom-styles/text or $custom-styles/header]" mode="custom-styles">
  <xsl:if test="$custom-styles/text/@font-family">
    <xsl:text> font-family:</xsl:text>
    <xsl:value-of select="$custom-styles/text/@font-family" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:if test="$custom-styles/header/@background-color">
    <xsl:text> background-color:</xsl:text>
    <xsl:value-of select="$custom-styles/header/@background-color" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:if test="$custom-styles/header/@text-color">
    <xsl:text> color:</xsl:text>
    <xsl:value-of select="$custom-styles/header/@text-color" />
    <xsl:text>;</xsl:text>
  </xsl:if>
</xsl:template>

<!-- overwrite text color as it's specified on sub elements in the header -->
<xsl:template match="*[@class = 'header-pslogo' or @class = 'header-title']
                      [$custom-styles/text/@font-family or $custom-styles/header/@text-color]" mode="custom-styles">
  <xsl:if test="$custom-styles/text/@font-family">
    <xsl:text> font-family:</xsl:text>
    <xsl:value-of select="$custom-styles/text/@font-family" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:text> color:</xsl:text>
  <xsl:value-of select="$custom-styles/header/@text-color" />
  <xsl:text>;</xsl:text>
</xsl:template>

<!--
  Footer
-->
<xsl:template match="*[@class = 'footer-wrap'][$custom-styles/text or $custom-styles/footer]" mode="custom-styles">
  <xsl:if test="$custom-styles/text/@font-family">
    <xsl:text> font-family:</xsl:text>
    <xsl:value-of select="$custom-styles/text/@font-family" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:if test="$custom-styles/footer/@background-color">
    <xsl:text> background-color:</xsl:text>
    <xsl:value-of select="$custom-styles/footer/@background-color" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:if test="$custom-styles/footer/@text-color">
    <xsl:text> color:</xsl:text>
    <xsl:value-of select="$custom-styles/footer/@text-color" />
    <xsl:text>;</xsl:text>
  </xsl:if>
</xsl:template>

<!-- overwrite text color as it's specified on the p element in the footer -->
<xsl:template match="p[ancestor::*[@class = 'footer-wrap']][$custom-styles/text or $custom-styles/footer/@text-color]" mode="custom-styles">
  <xsl:if test="$custom-styles/text/@font-family">
    <xsl:text> font-family:</xsl:text>
    <xsl:value-of select="$custom-styles/text/@font-family" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:if test="$custom-styles/footer/@text-color">
    <xsl:text> color:</xsl:text>
    <xsl:value-of select="$custom-styles/footer/@text-color" />
    <xsl:text>;</xsl:text>
  </xsl:if>
</xsl:template>

<!--
  Banners
-->
<xsl:template match="*[f:is-banner(.)][$custom-styles/text/@font-family or $custom-styles/banner]" mode="custom-styles">
  <xsl:if test="$custom-styles/text/@font-family">
    <xsl:text> font-family:</xsl:text>
    <xsl:value-of select="$custom-styles/text/@font-family" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:if test="$custom-styles/banner[@type = replace(current()/@class, '-wrap', '')]/@background-color">
    <xsl:text> background-color:</xsl:text>
    <xsl:value-of select="$custom-styles/banner[@type = replace(current()/@class, '-wrap', '')]/@background-color" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:if test="$custom-styles/banner[@type = replace(current()/@class, '-wrap', '')]/@text-color">
    <xsl:text> color:</xsl:text>
    <xsl:value-of select="$custom-styles/banner[@type = replace(current()/@class, '-wrap', '')]/@text-color" />
    <xsl:text>;</xsl:text>
  </xsl:if>
</xsl:template>

<!-- overwrite text color as it's specified on the p, b or span element in some banners -->
<xsl:template match="p[ancestor::*[f:is-banner(.)]][$custom-styles/text/@font-family or $custom-styles/banner/@text-color] |
                  span[ancestor::*[f:is-banner(.)]][$custom-styles/text/@font-family or $custom-styles/banner/@text-color] |
                     b[ancestor::*[f:is-banner(.)]][$custom-styles/text/@font-family or $custom-styles/banner/@text-color]" mode="custom-styles">
  <xsl:if test="$custom-styles/text/@font-family">
    <xsl:text> font-family:</xsl:text>
    <xsl:value-of select="$custom-styles/text/@font-family" />
    <xsl:text>;</xsl:text>
  </xsl:if>
  <xsl:variable name="banner-type" select="replace((ancestor::*[f:is-banner(.)])[1]/@class, '-wrap', '')" />
  <xsl:if test="$custom-styles/banner[@type = $banner-type]/@text-color">
    <xsl:text> color:</xsl:text>
    <xsl:value-of select="$custom-styles/banner[@type = $banner-type]/@text-color" />
    <xsl:text>;</xsl:text>
  </xsl:if>
</xsl:template>

<xsl:function name="f:is-banner" as="xs:boolean">
  <xsl:param name="element" />
  <xsl:sequence select="$element/@class = 'security-wrap' or
                        $element/@class = 'moderation-wrap' or
                        $element/@class = 'alert-wrap' or
                        $element/@class = 'task-wrap' or
                        $element/@class = 'membership-wrap' or
                        $element/@class = 'digest-wrap'" />
</xsl:function>

<xsl:function name="f:semi-colon" as="xs:string*">
  <xsl:param name="style" />
  <xsl:for-each select="$style">
    <xsl:value-of select="if (string(.) = '') then '' else
                      if (ends-with(., ';')) then . else
                      concat(., ';')" />
  </xsl:for-each>
</xsl:function>

</xsl:stylesheet>
