/**
 * $Id: NewRefIdValueURIHandlerTest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.tests;

import  junit.framework.TestSuite;

import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.helpers.SIDs;
import  com.idaremedia.antx.valueuri.NewRefIdValueURIHandler;

/**
 * Testsuite for {@linkplain NewRefIdValueURIHandler}.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,test
 **/

public final class NewRefIdValueURIHandlerTest extends ValueURIHandlerTestSkeleton
{
    /**
     * Initializes a new test case for named method.
     * @param methodname test case method's name (non-null)
     **/
    public NewRefIdValueURIHandlerTest(String methodname)
    {
        super("NewRefIdValueURIHandler::",methodname);
    }


    /**
     * Returns full test suite for NewRefIdValueURIHandler.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(NewRefIdValueURIHandlerTest.class);
    }

//  ---------------------------------------------------------------------------------------------------------
//  --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    protected ValueURIHandler newOUT()
    {
        return new NewRefIdValueURIHandler();
    }

//  ---------------------------------------------------------------------------------------------------------
//  ------------------------------------------- [ The Test Cases ] ------------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    /**
     * Common new refid verification checks.
     * @param refid the newly generated reference (before anything else done!)
     * @param XpectedPrefix expected prefix of new refid (non-null)
     **/
    private void verifyNewRefId(String refid, String XpectedPrefix)
    {
        assertNotNil(refid,"a refid generated");
        assertFalse(getProject().getReferences().containsKey(refid));
        assertTrue(refid.startsWith(XpectedPrefix),"Prefixed with '"+XpectedPrefix+"'");
    }


    /**
     * Common new pending refid verification checks.
     * @param refid the newly generated reference (before anything else done!)
     * @param XpectedPrefix expected prefix of new refid (non-null)
     **/
    private void verifyNewPendingRefId(String refid, String XpectedPrefix)
    {
        assertNotNil(refid,"a refid generated");
        assertIdent(getProject().getReferences().get(refid),FixtureExaminer.PENDING_REFERENCE);
        assertTrue(refid.startsWith(XpectedPrefix),"Prefixed with '"+XpectedPrefix+"'");
    }


    /**
     * Verify we can generate a project-unique reference always.
     * @since JWare/AntX 0.5
     **/
    public void testBaseline_AntX05()
    {
        checkBaseline();

        ValueURIHandler out = newOUT();
        String refid = out.valueFrom("","$newrefid:",m_rqlink);
        verifyNewRefId(refid,"refid");

        String varname = SIDs.next("var");
        refid = out.valueFrom("?"+varname,"$newrefid:?"+varname,m_rqlink);
        verifyNewPendingRefId(refid,"refid");
    }


    /**
     * Verifies we can generate references prefixed with custom
     * strings.
     * @since JWare/AntX 0.5
     **/
    public void testAllowCustomPrefixed_AntX05()
    {
        ValueURIHandler out = newOUT();
        String refid = out.valueFrom("test","$newrefid:test",m_rqlink);
        verifyNewRefId(refid,"test");
        refid = out.valueFrom("test2?","$newrefid:test2?",m_rqlink);
        verifyNewRefId(refid,"test2");
    }


    /**
     * Verify we can embed property references into refid value
     * URIs.
     * @since JWare/AntX 0.5
     **/
    public void testEmbeddedPropertyRefs_AntX05()
    {
        getProject().setProperty("seekrit","ssshhh");
        ValueURIHandler out = newOUT();
        String refid0 = out.valueFrom("@(seekrit)","$newrefid:@(seekrit)",m_rqlink);
        verifyNewRefId(refid0,"ssshhh");
        String refid1 = out.valueFrom("@(seekrit)?ignore","$newrefid:@(seekrit)?ignore",m_rqlink);
        verifyNewPendingRefId(refid1,"ssshhh");
        assertNotEqual(refid1,refid0,"Second generated reference");
    }


    /**
     * Verify we can automatically fillin a variable with
     * value of new refid.
     * @since JWare/AntX 0.5
     **/
    public void testCanSetVariableToNewRefId_AntX05()
    {
        String varname = SIDs.next("var");
        ValueURIHandler out = newOUT();
        String refid = out.valueFrom("abc?"+varname,"$newrefid:abc?"+varname,m_rqlink);
        verifyNewPendingRefId(refid,"abc");
        assertTrue(ExportedProperties.has(varname),"var("+varname+") created");

        varname = SIDs.next("var");
        assertFalse(ExportedProperties.has(varname));
        refid = out.valueFrom("?"+varname,"$newrefid:?"+varname,m_rqlink);
        verifyNewPendingRefId(refid,"refid");
        assertTrue(ExportedProperties.has(varname),"var("+varname+") created");
    }
}

/* end-of-NewRefIdValueURIHandlerTest.java */