/**
 * $Id: DateTimeValueURIHandlerTest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.tests;

import  java.text.Format;
import  java.util.Date;
import  java.util.Iterator;
import  java.util.Map;

import  junit.framework.TestSuite;

import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.ValueURIHandler;
import com.idaremedia.antx.valueuri.datetime.DateTimeValueURIHandler;

/**
 * Testsuite for {@linkplain DateTimeValueURIHandler}.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,test
 **/

public final class DateTimeValueURIHandlerTest extends ValueURIHandlerTestSkeleton
{
    private static final Map MAPPINGS = DateTimeValueURIHandler.copyOfMappings();
 
    /* 'Sun Oct 01 12:35:30 EDT 2000' = 970418130646L */
    private static final String OCT012000 = "970418130646";
    private static final Date OCT012000DATE = new Date(970418130646L);
    private static final Map OCT012000STRINGS = AntXFixture.newMap();
    static {
        Map m = OCT012000STRINGS;
        m.put("$gmtdate:","01-Oct-2000 GMT");
        m.put("$longtime:","12:35:30 PM");
        m.put("$cvsdate:","01 Oct 2000");
        m.put("$date:","01-Oct-2000");
        m.put("$shortdate:","01-Oct");
        m.put("$svndate:","01 Oct 2000");
        m.put("$datetime:","12:35:30 PM 01-Oct-2000");
        m.put("$longdate:","01-Oct-2000");
        m.put("$time:","12:35:30 PM");
        m.put("$longdatetime:","12:35:30 PM 01-Oct-2000");
        m.put("$shorttime:","12:35 PM");
        m.put("$shortdatetime:","01-Oct 12:35 PM");
        m.put("$gmtdatetime:","16:35:30 01-Oct-2000 GMT");
        m.put("$gmttime:","16:35:30 GMT");
        m.put("$changelogdate:","01 Oct 2000");
        m.put("$duration:","309902hr.12min.10sec.902ms");
        m.put("$query1:","001001123530-0400");
        m.put("$query2:","02000.October.01 AD 12:35 PM");
    }



    /**
     * Initializes a new test case for named method.
     * @param methodname test case method's name (non-null)
     **/
    public DateTimeValueURIHandlerTest(String methodname)
    {
        super("DateTimeValueURIHandler::",methodname);
    }


    /**
     * Returns full test suite for DateTimeValueURIHandler.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(DateTimeValueURIHandlerTest.class);
    }

//  ---------------------------------------------------------------------------------------------------------
//  --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    protected ValueURIHandler newOUT()
    {
        return new DateTimeValueURIHandler();
    }
    

    private long defineNow(ValueURIHandler out, long nowTS)
    {
        if (nowTS<0L) {
            nowTS = System.currentTimeMillis();
        }
        ((DateTimeValueURIHandler)out).setNow(nowTS);
        getProject().setProperty("now",String.valueOf(nowTS));
        return nowTS;

    }

//  ---------------------------------------------------------------------------------------------------------
//  ------------------------------------------- [ The Test Cases ] ------------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    /**
     * Verify there is at least one predefined datetime uri scheme.
     * @since JWare/AntX 0.5
     **/
    public void testBaseline_AntX05()
    {
        checkBaseline();

        assertNotNil(MAPPINGS,"Default Mappings");
        assertFalse(MAPPINGS.isEmpty(),"Default Mappings IsEmpty");
    }


    /**
     * Verifies that we can pass the symbolic timestamp string "now"
     * to all formatters and now have anything blow up.
     * @since JWare/AntX 0.5
     **/
    public final void testAllPredefinedFormatsWithNowTimestamp_AntX05()
    {
        ValueURIHandler out = newOUT();
        println(" NOW = "+defineNow(out,-1L));

        String result0,result1;
        Iterator itr = MAPPINGS.entrySet().iterator();
        
        while (itr.hasNext()) {
            Map.Entry e = (Map.Entry)itr.next();
            String scheme = e.getKey().toString();
            result0 = out.valueFrom("now",scheme+"now",m_rqlink);
            println(scheme+"now"+" = "+result0);
            assertNotNil(result0,scheme+"now");
            result1 = out.valueFrom("",scheme,m_rqlink);
            assertEqual(result1,result0,scheme);
            result1 = out.valueFrom("@(now)?",scheme+"@(now)?",m_rqlink);
            assertEqual(result1,result0,scheme+"@(now)?");
        }
    }



    /**
     * Verifies that we can pass a timestamp string to all formatters
     * and not have anything blow up (duration). All formatters
     * should return expected strings.
     * @since JWare/AntX 0.5
     **/
    public final void testTimestampFormatsWithFixedTimestamp_AntX05()
    {
        ValueURIHandler out = newOUT();
        Iterator itr = MAPPINGS.entrySet().iterator();
        String result;
        
        while (itr.hasNext()) {
            Map.Entry e = (Map.Entry)itr.next();
            String scheme = e.getKey().toString();
            Format dfmt = (Format)e.getValue();
            Boolean known = DateTimeValueURIHandler.isTimestampScheme(scheme);
            assertNotNil(known,"category of default scheme");
            if (known==Boolean.TRUE) {
                result = out.valueFrom(OCT012000,scheme+OCT012000,m_rqlink);
                assertEqual(result,OCT012000STRINGS.get(scheme),scheme+OCT012000+"(expected)");
                assertEqual(result,dfmt.format(OCT012000DATE),scheme+OCT012000);
            }
        }
    }



    /**
     * Verifies that we can pass a timestamp of zero and one to all
     * duration formatters without barfage.
     * @since JWare/AntX 0.5
     **/
    public final void testDurationFormatsHandleOneAndZero_AntX05()
    {
        ValueURIHandler out = newOUT();
        Iterator itr = MAPPINGS.entrySet().iterator();
        String result;
        
        while (itr.hasNext()) {
            Map.Entry e = (Map.Entry)itr.next();
            String scheme = e.getKey().toString();
            if (DateTimeValueURIHandler.isTimestampScheme(scheme)==Boolean.FALSE) {
                result = out.valueFrom("0",scheme+"0",m_rqlink);
                assertNil(result,scheme+"0");
                Format dfmt = (Format)e.getValue();
                result = out.valueFrom("1",scheme+"1",m_rqlink);
                assertEqual(result,dfmt.format(new Date(1)),scheme+"1");
            }
        }
    }



    /**
     * Verifies that for known schemes, the output's format is fixed
     * regardless of what the query string is set to.
     * @since JWare/AntX 0.5
     **/
    public void testPredefinedNameForcesFixedFormat_AntX05()
    {
        ValueURIHandler out = newOUT();
        defineNow(out,-1L);

        final String QUERY0 = "now?hh 'o''clock' a, zzzz";
        final String QUERY1 = "@(now)?yyMMddHHmmssZ";
        String customresult = out.valueFrom(QUERY1,"$blah:"+QUERY1,m_rqlink);
        println(" CUSTOM RESULT: "+customresult);

        String result0,result1,result2;
        Iterator itr = MAPPINGS.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry e = (Map.Entry)itr.next();
            String scheme = e.getKey().toString();
            result0 = out.valueFrom("",scheme,m_rqlink);
            assertNotNil(result0,scheme);
            result1 = out.valueFrom(QUERY0,scheme+QUERY0,m_rqlink);
            assertEqual(result1,result0,scheme+QUERY0);
            result2 = out.valueFrom(QUERY1,scheme+QUERY1,m_rqlink);
            assertEqual(result2,result1,scheme+QUERY1);
            assertNotEqual(result1,customresult,scheme+QUERY1);
        }
    }
    
    
    /**
     * Verifies that for unknown schemes, a custom format string is
     * necessary.
     * @since JWare/AntX 0.5
     **/
    public void testCustomSchemeNeedsCustomFormat_AntX05()
    {
        ValueURIHandler out = newOUT();
        defineNow(out,OCT012000DATE.getTime());
        
        String result = out.valueFrom("now","$query1:now",m_rqlink);
        assertNil(result,"$query1:now"); 
        result = out.valueFrom("","$query2:",m_rqlink);
        assertNil(result,"$query2:");

        final String QUERY1 = "@(now)?yyMMddHHmmssZ";
        final String QUERY2 = "?yyyyy.MMMMM.dd GGG hh:mm aaa";
        
        result = out.valueFrom(QUERY1,"$query1:"+QUERY1,m_rqlink);
        assertEqual(result,OCT012000STRINGS.get("$query1:"),"$query1:"+QUERY1);
        result = out.valueFrom(QUERY2,"$query2:"+QUERY2,m_rqlink);
        assertEqual(result,OCT012000STRINGS.get("$query2:"),"$query2:"+QUERY2);        
    }



    /**
     * Verifies that bogus inputs to the handler are managed
     * gracefully. Always returns <i>null</i>.
     * @since JWare/AntX 0.5
     **/
    public void testInvalidTimestampsHandledGracefully_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result;
        
        Iterator itr = MAPPINGS.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry e = (Map.Entry)itr.next();
            String scheme = e.getKey().toString();
            result = out.valueFrom("-111",scheme+"-111",m_rqlink);
            assertNil(result,scheme+"-111");
            result = out.valueFrom("NaN",scheme+"NaN",m_rqlink);
            assertNil(result,scheme+"NaN");
        }

        result = out.valueFrom("-111","$query1:-111?yyyyMMddd",m_rqlink);
        assertNil(result,"$query1:-111?yyyyMMddd");
        result = out.valueFrom("-111","$query1:NaN?yyyyMMddd",m_rqlink);
        assertNil(result,"$query1:NaN?yyyyMMddd");
    }



    /**
     * Verifies that our default fallback API only groks the "now"
     * timestamp.
     * @since JWare/AntX 0.5
     **/
    public void testOnlyNowHandledByDefaultFallback_AntX05()
    {
        ValueURIHandler out = newOUT();
        defineNow(out,OCT012000DATE.getTime());
        String result;

        result = out.getDefaultValue("$datetime:now",m_rqlink);
        assertEqual(result,out.valueFrom("","$datetime:",m_rqlink),"default $datetime:");
        result = out.getDefaultValue("$tstamp:now?yyyyMMddd",m_rqlink);
        assertEqual(result,out.valueFrom("now?yyyyMMddd","$tstamp:now?yyyyMMddd",m_rqlink),
                    "default $tstamp:now?yyyyMMddd");

        assertNil(out.getDefaultValue("$datetime:"+OCT012000,m_rqlink),
                  "default $datetime:"+OCT012000);
        assertNil(out.getDefaultValue("$tstamp:"+OCT012000+"?yyyyMMddd",m_rqlink),
                  "default $tstamp:"+OCT012000+"?yyyyMMddd");
    }
}

/* end-of-DateTimeValueURIHandlerTest.java */