/**
 * $Id: ShortenStringURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Value URI handler that tries to shorten a long string in a display friendly manner.
 * This handler is very useful for shortening long file paths into something easily
 * readable in build archive logs. The general form of the URI:
 * <span class="src"><nobr>$shorten:string[?[maxlength][,,[left|right][,,ellipses string]]]</nobr>
 * </span>.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;emit msgid="msg.generating.apidocs" msgarg1="<b>${$shorten:</b>@{html}}"/&gt;
 *
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="shorten"
 *             value="com.idaremedia.antx.valueuri.ShortenStringURIHandler"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 * @see       BasenameValueURIHandler
 **/

public final class ShortenStringURIHandler extends ValueURIHandlerSkeleton
{
    static final int LEFT=0;
    static final int RIGHT=LEFT+1;
    private static final String DDD="...";

    /**
     * Default trim length for shortened strings ({@value}).
     */
    public static final int MAXLEN= 41;

    /**
     * Default minimum length for shortened strings (<i>3</i>).
     */
    public static final int MINLEN= DDD.length();


    /**
     * Initializes a new shorten string value uri handler.
     **/
    public ShortenStringURIHandler()
    {
        super();
    }


    /**
     * Tries to shorten the incoming string as requested. If string does
     * not need shortening it is returned unchanged (property references
     * are resolved).
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        final Project P = clnt.getProject();
        int maxlength = MAXLEN;
        String ellipses= DDD;
        int dropped = LEFT;
        String longstring = uriFragment;
        final int urilen= uriFragment.length();

        int i = longstring.indexOf("?");
        if (i>=0) {
            longstring = longstring.substring(0,i++);
            if (i<urilen) {
                String s;
                int j = uriFragment.indexOf(",,",i);
                if (j>=0) {
                    s = Tk.resolveString(P,uriFragment.substring(i,j),true);
                    maxlength = Tk.integerFrom(s,MAXLEN);

                    i = uriFragment.indexOf(",,",j+2);
                    if (i<0) {
                        s = uriFragment.substring(j+2);
                        dropped = leftOrRight(s,LEFT,P);
                    } else {
                        s = uriFragment.substring(j+2,i);
                        dropped = leftOrRight(s,LEFT,P);
                        i += 2;
                        if (i<urilen) {
                            s = uriFragment.substring(i);
                            ellipses = Tk.resolveString(P,s,true);
                        }
                    }
                } else {
                    s = Tk.resolveString(P,uriFragment.substring(i),true);
                    maxlength = Tk.integerFrom(s,MAXLEN);
                }
            }
        }
        if (maxlength<MINLEN) {
            maxlength=MINLEN;
        }
        longstring = Tk.resolveString(P,longstring,true);
        final int strlen = longstring.length();
        if (strlen>maxlength) {
            switch (dropped) {
                case LEFT: {
                    int from = strlen-maxlength+ellipses.length();
                    if (from>=strlen) {
                        from = strlen-maxlength+DDD.length();
                        ellipses = DDD;
                    }
                    longstring= ellipses+longstring.substring(from);
                    break;
                }
                default: {
                    int to = maxlength-ellipses.length();
                    if (to<=0) {
                        to = maxlength-DDD.length();
                        ellipses = DDD;
                    }
                    longstring = longstring.substring(0,to)+ellipses;
                    break;
                }
            }
        }
        return longstring;
    }


    static int leftOrRight(String s, int dflt, Project P)
    {
        s = Tk.resolveString(P,s,true);
        s = s.toLowerCase();

        if ("left".equals(s)) {
            return LEFT;
        }
        if ("right".equals(s)) {
            return RIGHT;
        }
        return dflt;
    }
}


/* end-of-ShortenStringURIHandler.java */