/**
 * $Id: NewRefIdValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri;

import  java.util.Map;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.SIDs;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Value URI handler that generates a new reference id that does not already exist in
 * the named project. Very useful in tests and SAMS artifact definition methods that
 * need to add data objects to the current project's fixture without worrying about name
 * collisons. Be careful when using this value URI as part of an overlaid property's
 * value&#8212; the value <em>will change everytime</em> the property is dereferenced!
 * <p/>
 * The general URI format: <span class="src">$newrefid:[prefix][?variable-name]</span>
 * where <span class="src">prefix</span> is an optional prefix that is used in new
 * reference name (useful for debugging), and <span class="src">variable-name</span> is
 * the name of the mutable variable that should be updated with new id.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;assign var="myjunit" value="${<b>$newrefid:junit</b>}"/&gt;
 *    &lt;createtask name="${$var:myjunit}"...&gt;

 * -or better-
 *    &lt;createtask name="${<b>$newrefid:junit?myjunit</b>}"&gt;
 *       &lt;junit.../&gt;
 *    &lt;/createtask&gt;
 *
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="newrefid"
 *             value="com.idaremedia.antx.valueuri.NewRefIdValueURIHandler"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 * @see       SIDsValueURIHandler
 **/

public final class NewRefIdValueURIHandler extends ValueURIHandlerSkeleton
{
    private static final String DEFAULT_PREFIX= "refid";

    /**
     * Initalizes a new (unique) refid value generator.
     **/
    public NewRefIdValueURIHandler()
    {
        super();
    }


    /**
     * Returns a project-unique reference identifier. If the name of
     * a variable is not provided, the reference object should be
     * inserted ASAP to minimize the collison potential.
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        final Project project = clnt.getProject();
        String variable = null;

        String prefix = DEFAULT_PREFIX;
        int i = uriFragment.indexOf("?");
        if (i>=0) {
            prefix = uriFragment.substring(0,i++);
            if (prefix.length()==0) {
                prefix = DEFAULT_PREFIX;
            } else {
                prefix = Tk.resolveString(project,prefix,true);
            }
            if (i<uriFragment.length()) {
                variable = Tk.resolveString(project,uriFragment.substring(i),true);
            }
        } else if (uriFragment.length()>0) {
            prefix = Tk.resolveString(project,uriFragment,true);
        }

        final Map refs = clnt.getProject().getReferences();
        SIDs.Finder refidFinder = new SIDs.Finder() {
                public boolean exists(String sid) {
                    return refs.containsKey(sid);
                }
            };
        String refid;
        synchronized(refs) {
            refid = SIDs.next(refidFinder,prefix);
            if (!Tk.isWhitespace(variable)) {
                refs.put(refid,FixtureExaminer.PENDING_REFERENCE);
                ExportedProperties.set(variable,refid);
            }
        }
        return refid;
    }
}

/* end-of-NewRefIdValueURIHandler.java */