/**
 * $Id: StringItemListHandle.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2003-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.util.Iterator;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.apis.FlexStringFriendly;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.ownhelpers.LocalTk;

/**
 * Helper that can refer to any {@linkplain StringItemList} implementatation instance.
 * Useful when an task or type implementation is only interested in itemlists in a
 * generic sense and needs to define a bean-friendly typed parameter setter. Instead of
 * using a specific itemlist subclass, the task can define their setter in terms of a
 * StringItemListHandle.
 * <p>
 * <b>Examples:</b><pre>
 * --Given Two KindsOf ItemLists--
 *   &lt;<b>strings</b> id="ignoredfiles" prefix="*."&gt;
 *     &lt;string value="last"&gt;
 *     &lt;string value="jar"&gt;
 *     &lt;string value="gz"&gt;
 *     &lt;string value="tgz"&gt;
 *   &lt;/strings&gt;
 *   &lt;<b>urls</b> id="ftpservers"&gt;
 *     &lt;url value="ftp://ftp.apache.org/"/&gt;
 *     &lt;url value="ftp://ftp.eclipse.org/"/&gt;
 *   &lt;/urls&gt;
 *
 * --MkFileTask uses a StringItemListHandle to handle both list types as &lt;lines&gt;--
 *
 *   &lt;<b>newfile</b> path="${work}/lib/.cvsignore"&gt;
 *     &lt;line value="classes"/&gt;
 *     &lt;lines listref="ignoredfiles"/&gt;
 *   &lt;/newfile&gt;
 *
 *   &lt;<b>newfile</b> path="${tempdir}/ftp.in" persist="no"&gt;
 *     &lt;lines listref="ftpservers"/&gt;
 *   &lt;/newfile&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded (once fully configured)
 * @.group   api,helper
 **/

public class StringItemListHandle extends AssertableProjectComponent
    implements Cloneable, ListFriendly
{
    /**
     * Initializes a new StringItemList handle instance.
     * @see #setListRef setListRef()
     **/
    public StringItemListHandle()
    {
        super(AntX.starters+"ItemListHandle:");
    }



    /**
     * Initializes a StringItemList handle with list reference.
     * @param ref the existing reference to an item list (non-null)
     * @see #setListRef setListRef()
     **/
    public StringItemListHandle(Reference ref)
    {
        super(AntX.starters+"ItemListHandle:");
        require_(ref!=null && ref.getRefId()!=null,"ctor- nonzro refId");
        setListRef(LocalTk.referenceFor(ref));
    }



    /**
     * Returns a clone of this itemlist handle. Clones the underlying
     * reference-- not the refered-to itemlist.
     **/
    public final Object clone()
    {
        try {
            StringItemListHandle cloned = (StringItemListHandle)super.clone();
            if (m_listReference!=null) {
                cloned.m_listReference = 
                    LocalTk.referenceFor(m_listReference.getRefId(),getProject());
            }
            return cloned;
        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }



    /**
     * Initializes this handle's refered-to itemlist.
     * @param r the reference (non-null)
     **/
    public final void setListRef(Reference r)
    {
        require_(r!=null,"setRef- nonzro ref");
        m_listReference = r;
    }



    /**
     * Returns this handle's current itemlist reference. Can return
     * <i>null</i> if never set.
     **/
    public final Reference getListRef()
    {
        return m_listReference;
    }



    /**
     * Tells this itemlist handle whether to return raw unprocessed
     * strings of the target list. Defaults "no".
     * @param raw <i>true</i> if unprocessed strings (by default).
     * @since JWare/AntX 0.5
     **/
    public final void setRaw(Boolean raw)
    {
        if (StringItemList.RAW.equals(raw)) {
            m_how = StringItemList.RAW;
        } else {
            m_how = StringItemList.PROCESSED;
        }
    }



    /**
     * Returns the string itemlist to which this handle is referring.
     * @param theProject [optionll] project from which the reference is
     *         read. If <i>null</i> this handle uses its own enclosing
     *         project.
     * @throws BuildException if this handle undefined, the reference
     *         doesn't exist, or the refered-to-thing is not a string
     *         itemlist.
     **/
    public final StringItemList getList(Project theProject)
    {
        String error = null;

        if (getListRef()==null) {
            error = uistrs().get("type.needs.this.attr",
                                 getUsedForName(),"listref");
            log(error,Project.MSG_ERR);
            throw new BuildException(error);
        }

        if (theProject==null) {
            theProject= getProject();
            verify_(theProject!=null,"getRefObj- hav project");
        }

        String refid = getListRef().getRefId();
        Object o = theProject.getReference(refid);

        if (o==null) {
            error = uistrs().get("task.missing.refid", refid);
        }
        else if (!StringItemList.class.isAssignableFrom(o.getClass())) {
            error = uistrs().get("task.bad.refid", refid,
                                 StringItemList.class.getName(),
                                 o.getClass().getName());
        }
        if (error!=null) {
            log(error,Project.MSG_ERR);
            throw new BuildException(error);
        }
        return (StringItemList)o;
    }



    /**
     * Convenience to return a string iterator for a <em>snapshot</em>
     * of the referred-to itemlist.
     * @see StringItemList#readonlyStringIterator(Project)
     *  readonlyStringIterator(Project)
     * @see #setRaw setRaw(boolean)
     **/
    public final Iterator readonlyStringIterator(Project theProject)
    {
        return getList(theProject).readonlyStringIterator(theProject,m_how);
    }



    /**
     * Returns <i>true</i> if this handle is undefined or its targetted
     * list item is empty.
     **/
    public final boolean isEmpty()
    {
        if (getListRef()==null) {
            return true;
        }
        verifyInProject_("isEmpty");
        StringItemList list = getList(getProject());
        return list.isEmpty();
    }




    /**
     * Returns the number of items in this list's targetted list item.
     * Will return zero if this handle has no list.
     **/
    public final int size()
    {
        if (getListRef()==null) {
            return 0;
        }
        verifyInProject_("isEmpty");
        return getList(getProject()).size();
    }



    /**
     * Helper that returns a newline-delimited list of an itemlist's
     * stringified form. Returns the empty string if the list is
     * <i>null</i>.
     * @param il the itemlist to be converted
     * @param P [optional] project from which properties resolved.
     **/
    public static final String byLineStringFrom(StringItemList il, Project P)
    {
        if (il==null) {
            return "";
        }
        int i=0;
        StringBuffer sb = new StringBuffer(100);
        Iterator itr= il.readonlyStringIterator(P);
        while (itr.hasNext()) {
            if (i>0) {
                sb.append(Strings.NL);
            }
            sb.append(itr.next());
            i++;
        }
        return sb.substring(0);
    }



    /**
     * Returns a newline-delimited list of this handle's underlying
     * itemlist. Returns the empty string if this handle has no
     * defined itemlist reference.
     * @param P [optional] project from which reference read.
     * @see #byLineStringFrom(StringItemList,Project)
     *  byLineStringFrom(StringItemList,Project)
     **/
    public final String byLineStringFrom(Project P)
    {
        if (getListRef()==null) {
            return "";
        }
        return byLineStringFrom(getList(P), P);
    }



    /**
     * Returns a flex-string friendly value of this handle's refered-to
     * itemlist. If the underlying itemlist implements the
     * FlexStringFriendly interface, it's method is used directly;
     * otherwise, this method returns a newline-delimined list of
     * the underlying itemlist's strings.
     * @see #byLineStringFrom(Project)
     * @param P [optional] project from which reference read.
     **/
    public String stringFrom(Project P)
    {
        if (getListRef()==null) {
            return "";
        }
        StringItemList il = getList(P);
        if (il instanceof FlexStringFriendly) {
            return ((FlexStringFriendly)il).stringFrom(P);
        }
        return byLineStringFrom(il,P);
    }



    /**
     * Returns the underlying itemlist's string form. Uses this
     * object's enclosing project as source of reference.
     **/
    public String toString()
    {
        if (getListRef()==null) {
            return "";
        }
        verifyInProject_("toString");
        return getList(getProject()).toString();
    }



    /**
     * Returns this handle's used-as field name. Defaults to
     * "items" if never defined. Never returns <i>null</i>.
     **/
    public final String getUsedForName()
    {
        return m_typicalName;
    }



    /**
     * Customizes this handle's used-as field name. Used in error
     * messages.
     **/
    public final void setUsedForName(String name)
    {
        require_(name!=null,"setUsedForNam- nonzro nam");
        m_typicalName = name;
    }


    private Reference m_listReference;
    private String m_typicalName= "items";
    private Boolean m_how = StringItemList.PROCESSED;
}

/* end-of-StringItemListHandle.java */
