/**
 * $Id: ManagerTask.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.lang.reflect.Method;
import  java.lang.reflect.Modifier;
import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableLibDefinition;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.parameters.RecoveryEnabled;

/**
 * Skeleton for an fixture management task that can perform any of a small set of
 * allowed actions. Typically manager tasks usually support (at least) an "install"
 * and an "uninstall" action although these are not necessary.
 * <p/>
 * When initialized, a manager task will inspect its method table for all <em>public</em>
 * void methods that begin with the action verb "<span class="src">do</span>" as in
 * "<span class="src">doInstall</span>" or "<span class="src">doResolve</span>". <em>The
 * case of the leading "do" must be all lowercase.</em> The part of the method name
 * that follows the "do" is called the action's name (it is
 * {@linkplain com.idaremedia.antx.helpers.Tk#lowercaseFrom(String) normalized} so
 * capitalization does not matter). In the preceding example, having a public
 * "<span class="src">doInstall</span>" method means you have a
 * "<span class="src">install</span>" action available. Each action method must return
 * no outputs and take a single <span class="src">Map</span> parameter. This
 * readonly map contains any options specified by the script and/or the application
 * caller. Whether or not this parameter can be <i>null</i> is subclass defined (the
 * incoming parameter is passed as-is from <span class="src">doAction</span>).
 * <p/>
 * The base manager task does not enforce any particular script-facing form (simple
 * error handling aside); subclasses must decide how a script triggers a particular
 * action. Simple manager tasks usually support a simple "<span class="src">action</span>"
 * parameter without any control options. More complex manager tasks can create
 * individual instantiable tasks from a common superclass. However the action name is
 * determined, subclasses must call the inherited
 * {@linkplain #doAction doAction(&#8230;)} utility method to call the matching method.
 * <p/>
 * Manager tasks should be antlib-compatible. This implies that a manager task
 * must take care with its dependencies as well as the actual Ant-based operations it
 * performs because it is likely to be executed early in an execution iteration as part
 * of its initialization.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,helper
 **/

public abstract class ManagerTask extends AssertableLibDefinition
    implements RecoveryEnabled
{
    /**
     * Initializes a new manager task.
     * @param iam CV-label (non-null)
     **/
    protected ManagerTask(String iam)
    {
        super(iam);
    }



    /**
     * Struct that keeps the script facing action name associated
     * with its matching class method.
     * @since     JWare/AntX 0.5
     * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version   0.5
     */
    private static class ActionEntry
    {
        Method method;
        String name;

        ActionEntry(String name, Method method) {
            this.name = name;
            this.method = method;
        }
        boolean matches(String othername) {
            return this.name.equals(othername);
        }
        void call(ManagerTask target, Map params) {
            try {
                method.invoke(target, new Object[]{params});
            } catch(Exception anyX) {
                if (anyX instanceof BuildException) {
                    throw ((BuildException)anyX);
                }
                throw new BuildException(anyX,target.getLocation());
            }
        }
        public String toString() {
            return this.name;
        }
    }



    /**
     * Creates and adds a new method descriptor to this
     * manager's set of actions.
     * @param name normalized action name (non-null)
     * @param method the associated class method (non-null)
     **/
    private void newActionMethod(String name, Method method)
    {
        if (m_actors==null) {
            m_actors = AntXFixture.newList();
        }
        m_actors.add(new ActionEntry(name,method));
    }



    /**
     * Searches for a matching method descript for the given
     * action name. Will return <i>null</i> if no match.
     * @param name action name (non-null)
     * @return descriptor or <i>null</i>
     **/
    private ActionEntry findActionMethod(String name)
    {
        if (m_actors!=null) {
            name = Tk.lowercaseFrom(name);
            for (int i=0,N=m_actors.size();i<N;i++) {
                ActionEntry e = (ActionEntry)m_actors.get(i);
                if (e.matches(name)) {
                    return e;
                }
            }
        }
        return null;
    }



    /**
     * Inspects this manager task and creates a table of all
     * public "do" methods. This method must be called at
     * initialization time or <em>before</em> this task is
     * executed. Can be called multiple times; will create the
     * set only once.
     **/
    private void buildActionMethods()
    {
        if (m_actors==null) {
            verify_(Modifier.isPublic(getClass().getModifiers()), "init- public class");
            Method[] methods = getClass().getMethods();
            String loc = LocalTk.purtyLocation(getLocation());
            final boolean logit = getProject()!=null;
            for (int i=0;i<methods.length;i++) {
                Method m = methods[i];
                String name = m.getName();
                Class returnType = m.getReturnType();
                Class[] args = m.getParameterTypes();

                if (name.startsWith("do") && !name.equals("do") &&
                    java.lang.Void.TYPE.equals(returnType) &&
                    args.length == 1 &&
                    Map.class.equals(args[0])) {

                    String actionname = Tk.lowercaseFrom(name.substring(2));
                    newActionMethod(actionname, m);
                    if (logit) {
                        log("Recording action '"+actionname+"' @ "+loc, Project.MSG_DEBUG);
                    }
                }
            }
        }
    }



    /**
     * Returns the number of valid actions this tasks knows about.
     * Will return zero if this task has not been initialized or there
     * are no public "do" action methods.
     **/
    public int actionMethodCount()
    {
        return m_actors==null ? 0 : m_actors.size();
    }



    /**
     * Builds this task's set of callable actions.
     */
    protected void initonce()
    {
        super.initonce();
        buildActionMethods();
    }



    /**
     * Called to trigger the action method associated with the
     * given name. Subclasses can extend this method to do pre-filtering
     * before callling this inherited method (where the action's
     * name <em>must</em> match a "do" method's name).
     * @param action name of action to trigger (non-null)
     * @param params generic parameters to method
     * @param clnt [optional] external problem handler (non-null)
     * @throws BuildException if unknown action and haltiferror is true.
     **/
    protected void doAction(String action, Map params, Requester clnt)
    {
        require_(action!=null,"do- nonzro action");
        ActionEntry e = findActionMethod(action);
        if (e!=null) {
            e.call(this,params);
        }
        else {
            if (clnt==null) {
                clnt = new Requester.ForComponent(this);
            }
            String error = Iteration.uistrs().get("task.manager.err.unknown.operation",action);
            if (isHaltIfError()) {
                clnt.problem(error, Project.MSG_ERR);
                throw new BuildException(error, clnt.getLocation());
            }
            clnt.problem(error, Project.MSG_WARN);
        }
    }



    /**
     * Ensures this task's set of callable actions have been
     * constructed. This backup call is performed in case the application
     * fails to call the <span class="src">init</span> method for
     * a programmatically created manager task.
     */
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);
        buildActionMethods();
    }



    /**
     * Returns a delimited string of all of this task's valid action
     * names. The caller supplies the name separating token. Must be
     * called <em>after</em> this task's actions have been determined
     * (i&#46;e&#46; after this task has been initialized). This method
     * is useful when generating error messages or diagnostics.
     * @param separator the separator token (pass <i>null</i> to use "|")
     * @return string of action names or empty string if none.
     **/
    public final String validActionNames(String separator)
    {
        if (m_actors!=null) {
            StringBuffer sb = AntXFixture.newStringBuffer();
            if (separator==null) {
                separator = "|";
            }
            for (int i=0,N=m_actors.size();i<N;i++) {
                ActionEntry e = (ActionEntry)m_actors.get(i);
                if (i>0) {
                    sb.append(separator);
                }
                sb.append(e.name);
            }
            return sb.substring(0);
        }
        return "";
    }



    /**
     * Tells this task whether to signal an build error if it does not
     * recognize the requested action or is unable to complete
     * the action successfully.
     * @param halt <i>false</i> if should continue.
     */
    public void setHaltIfError(boolean halt)
    {
        m_willHalt = halt;
    }



    /**
     * Returns <i>true</i> if this task will signal an error if it is
     * unable to complete the action. Defaults to <i>true</i> if not
     * set explicitly.
     * @see #setHaltIfError setHaltIfError()
     */
    public boolean isHaltIfError()
    {
        return m_willHalt;
    }



    private List m_actors;
    private boolean m_willHalt=true;//NB:die by default!
}

/* end-of-ManagerTask.java */