/**
 * $Id: MacroMaker.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.TaskContainer;
import  org.apache.tools.ant.UnknownElement;
import  org.apache.tools.ant.taskdefs.MacroDef;

import  com.idaremedia.antx.AssertableLibDefinition;
import  com.idaremedia.antx.apis.Nameable;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;
import  com.idaremedia.antx.ownhelpers.UEContainerProxy;

/**
 * Base implementation of a macro proxy that limits what gets put in the macro. This
 * class is a <em>&lt;macrodef&gt;</em> builder; it always creates a macro in the
 * enclosing project (the macro may, in turn, add its own type definition to the project).
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,infra
 * @.pattern  GoF.Proxy
 * @.pattern  GoF.Builder
 * @.expects  Ant 1.6
 **/

public abstract class MacroMaker extends AssertableLibDefinition
    implements StrictMacroDef, Nameable, TaskContainer
{
    /**
     * Assumed name of the builtin Ant
     * <span class="src">&lt;macrodef&gt;</span> task.
     **/
    public static final String MACRO_TASKNAME="macrodef";


    /**
     * Initializes a new macro maker instance for a subclass.
     * @param iam CV-label (non-null)
     **/
    protected MacroMaker(String iam)
    {
        super(iam);
    }


    /**
     * Called to initialize this definition before any nested items
     * are added or (script-facing) parameters applied. The default
     * implementation creates the underlying macro definition.
     * If you override this method, you must call this inherited
     * method before using the underlying macro definition.
     * @see #getMacro
     **/
    protected void initonce()
    {
        super.initonce();
        if (getMacroImpl()==null) {
            verifyInProject_("init");
            createMacroImpl();
        }
    }


    /**
     * Execute this definition; same as {@linkplain #executeMacroImpl}
     * by default. Subclasses can decorate the macro's execution by
     * overriding this method.
     **/
    public void execute()
    {
        executeMacroImpl();
    }


    /**
     * Installs this definition's underlying macro definition.
     * @throws BuildException if macro definition does
     **/
    protected final void executeMacroImpl()
    {
        getMacro().execute();
    }


    /**
     * Initializes this definition's enclosing project. If called
     * after {@linkplain #init init()} it will also update our macro
     * definition's project.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        if (getMacroImpl()!=null) {
            getMacroImpl().setProject(P);
        }
    }


    /**
     * Sets a custom class loader for this definition. Will
     * also update the underlying macro definition's loader.
     * @param cL the class loader
     **/
    public void setAntlibClassLoader(ClassLoader cL)
    {
        super.setAntlibClassLoader(cL);
        getMacro().setAntlibClassLoader(cL);
    }


    /**
     * Sets this definition's namespace URI. Will also update the
     * underlying macro definition's URI.
     * @param uri the uri (non-null)
     **/
    public void setURI(String uri)
    {
        super.setURI(uri);
        getMacro().setURI(getURI());
    }


    /**
     * Sets this definition's and the macro's name. You must
     * specify a definition name.
     * @param name the macro's name (non-whitespace)
     * @see #setId setId
     **/
    public final void setName(String name)
    {
        require_(!Tk.isWhitespace(name),"setName- nonzro name");
        getMacro().setName(name);
        m_macroName = name;
    }


    /**
     * Returns this definition's marco's name. Will return
     * <i>null</i> if name never set.
     **/
    public final String getName()
    {
        return m_macroName;
    }


    /**
     * Sets this maker's global reference id and default macro
     * name if one has not been defined explicitly. This method is
     * a convenience that let's script authors use the builtin
     * Ant <span class="src">id</span> attribute to create a
     * reference and define the macro's name in single step.
     **/
    public void setId(String id)
    {
        if (m_macroName==null) {
            m_macroName = id;
        }
    }


    /**
     * Adds a new attribute definition to this definition's macro.
     * Subclasses should call this method from their script-facing
     * APIs.
     * @param attr the attribute definition (non-null)
     **/
    protected final void addMacroParameter(MacroDef.Attribute attr)
    {
        require_(attr!=null,"addAttr- nonzro attr");
        getMacro().addConfiguredAttribute(attr);
    }


    /**
     * Adds a new template element to this definition's macro.
     * Subclasses should call this method from their script-facing
     * APIs or at initialization time to install client-supplied
     * elements.
     * @param name the name of the element client will provide (non-null)
     * @param optional <i>true</i> if this is an optional item
     **/
    protected final void addMacroElement(String name, boolean optional)
    {
        require_(name!=null,"addElem- nonzro name");
        MacroDef.TemplateElement elem = new MacroDef.TemplateElement();
        elem.setName(name);
        elem.setOptional(optional);
        getMacro().addConfiguredElement(elem);
    }


    /**
     * Factory method for this definition's underlying macro
     * definition. This default implementation creates a standard
     * Ant <span class="src">&lt;macrodef&gt;</span>. Expects this
     * task's project to be already initialized. If you override
     * this method and do not use this default implementation, you
     * <em>must</em> call {@linkplain #setMacroImpl setMacroImpl()}
     * before returning.
     * @see #getMacro
     * @see #MACRO_TASKNAME
     **/
    protected MacroDef createMacroImpl()
    {
        Project P= getProject();
        MacroDef impl = (MacroDef)P.createTask(MACRO_TASKNAME);
        impl.setProject(P);
        impl.setLocation(getLocation());
        impl.setOwningTarget(getOwningTarget());
        impl.init();
        setMacroImpl(impl);
        return getMacro();
    }


    /**
     * Initializes this definition's underlying macro definition.
     * This method must be called from the macro
     * {@linkplain #createMacroImpl factory method}.
     * @param impl the definition object (non-null)
     * @.sideeffect A nested sequential is automatically added to
     *        macro definition.
     **/
    protected final void setMacroImpl(MacroDef impl)
    {
        require_(impl!=null,"setImpl- nonzro impl");
        m_macroDefinition = impl;
        m_macroMainImpl = m_macroDefinition.createSequential();
        m_macroMain = m_macroMainImpl;//dflt adds directly to macro
    }


    /**
     * Returns this definition's underlying macro definition.
     * Will return <i>null</i> if macro not created yet.
     * @see #getMacro
     **/
    private MacroDef getMacroImpl()
    {
        return m_macroDefinition;
    }


    /**
     * Installs a single top-level task that encloses all tasks
     * declared in this macro. This is useful to install automatic
     * configuration or protected tasksets for macro's contents. This
     * method can be called <em>once</em> at initialization time
     * <em>after</em> the underlying macro has been created (usually
     * as a customization to {@linkplain #createMacroImpl}). Use the
     * returned handle to further tweak the body element's attributes
     * or to manually add required nested elements.
     * @param tag unqualified (task) name of top-level task (non-null)
     * @param attrs [optional] automatic parameters for new task
     * @param inheritNS <i>true</i> if this macro's namespace should
     *            be passed on to top-level element
     * @see TaskExaminer#newUEProxy TaskExaminer.newUEProxy(&#8230;)
     * @see #addImplTask addImplTask
     **/
    protected final UEContainerProxy installBodyWrap(String tag,
                                                     Map attrs,
                                                     boolean inheritNS)
    {
        verify_(m_macroMainImpl.getNested().isEmpty(),
                "instalBodyWrap- none installed");

        Task nsSource=null;//NB:=>Ant namespace
        if (inheritNS) {
            nsSource = this;
        }
        UnknownElement rootItem =
            TaskExaminer.newUEProxy(tag,attrs,nsSource,getProject());

        m_macroMainImpl.addTask(rootItem);

        UEContainerProxy bodyProxy = new UEContainerProxy(rootItem);
        m_macroMain = bodyProxy;

        return bodyProxy;
    }


    /**
     * Returns this definition's underlying macro definition ensuring
     * that said definition is not <i>null</i>. This accessor is exposed
     * so that subclasses can tweak the non-body related macro parameters
     * like attributes and elements directly. To modify the macro's
     * main definition <em>use the {@linkplain #getMacroBody} method</em>.
     * @throws IllegalStateException if macro definition not initialized
     * @see #createMacroImpl
     **/
    protected final MacroDef getMacro()
    {
        ensure_(m_macroDefinition!=null,"macro- inited");
        return m_macroDefinition;
    }


    /**
     * Returns this definition's underlying macro's task container
     * or body as we call it. The returned body is never <i>null</i>.
     * Subclasses can add required task <em>proxies</em> directly
     * to this container at initialization time.
     * @throws IllegalStateException if macro definition not initialized
     * @see #addImplTask addImplTask
     **/
    protected final TaskContainer getMacroBody()
    {
        ensure_(m_macroMain!=null,"macroBody- inited");
        return m_macroMain;
    }


    /**
     * Called to include the given task proxy in this definition's
     * macro. The proxy is first screened through {@linkplain #includeTask
     * includeTask()} before it is added to the underlying macro's
     * <span class="src">&lt;sequential&gt;</span> list (or a top-level
     * body wrapper). Note that this method is called <em>after</em> any
     * explicit "<span class="src">add</span>" and/or
     * "<span class="src">create</span>" methods in your subclass.
     * @param task task proxy to be added (non-null UnknownElement)
     * @throws BuildException if task is unacceptable
     **/
    public final void addTask(Task task)
    {
        require_(task instanceof UnknownElement,"addTask- proxy task");

        Class taskClass = TaskExaminer.trueClass(task);

        if (taskClass!=null && includeTask(taskClass, task)) {
            getMacroBody().addTask(task);
        } else {
            String ermsg = uistrs().get("macro.nested.task.disallowed",
                                        task.getTaskName(), this.getTaskName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    /**
     * Returns <i>true</i> if a task of given class can be
     * added to this definition's body. This method is the primary
     * control mechanism that subclasses use to control what gets
     * included in final macro. Return <i>false</i> if task
     * should not be included (will trigger a build exception). This
     * default method always returns <i>true</i>.
     * @param taskClass candidate task's class (non-null)
     * @param taskProxy task proxy item (non-null UnknownElement)
     **/
    protected boolean includeTask(Class taskClass, Task taskProxy)
    {
        return true;
    }



    /**
     * Call to include an implementation defined (and required) nested
     * macro task(set). This method is usually used in tandem with the
     * body wrap functionality to install predefined special elements
     * of a strict enclosing taskset. You can use the returned handle
     * to add nested elements to a special task(set).
     * @param tag unqualified (task) name of special task (non-null)
     * @param attrs [optional] automatic parameters for new task
     * @param inheritNS <i>true</i> if this macro's namespace should
     *            be passed on to new element
     **/
    protected final UEContainerProxy addImplTask(String tag, Map attrs,
                                                 boolean inheritNS)
    {
        Task nsSource=null;//NB:=>Ant namespace
        if (inheritNS) {
            nsSource = this;
        }
        UnknownElement implItem =
            TaskExaminer.newUEProxy(tag,attrs,nsSource,getProject());

        getMacroBody().addTask(implItem);

        return new UEContainerProxy(implItem);
    }


    private MacroDef m_macroDefinition;
    private MacroDef.NestedSequential m_macroMainImpl;
    private TaskContainer m_macroMain;
    private String m_macroName;
}

/* end-of-MacroMaker.java */
