/**
 * $Id: ConditionalTaskSet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.ownhelpers.OptionalExecuteHelper;
import  com.idaremedia.antx.parameters.Conditional;

/**
 * Taskset that has builtin support for the standard if/unless conditional execution. 
 * This taskset is <em>not</em> the standard AntX &lt;do&gt; task which supports a
 * large collection of condition types. Instead this class is a lite variant of 
 * the always-executed taskset that AntX-dependent tools use as a starting point for 
 * their own task containers.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 **/

public class ConditionalTaskSet extends TaskSet implements Conditional
{
    /**
     * Initializes a new empty conditional taskset.
     **/
    public ConditionalTaskSet()
    {
        super();
    }



    /**
     * Initializes a new CV-labeled empty conditional taskset.
     * @param iam CV-label (non-null)
     **/
    protected ConditionalTaskSet(String iam)
    {
        super(iam);
    }



    /**
     * Updates this task's and any helper's enclosing project.
     **/
    public void setProject(Project project)
    {
        super.setProject(project);
        m_guard.setProject(project);
    }



    /**
     * Initializes new configuration-delaying taskset.
     * @param iam CV-label (non-null)
     * @param delayConfigure <i>true</i> if nested tasks configured
     *        when they're actually executed
     **/
    protected ConditionalTaskSet(String iam, boolean delayConfigure)
    {
        super(iam,delayConfigure);
    }



    /**
     * Runs nested tasks if all 'if' conditiona met and not if
     * any 'unless' conditions met.
     **/
    protected void performNestedTasks() throws BuildException
    {
        boolean passIf= testIfCondition();

        if (passIf && testUnlessCondition()) {
            performTheCheckedTasks();
        }
        else {
            logSkippedBecause(passIf, Project.MSG_VERBOSE);
        }
    }



    /**
     * Called to perform this taskset's nested tasks <em>after</em>
     * the various if/unless conditions have been verified. Subclasses
     * should override this method if they want to keep the standard
     * condition testing identical.
     * @since JWare/AntX 0.4
     **/
    protected void performTheCheckedTasks()
    {
        performTheTasksList();
    }



    /**
     * Adds a simple if-condition to this taskset.
     **/
    public void setIf(String property)
    {
        m_guard.setIf(property);
    }



    /**
     * Returns this taskset's (raw) if-condition if any. Returns
     * empty string if condition never set.
     **/
    public String getIfProperty()
    {
        String s = m_guard.getIfProperty();
        return s==null ? "" : s;
    }



    /**
     * Adds the unless-condition to this taskset.
     **/
    public void setUnless(String property)
    {
        m_guard.setUnless(property);
    }



    /**
     * Returns this taskset's (raw) unless-condition if any.
     * Returns empty string if condition never set.
     **/
    public String getUnlessProperty()
    {
        String s = m_guard.getUnlessProperty();
        return s==null ? "" : s;
    }



    /**
     * Tests whether or not all of this taskset's "if"
     * conditions are met.
     * @return <i>true</i> if all "if" condition are met
     * @see OptionalExecuteHelper
     **/
    public boolean testIfCondition()
    {
        return m_guard.testIfCondition();
    }



    /**
     * Tests whether or not all of this taskset's "unless"
     * conditions are met.
     * @return <i>true</i> if all "unless" conditions are met
     * @see OptionalExecuteHelper
     **/
    public boolean testUnlessCondition()
    {
        return m_guard.testUnlessCondition();
    }



    /**
     * Helper that shows why this task didn't run (if/unless).
     * Message is logged at requested level.
     **/
    protected final void logSkippedBecause(boolean passIf, int logLevel)
    {
        m_guard.logSkippedBecause(this,passIf,logLevel);
    }



    /**
     * Helper that shows why this task didn't run (if/unless).
     * Message is logged at VERBOSE level.
     **/
    protected final void logSkippedBecause(boolean passIf)
    {
        m_guard.logSkippedBecause(this,passIf,Project.MSG_VERBOSE);
    }



    /** Actual if/unless determiner. **/
    protected final OptionalExecuteHelper m_guard =
        new OptionalExecuteHelper();
}

/* end-of-ConditionalTaskSet.java */
