/**
 * $Id: PropertiesList.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.io.File;
import  java.io.IOException;
import  java.io.InputStream;
import  java.net.URL;
import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;
import  java.util.Properties;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.GenericParameters;
import  com.idaremedia.antx.helpers.InnerNameValuePair;
import  com.idaremedia.antx.helpers.InputFileLoader;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.ownhelpers.ReferenceHandle;
import  com.idaremedia.antx.parameters.FlexSourceSupport;
import  com.idaremedia.antx.starters.ListFriendly;

/**
 * List of string name-value pairs that can be used as source for loop iteration,
 * macro arguments, overlays, value URIs, etc. When printed as a string a properties
 * list elements are separated by the platform's newline string unless its
 * <span class="src">delim</span> parameter has been set. A PropertiesList is the
 * standard AntX bean for {@linkplain GenericParameters} or
 * <span class="src">Properties</span>.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;<b>properties</b> id="javadoc.options"&gt;
 *       &lt;property name="visibility" value="package"/&gt;
 *       &lt;property name="maintitle" value="My API Documentation"/&gt;
 *       &lt;property name="copyleft" value="(c) 2004 Me, Myself, and I."/&gt;
 *       ...
 *    &lt;/properties&gt;
 *    ...
 *    &lt;macrodef name="apidocs"&gt;
 *       &lt;attribute name="args"/&gt;
 *       &lt;sequential&gt;
 *          &lt;javadoc
 *               access="${$prop:@{args}->visibility}"
 *               windowtitle="${$prop:@{args}->maintitle}"
 *               .../&gt;
 *    ...
 *    &lt;apidocs args="javadoc.options"/&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 **/

public final class PropertiesList extends GenericParameters
    implements ListFriendly, FlexSourceSupport, FixtureComponent, AntLibFriendly
{
    private static final String IAM_= AntX.fixture+"PropertiesList:";


    /** Default delimiter used by all property lists. **/
    public static final String DEFAULT_DELIMITER = Strings.NL;


    /**
     * Initializes a new empty properties collection bean.
     **/
    public PropertiesList()
    {
    }



    /**
     * Initializes a new empty properties collection bean
     * associated with project.
     * @param project this object's enclosing project
     **/
    public PropertiesList(Project project)
    {
        setProject(project);
    }



    /**
     * Capture our identifier for feedback since types don't 
     * always get correct location information.
     * @param id our identifier (via build script)
     **/
    public void setId(String id)
    {
        m_Id= id;
    }



    /**
     * Tries to return an unique identifier for this list.
     **/
    public String getId()
    {
        return m_Id;
    }



    /**
     * Sets a custom delimiter for this properties list. This
     * delimiter will be used by all stringification(TM)
     * methods.
     * @param delimiter new delimiter (non-null)
     **/
    public void setDelim(String delimiter)
    {
        AntX.require_(delimiter!=null,IAM_,"setDelim- nonzro delim");
        m_delim = delimiter;
    }



    /**
     * Returns the delimiter this list will use for all
     * stringification(TM). Never returns <i>null</i>.
     **/
    public String getDelim()
    {
        return m_delim;
    }



    /**
     * From another properties-like object, add all unique contents 
     * (no overwriting of locally defined items). The referenced
     * thing must be an object that we can translate into a standard
     * <span class="src">Properties</span> object.
     * @param other reference to other properties item (non-null)
     * @throws BuildException if reference is malformed or
     *         referred-to thing is not convertible to a Properties.
     **/
    public void addConfiguredProperties(ReferenceHandle other)
    {
        AntX.require_(other!=null,IAM_,"addProps- nonzro handle");
        Reference r = other.getRefId();
        if (r==null || r.getRefId()==null || r.getRefId().equals(m_Id)) {
            String error = Iteration.uistrs().get
                ("task.bad.nested.item","properties","Invalid 'refid'");
            log(error,Project.MSG_ERR);
            throw new BuildException(error);
        }
        boolean continu = true;
        if (other.getHaltIfMissingFlag()==Boolean.FALSE) {
            Object referredTo = getProject().getReference(r.getRefId());
            if (referredTo==null) {
                continu = false;
            }
        }
        if (continu) {
            Properties otherP = FixtureExaminer.
                getReferencedProperties(getProject(),r.getRefId(),null);
            Iterator itr = otherP.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry kv = (Map.Entry)itr.next();
                String key = (String)kv.getKey();
                if (get(key)==null) {
                    put(key, (String)kv.getValue());
                }
            }
        }
    }



    /**
     * Returns an iterator for all stringified key-value pairs
     * in this list. Each item is a single string of form: 
     * <i>key</i>=<i>value</i> where <i>value</i> can be the
     * empty string.
     * @param theProject [optional] the context project (can be 
     *        <i>null</i>)
     * @see #getDelim
     */
    public Iterator readonlyStringIterator(Project theProject)
    {
        Map items = copyOfParameterObjects();
        List kvpairs = AntXFixture.newList(items.size());
        Iterator itr = items.values().iterator();
        while (itr.hasNext()) {
            InnerNameValuePair e = (InnerNameValuePair)itr.next();
            String v = e.getValue(theProject,true);
            if (v==null) {
                kvpairs.add(""+e.getName());
            } else {
                kvpairs.add(""+e.getName()+"="+v);
            }
        }
        return kvpairs.iterator();
    }



    /**
     * Returns the string-form appropriate for this object.
     * @param theProject [optional] the context project (can be 
     *        <i>null</i>)
     **/
    public String stringFrom(Project theProject)
    {
        StringBuffer sb = AntXFixture.newLargeStringBuffer();
        final String theDelimiter = getDelim();

        int N=0;
        Iterator itr= readonlyStringIterator(theProject);

        while (itr.hasNext()) {
            if (N>0) {
                sb.append(theDelimiter);
            }
            sb.append(itr.next().toString());
            N++;
        }
        itr=null;
        return sb.substring(0);
    }



    /**
     * Adds the contents of the given resource file to this list.
     * The resource file must contain a standard Properties-formatted set
     * of key-value pairs. This method acts as a no-op if the no such
     * resource exists.
     * @param rn (subpath) name of resoure (non-null)
     * @throws BuildException if unable to read resoure once opened
     **/
    public void setResource(String rn)
    {
        AntX.require_(rn!=null,IAM_,"setRez- nonzro name");

        InputStream is = LocalTk.getSystemResourceAsStream(rn, getProject());
        if (is!=null) {
            try {
                Properties values = new Properties();
                values.load(is);
                mergeOther(values);
            } catch(IOException iox) {
                String error = Iteration.uistrs().get("task.bad.configfile",
                    "resource("+rn+")", iox.getMessage());
                log(error,Project.MSG_ERR);
                throw new BuildException(error,iox);
            }
        } else {
            String warning = Iteration.uistrs().get("task.nostream.from.rez", rn);
            log(warning,Project.MSG_WARN);
        }
    }



    /**
     * Adds the contents of the named Properties file to this list.
     * The contents of the file are <em>merged</em> with
     * this lists current contents, unconditionally replacing any
     * pre-existing items of same names. This method does nothing if
     * the named file does not exist.
     * @param filepath absolute path or project-relative path (non-null)
     * @throws BuildException if unable to read existing file for any reason.
     **/
    public void setFile(String filepath)
    {
        AntX.require_(filepath!=null,IAM_,"setFile- nonzro name");

        File f = getProject().resolveFile(filepath);
        if (f.canRead()) {
            try {
                URL url = AntXFixture.fileUtils().getFileURL(f);
                Properties values = InputFileLoader.loadProperties(url,null);
                mergeOther(values);
            } catch(IOException ioX) {
                String error = Iteration.uistrs().get("task.bad.configfile",
                    filepath, ioX.getMessage());
                log(error,Project.MSG_ERR);
                throw new BuildException(error,ioX);
            }
        } else {
            String warning = Iteration.uistrs().get("task.err.filenotfound", filepath);
            log(warning, Project.MSG_WARN);
        }
    }



    /**
     * Adds the contents of the named Properties URL to this list.
     * The contents of the resource are <em>merged</em> with
     * this lists current contents, unconditionally replacing any
     * pre-existing items of same names.
     * @param urlstring location of Properties file (non-null)
     * @throws BuildException if unable to load resource for any reason.
     **/
    public void setURL(String urlstring)
    {
        AntX.require_(urlstring!=null,IAM_,"setURL- nonzro URL");
        
        try {
            URL url = new URL(urlstring);
            Properties values = InputFileLoader.loadProperties(url,null);
            mergeOther(values);
        } catch(IOException ioX) {
            String error = Iteration.uistrs().get("task.bad.configfile",
                "URL("+urlstring+")", ioX.getMessage());
            log(error,Project.MSG_ERR);
            throw new BuildException(error,ioX);
        }
    }


    private String m_Id = "n/d";
    private String m_delim = DEFAULT_DELIMITER;
}


/* end-of-PropertiesList.java */